MODULE ice_thickness_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE

  REAL(dp), PARAMETER :: epsilon = 0.000001_dp    ! Tolerance

CONTAINS
    FUNCTION numflux(HiA, HiB, UsA, UsB, D_2DA, D_2DB, HsA, HsB, maskA, maskB, dx_or_dy) RESULT(FluxA_to_B)
      ! The flux from 1 to 2.
      ! Numerical flux function. This is the discretization of the flux between two 
      ! grid points. One of the grid points has thickness HiA and velocity UsA while
      ! the other has thickness HiB and velocity UsB.
      ! For a mass conserving discretization it is required that:
      !
      ! numflux(HiA,HiB,UsA,UsB,D_2DA,D_2DB,HsA,HsB,maskA,maskB) == 
      !                        - numflux(HiB,HiA,-UsB,-UsA,D_2DB,D_2DA,HsB,HsA,maskB,maskA)

      ! Input variables:
      REAL(dp), INTENT(IN) :: HiA, HiB, UsA, UsB, D_2DA, D_2DB, HsA, HsB, dx_or_dy
      INTEGER,  INTENT(IN) :: maskA, maskB
      
      ! Result variables:
      REAL(dp)             :: FluxA_to_B
      
      ! local variables
      REAL(dp)             :: Flux_div, Flux_ssa, HiF
     
      ! SIA and SSA fluxes are calculated seperately. In case of the original version, 
      ! with SIA on land, SSA on shelf and Weertman sliding velocities, this works also, since SSA are zero on land (right?)
      ! and sliding velocities are included in the Diffusivity D_2D
      
      ! First determine the staggered diffusivity flux from the SIA (land) velocities betweem grid A and B:
      IF((maskA /= C%type_shelf) .AND. (maskB /= C%type_shelf)) THEN
        Flux_div = (D_2DA+D_2DB) * (HsB-HsA) / (2._dp * dx_or_dy) 
      ELSE
        Flux_div = 0._dp
      END IF
      
      ! SSA flux for shelf (and basal velocities in case of using PISM version: choice_sliding_method = 2)
      ! first determine the staggered value of the SSA velocity:
      Flux_ssa = (UsA + UsB) / 2._dp
      ! choose, depending on the sign of the staggered velocity, which ice grid point to use of the ice thickness:
      IF (Flux_ssa >= 0._dp) THEN
        HiF = HiA
      ELSE 
        HiF = HiB
      END IF 
      IF (HiF == C%Hi_min) Flux_ssa = 0._dp
      Flux_ssa = Flux_ssa * HiF

      ! Sum the two fluxes in between grid point A and B
      FluxA_to_B = Flux_div + Flux_ssa

    END FUNCTION numflux


    SUBROUTINE ice_thickness(Hi, D_2D, MB, Hs, U_SSA, V_SSA, mask, Hi_new, MB_applied, domain_leaving_flux, mass_conservation, fluxes_per_dt)

      ! Input variables:
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hi
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: D_2D
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: MB
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hs
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: U_SSA               ! SSA velocity for sheet and shelf in the x direction
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: V_SSA               ! SSA velocity for sheet and shelf in the y direction
      INTEGER,  DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: mask     

      ! Output variable:
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: Hi_new              ! The new calculated ice thickness [m]
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: MB_applied          ! The real applied mass balance per time step [mieq per T_ANT%dt]
      REAL(dp),                                   INTENT(OUT) :: domain_leaving_flux
      REAL(dp),                                   INTENT(OUT) :: mass_conservation   ! In case all ice volume is conserved at (2:C%NX_ant-1,2:C%NY_ant-1) this variable is zero

      ! Added BdB 10/2010 - for usage in Isotopes routine
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, 6), INTENT(OUT) :: fluxes_per_dt       ! All fluxes (from 2:NX-1,2:NY-1) In meter per T_ANT%dt   >= 0

      ! Local variable:
      INTEGER                                                :: i, j
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )             :: dHi
      
      ! Output: dHi
      CALL ice_thickness_changes_explicit(Hi, D_2D, MB, Hs, U_SSA, V_SSA, mask, dHi, MB_applied, domain_leaving_flux, fluxes_per_dt)

      ! Update Hi_new:
      Hi_new = Hi + dHi

      ! The domain boundary points get the value of the nearest inward points. This is saver in case of 
      ! grounded points at the domain boundary, and actually makes almost no difference with applying 
      ! dHi(at domain boundary) = dHi(nearest inward point). The assumption is a smooth domain boundary.
      Hi_new(   1,        :) = Hi_new(         2,          :)
      Hi_new(C%NX_ant,    :) = Hi_new(C%NX_ant-1,          :) 
      Hi_new(   :,        1) = Hi_new(         :,          2) 
      Hi_new(   :, C%NY_ant) = Hi_new(         :, C%NY_ant-1)
      
      DO j = 1, C%NY_ant
      DO i = 1, C%NX_ant
        IF(Hi_new(i,j) < C%Hi_min - epsilon) WRITE(UNIT=C%stderr, FMT='(A, I3, A, I3, A, F12.3, A, F14.2)') &
         'ALARM ANT: Hi_new(i=', i, ',j=', j, ') = ', Hi_new(i,j), ' < C%Hi_min. Only with an ADI scheme this is allowed! at time ', T_ANT%time
      END DO
      END DO
      
      ! To make sure Hi will not just be below C%Hi_min at the 16-th digit (this happens):
      Hi_new = MAX(Hi_new, C%Hi_min)
      
      ! For the domain minus the domain boundary (2:C%NX_ant-1,2:C%NY_ant-1) the ice changed are added, if correct this will be zero:
      mass_conservation = SUM(Hi_new(2:C%NX_ant-1,2:C%NY_ant-1) - Hi(2:C%NX_ant-1,2:C%NY_ant-1) - MB_applied(2:C%NX_ant-1,2:C%NY_ant-1)) + domain_leaving_flux
      
      IF(.FALSE.) THEN
       WRITE(UNIT=C%stdlog, FMT='(6(A, E22.15), A,F14.2)') &
       'Mass conservation ANT: Hi_new = ',                    SUM(Hi_new    (2:C%NX_ant-1,2:C%NY_ant-1)), &
       ', Hi = ',                                             SUM(Hi        (2:C%NX_ant-1,2:C%NY_ant-1)), &
       ', MB_applied = ',                                     SUM(MB_applied(2:C%NX_ant-1,2:C%NY_ant-1)), &
       ', domain_leaving_flux = ',                            domain_leaving_flux, &
       ', mass conservation = ',                              mass_conservation, &
       ' which is ',                                          100._dp * mass_conservation / SUM(Hi_new(2:C%NX_ant-1,2:C%NY_ant-1) &
        - Hi(2:C%NX_ant-1,2:C%NY_ant-1)), ' %  Time = ', T_ANT%time
      END IF

    END SUBROUTINE ice_thickness


    SUBROUTINE ice_thickness_changes_explicit(Hi, D_2D, MB, Hs, Us, Vs, mask, dHi, mass_balance_per_dt, domain_leaving_flux, fluxes_per_dt)
      ! Update the thickness of the layer of ice for all types of points to time t.
      ! At every point (i,j) the outgoing flux to each of the four neighbour points
      ! is calculated. Also both the incoming and outgoing massbalance-flux for (i,j)
      ! is calculated seperatetly. Then there follows a check: if the total outgoing 
      ! ice flux at (i,j) exceeds the total present amount of ice, the four fluxes to
      ! the neighbours points are downscaled with the same factor (rescale_factor) causing a
      ! compatible outgoing flux: Not more went out than was inside.

      ! Input variables:
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hi               ! In meter
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: D_2D             ! In meter per year   <= 0
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: MB               ! In meter per year
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Hs               ! In meter
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Us               ! In meter per year
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: Vs               ! In meter per year
      INTEGER,  DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(IN)  :: mask

      ! Output variables:
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: dHi                   ! Note that the domain boundary points of dHi do not get a value.
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   ), INTENT(OUT) :: mass_balance_per_dt   ! In meter per T_ANT%dt
      REAL(dp),                                   INTENT(OUT) :: domain_leaving_flux   ! The domain leaving flux [meter per T_ANT%dt (and per C%dx_ant * C%dy_ant)]
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, 6), INTENT(OUT) :: fluxes_per_dt         ! All fluxes (from 2:NX-1,2:NY-1) In meter per T_ANT%dt   >= 0

      ! Local variables:
      INTEGER                                                 :: i, j
      REAL(dp)                                                :: Hi_available          ! In meter for this T_ANT%dt
      REAL(dp)                                                :: rescale_factor        ! dimensionless  0 <= rescale_factor <= 1
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: flux_XP_per_dt        ! In meter per T_ANT%dt   >= 0
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: flux_XM_per_dt        ! In meter per T_ANT%dt   >= 0
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: flux_YP_per_dt        ! In meter per T_ANT%dt   >= 0
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: flux_YM_per_dt        ! In meter per T_ANT%dt   >= 0
      REAL(dp), DIMENSION(C%NX_ant, C%NY_ant   )              :: advective_flux_per_dt ! In meter per T_ANT%dt   >= 0

      INTEGER                                                 :: intervention_counter = 0
      INTEGER                                                 :: check
      LOGICAL                                                 :: check_again
      REAL(dp)                                                :: exceeded_decrease
      REAL(dp)                                                :: domain_entering_flux

      ! initialize the fluxes:
      advective_flux_per_dt = 0._dp 
      flux_XP_per_dt        = 0._dp
      flux_XM_per_dt        = 0._dp
      flux_YP_per_dt        = 0._dp
      flux_YM_per_dt        = 0._dp  
      fluxes_per_dt         = 0._dp
      mass_balance_per_dt   = 0._dp



      ! Calculate dHi at interior points
      dHi = 0._dp



      DO j = 2, C%NY_ant - 1
      DO i = 2, C%NX_ant - 1
        ! Fluxes from grid point (i,j) to the neighbouring grid points (be aware that D_2D is negative,
        ! see calculate_D_uv_3D()). The fluxXP, fluxXM, FluxYP, fluxYM correspond respectively to the
        ! right, left, up and down flux at the staggered grid (on half gridsizes) around point(i,j).
        ! Flux from (i,j) to (i+1,j):
        
        flux_XP_per_dt(i,j) =   MAX(0._dp, numflux(Hi(i,j),   Hi(i+1,j), &
                                               Us(i,j),   Us(i+1,j), &            
                                             D_2D(i,j), D_2D(i+1,j), &
                                               Hs(i,j),   Hs(i+1,j), &
                                             mask(i,j), mask(i+1,j), C%dx_ant) * T_ANT%dt / C%dx_ant)
        ! Flux from (i,j) to (i-1,j):                                    
        flux_XM_per_dt(i,j) = - MIN(0._dp, numflux(Hi(i-1,j),   Hi(i,j), &
                                               Us(i-1,j),   Us(i,j), &            
                                             D_2D(i-1,j), D_2D(i,j), &
                                               Hs(i-1,j),   Hs(i,j), &
                                             mask(i-1,j), mask(i,j), C%dx_ant) * T_ANT%dt / C%dx_ant)
        ! Flux from (i,j) to (i,j+1):                                    
        flux_YP_per_dt(i,j) =   MAX(0._dp, numflux(Hi(i,j),   Hi(i,j+1), &
                                               Vs(i,j),   Vs(i,j+1), &            
                                             D_2D(i,j), D_2D(i,j+1), &
                                               Hs(i,j),   Hs(i,j+1), &
                                             mask(i,j), mask(i,j+1), C%dy_ant) * T_ANT%dt / C%dy_ant)

        ! Flux from (i,j) to (i,j-1):                                  
        flux_YM_per_dt(i,j) = - MIN(0._dp, numflux(Hi(i,j-1),   Hi(i,j), &
                                               Vs(i,j-1),   Vs(i,j), &            
                                             D_2D(i,j-1), D_2D(i,j), &
                                               Hs(i,j-1),   Hs(i,j), &
                                             mask(i,j-1), mask(i,j), C%dy_ant) * T_ANT%dt / C%dy_ant)
                                                                                              

        ! Advective flux leaving point (i,j) per T_ANT%dt:
        advective_flux_per_dt(i,j) = flux_XP_per_dt(i,j) + flux_XM_per_dt(i,j) + flux_YP_per_dt(i,j) + flux_YM_per_dt(i,j)
        
        ! Prevent a negative ice thickness due to:
        !  - A too large negative mass balance: Then the applied MB is taken equal to 100% melt of the 
        !                                       available ice volume over that T_ANT%dt
        !  - Too large advective fluxes:        Then the left part of the available ice volume plus the (negative/positive)
        !                                       mass balance, over that T_ANT%dt, is equally devided over the four advective
        !                                       fluxes with the  rescale_factor  
        ! In this scheme no more ice can come from a point than there was present.
        
        ! The maximum ice column (for this i,j point) available for melt and the advective flux per T_ANT%dt:
        Hi_available = Hi(i,j) - C%Hi_min
        ! The mass balance available during this time interval T_ANT%dt:
        mass_balance_per_dt(i,j) = MB(i,j) * T_ANT%dt
        ! Initializing the rescale factor at the non rescaled value:
        rescale_factor = 1._dp

        ! Intervention for a very large negative mass balance (Actually this should not occur with a proper MB):
        IF(mass_balance_per_dt(i,j) < - (C%maximum_dHi_change_per_year * T_ANT%dt)) THEN
         mass_balance_per_dt(i,j) = - C%maximum_dHi_change_per_year * T_ANT%dt
         WRITE(UNIT=C%stdlog, FMT='(A, I3, A, I3, A, F8.2, A, F14.2)') ' ANT mass_balance_per_dt(i=', i, ',j=', j, ') is limited to ', &
          - C%maximum_dHi_change_per_year * T_ANT%dt, ', at time ', T_ANT%time
        END IF

        ! First we check if a negative mass balance would melt all the available ice in this single time step T_ANT%dt: 
        IF(-mass_balance_per_dt(i,j) >= Hi_available) THEN
         ! In such a case, there can be melted not more than the available ice amount at this grid point: 
         mass_balance_per_dt(i,j) = -Hi_available
         ! And there is nothing left for the outward fluxes from this point to its neighbours:
         rescale_factor = 0._dp

        ! Next, we check for two situations: 
        !  1. The advective flux combined with a negative mass balance are not allowed to exceed the available amount of ice.
        !  2. The advective flux is not allowed to exceed the available amount of ice plus the positive mass balance for this T_ANT%dt.
        ! In both cases we first take the contribution of the mass balance into acount and then rescale the fluxes in such a way
        ! that in total not more than the available ice is taken away at this time step T_ANT%dt. This guarantees mass conservation.
        ELSE IF(advective_flux_per_dt(i,j) - mass_balance_per_dt(i,j) > Hi_available) THEN
         rescale_factor = (Hi_available + mass_balance_per_dt(i,j)) / advective_flux_per_dt(i,j)
        END IF

        ! In the remaining cases the mass_balance and the advective fluxes where already well calculated
        ! before the if-statement, and thus remains: rescale_factor = 1._dp as initiated
        
        ! Rescale the fluxes, and keep those fluxes stored:
        flux_XP_per_dt(i,j)        = rescale_factor * flux_XP_per_dt(i,j)
        flux_XM_per_dt(i,j)        = rescale_factor * flux_XM_per_dt(i,j)
        flux_YP_per_dt(i,j)        = rescale_factor * flux_YP_per_dt(i,j)
        flux_YM_per_dt(i,j)        = rescale_factor * flux_YM_per_dt(i,j)
        advective_flux_per_dt(i,j) = rescale_factor * advective_flux_per_dt(i,j) 

        ! Due to its four neighbours there can be maximal four additions to the dHi of one grid point at different places 
        ! in the double loop, and one subtraction from the out_flux from the point itself:
        dHi(i+1,j  ) = dHi(i+1,j  ) + flux_XP_per_dt(i,j)
        dHi(i-1,j  ) = dHi(i-1,j  ) + flux_XM_per_dt(i,j)
        dHi(i  ,j+1) = dHi(i  ,j+1) + flux_YP_per_dt(i,j)
        dHi(i  ,j-1) = dHi(i  ,j-1) + flux_YM_per_dt(i,j)
        dHi(i  ,j  ) = dHi(i  ,j  ) - advective_flux_per_dt(i,j) + mass_balance_per_dt(i,j)

      END DO
      END DO


      intervention_counter = 0

      DO check = 1, 20
        check_again = .FALSE.

        DO j = 2, C%NY_ant - 1
        DO i = 2, C%NX_ant - 1
          exceeded_decrease = dHi(i,j) + C%maximum_dHi_change_per_year * T_ANT%dt
          IF(exceeded_decrease < 0._dp) THEN
           ! Back correction of the fluxes: keeping mass conservation in case of an intervention:
           IF(.TRUE.) THEN
!            WRITE(UNIT=C%stdlog,FMT='(A,2I4,A,I3,2(A,F12.4))') 'ANT Exceeded maximum dHi change of 100 m at ',i,j,' with mask = ',mask(i,j), &
!            ' advective flux is: ',advective_flux_per_dt(i,j), ' exceeded_decrease = ',exceeded_decrease

            ! Calculate (again) a rescale factor, to adjust the fluxes to make them compatible with the intervention:
            rescale_factor = exceeded_decrease / advective_flux_per_dt(i,j)
            flux_XP_per_dt(i,j)        = rescale_factor * flux_XP_per_dt(i,j)
            flux_XM_per_dt(i,j)        = rescale_factor * flux_XM_per_dt(i,j)
            flux_YP_per_dt(i,j)        = rescale_factor * flux_YP_per_dt(i,j)
            flux_YM_per_dt(i,j)        = rescale_factor * flux_YM_per_dt(i,j)
            ! Only necessary for an eventual monitoring:
            advective_flux_per_dt(i,j) = advective_flux_per_dt(i,j) + exceeded_decrease

            dHi(i+1,j  ) = dHi(i+1,j  ) + flux_XP_per_dt(i,j)
            dHi(i-1,j  ) = dHi(i-1,j  ) + flux_XM_per_dt(i,j)
            dHi(i  ,j+1) = dHi(i  ,j+1) + flux_YP_per_dt(i,j)
            dHi(i  ,j-1) = dHi(i  ,j-1) + flux_YM_per_dt(i,j)
           END IF
           dHi(i,j)    = dHi(i,j)   - exceeded_decrease

           intervention_counter = intervention_counter + 1 
           check_again = .TRUE.

!           WRITE(UNIT=C%stdlog, FMT='(A, I3, A, I3, A, F8.2, A, I2, A, I5, A, F14.2)') ' ANT Minimum dHi(i=', i, ',j=', j, ') is limited to ', &
!            - C%maximum_dHi_change_per_year * T_ANT%dt, ' in loop ', check, ', at intervention', intervention_counter, ', at time ', T_ANT%time
          END IF

        END DO
        END DO

        ! Check for leaving the intervention_counter-loop:
        IF(check_again .eqv. .FALSE.) EXIT
      END DO

      
      ! Added BdB 10/2010 for usage in isotopes routine: put fluxes in one field for output
      fluxes_per_dt(:,:,1) = flux_XP_per_dt
      fluxes_per_dt(:,:,2) = flux_XM_per_dt
      fluxes_per_dt(:,:,3) = flux_YP_per_dt
      fluxes_per_dt(:,:,4) = flux_YM_per_dt
      fluxes_per_dt(:,:,5) = advective_flux_per_dt
      fluxes_per_dt(:,:,6) = mass_balance_per_dt
      
      CALL message_concerning_dHi_changes(intervention_counter, check - 1, dHi)
      
      ! The advective flux (per C%dx_ant * C%dy_ant) leaving the domain through the domain edges:
      domain_leaving_flux  = SUM(Max(0._dp, flux_XM_per_dt(2,2:C%NY_ant-1))) + SUM(Max(0._dp, flux_XP_per_dt(C%NX_ant-1,2:C%NY_ant-1))) + &
                             SUM(Max(0._dp, flux_YM_per_dt(2:C%NX_ant-1,2))) + SUM(Max(0._dp, flux_YP_per_dt(2:C%NX_ant-1,C%NY_ant-1))) 
      domain_entering_flux = SUM(Min(0._dp, flux_XM_per_dt(2,2:C%NY_ant-1))) + SUM(Min(0._dp, flux_XP_per_dt(C%NX_ant-1,2:C%NY_ant-1))) + &
                             SUM(Min(0._dp, flux_YM_per_dt(2:C%NX_ant-1,2))) + SUM(Min(0._dp, flux_YP_per_dt(2:C%NX_ant-1,C%NY_ant-1))) 
      IF(domain_entering_flux < 0._dp) WRITE(UNIT=C%stdlog, FMT='(A, F12.2, A, F12.2, A, F14.2)')    &
       'ALARM ANT: a non zero entering flux is detected: domain_leaving_flux = ', domain_leaving_flux,  &
       ', domain_entering_flux = ', domain_entering_flux, ', at time: ', T_ANT%time
    END SUBROUTINE ice_thickness_changes_explicit
    


    SUBROUTINE message_concerning_dHi_changes(intervention_counter, loop_amount, dHi)
      ! Write a message about the minimum and maximum dHi changes, and about the interventions
      ! in case they occured.

      ! Input variable:
      INTEGER                                            :: intervention_counter
      INTEGER                                            :: loop_amount
      REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN) :: dHi

      ! Local variables:
      INTEGER,  DIMENSION(2)                     :: location_of_dHi_maximum
      INTEGER,  DIMENSION(2)                     :: location_of_dHi_minimum
      
      location_of_dHi_maximum = MAXLOC(dHi)                                                                             
      location_of_dHi_minimum = MINLOC(dHi)
      
      IF(intervention_counter > 0) THEN                                                                                             
       WRITE(UNIT=C%stdlog, FMT='(A, I3, A, I3, A, F8.2,   A, I5, A, I2, A,   A, I3, A, I3, A, F8.2, A, F14.2)') &
        ' ANT Minimum dHi(i=', location_of_dHi_minimum(1), ',j=', location_of_dHi_minimum(2), ') is limited to ', - C%maximum_dHi_change_per_year * T_ANT%dt, &
        ' with', intervention_counter, ' intervention(s), in ', loop_amount, ' loop(s).', &
        '  Maximum dHi(i=', location_of_dHi_maximum(1), ',j=', location_of_dHi_maximum(2), ') = ', dHi(location_of_dHi_maximum(1),location_of_dHi_maximum(2)), &
        ', at time ', T_ANT%time
      ELSE                                                                                                                                                
!!       WRITE(UNIT=C%stdlog, FMT='(2(A, I3, A, I3, A, F8.2), A, F14.2)') &
!!        ' ANT Minimum dHi(i=', location_of_dHi_minimum(1), ',j=', location_of_dHi_minimum(2), ') = ', dHi(location_of_dHi_minimum(1),location_of_dHi_minimum(2)), &
!!        '. Maximum dHi(i=', location_of_dHi_maximum(1), ',j=', location_of_dHi_maximum(2), ') = ', dHi(location_of_dHi_maximum(1),location_of_dHi_maximum(2)), &
!!        ', at time ', T_ANT%time
      END IF                                                                                                                                              
    END SUBROUTINE message_concerning_dHi_changes
    


    SUBROUTINE intervention_for_too_large_dHi_changes(dHi)
      ! Do not allow too large changes of Hi in one time step (in an attempt to get a stable behavior, e.g. at the start of a run).                 
      ! This intervention/ALARM should disappear after a few hundred of time steps.  

      ! In/Output variable:
      REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(INOUT) :: dHi

      ! Local variables:
      INTEGER,  DIMENSION(2)                        :: location_of_dHi_maximum
      INTEGER,  DIMENSION(2)                        :: location_of_dHi_minimum
      
      location_of_dHi_maximum = MAXLOC(dHi)                                                                             
      location_of_dHi_minimum = MINLOC(dHi) 
      IF(MAXVAL(ABS(dHi)) > (C%maximum_dHi_change_per_year * T_ANT%dt)) THEN                                                                                             
       WRITE(UNIT=C%stdlog, FMT='(A, I6, A, 2(A, I3, A, I3, A, F8.2), A, F14.2)') &
        ' ALARM ANT: ', COUNT(ABS(dHi) > (C%maximum_dHi_change_per_year * T_ANT%dt)), ' point(s) with |dHi| > C%maximum_dHi_change_per_year * T_ANT%dt.', &
        '  Minimum dHi(i=', location_of_dHi_minimum(1), ',j=', location_of_dHi_minimum(2), ') = ', dHi(location_of_dHi_minimum(1),location_of_dHi_minimum(2)), &
        ', Maximum dHi(i=', location_of_dHi_maximum(1), ',j=', location_of_dHi_maximum(2), ') = ', dHi(location_of_dHi_maximum(1),location_of_dHi_maximum(2)), &
        ', at time ', T_ANT%time
       WHERE(dHi < - (C%maximum_dHi_change_per_year * T_ANT%dt)) dHi = - C%maximum_dHi_change_per_year * T_ANT%dt
       WHERE(dHi >   (C%maximum_dHi_change_per_year * T_ANT%dt)) dHi =   C%maximum_dHi_change_per_year * T_ANT%dt
      !ELSE                                                                                                                                                
      ! WRITE(UNIT=C%stdlog, FMT='(2(A, I3, A, I3, A, F8.2), A, F14.2)') &
      !  ' ANT Minimum dHi(i=', location_of_dHi_minimum(1), ',j=', location_of_dHi_minimum(2), ') = ', dHi(location_of_dHi_minimum(1),location_of_dHi_minimum(2)), &
      !  ', Maximum dHi(i=', location_of_dHi_maximum(1), ',j=', location_of_dHi_maximum(2), ') = ', dHi(location_of_dHi_maximum(1),location_of_dHi_maximum(2)), &
      !  ', at time ', T_ANT%time
      END IF                                                                                                                                              
  END SUBROUTINE intervention_for_too_large_dHi_changes
END MODULE ice_thickness_ant_module
