MODULE ice_surface_temperature_ant_module
  USE configuration_main_module, ONLY: dp, C, Model_Time
  USE configuration_ant_module, ONLY: C_ANT, T_ANT
  IMPLICIT NONE

CONTAINS
  SUBROUTINE ice_surface_temperature(mask, Hs, sealevel, Hi, gT_offset, Ts, T2m_mon)
    ! Calculation of the ice-surface temperature Ts (being Ti(z=1,:,:))
    ! The main ingredients are:
    ! - Parametrizations of ice-surface temperature (by Fortuin/Huybrechts regression of the 
    !   measurements at the ice sheet surface, or by the Lipzig's LAM model)
    ! - Correction of current temperature to earlier times with the temperature deviations
    USE parameters_main_module, ONLY: T0
    
    ! Input variables:
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: mask         ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hs           ! Height of surface                              
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: sealevel     ! sea level relative to PD
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hi           ! Ice thickness
    REAL(dp),                                  INTENT(IN)  :: gT_offset    ! Interpolated forcing global Temperature offset 
    
    ! Output variable: Ts
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Ts           ! Temperature of ice                             
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(OUT) :: T2m_mon      ! 2-meter air temperature [K]   

    SELECT CASE(C_ANT%choice_Ts_model)
    CASE(1)
     ! Surface temperature parameterized by a regression by Fortuin/Huybrechts:
     ! Output: Ts                                                                            
     CALL Fortuin_parameterization_Ts(Hs - sealevel, Ts)
    CASE(2)
     ! Surface temperature of Lipzig's LAM model:
     ! Output: Ts                                                                            
     CALL Lipzig_parameterization_Ts(Hs - sealevel, Ts)
    CASE(9)
     ! Surface temperature of Lipzig's LAM model for monthly temperatures:
     ! Output: Ts, T2m_mon                                                                            
     CALL Lipzig_parameterization_monthly_Ts(mask, Hs, sealevel, Ts, T2m_mon)
     T2m_mon = T2m_mon + gT_offset
    CASE(10) 
     ! Surface temperature adjusted with a constant lapse rate:
     ! Output: Ts, T2m_mon                                                                            
     CALL Constant_lapserate_monthly_Ts(mask, hs, sealevel, Ts, T2m_mon)
     T2m_mon = T2m_mon + gT_offset
    CASE DEFAULT
     STOP ' Stop message from: ice_surface_temperature:  A non valid choice for C_ANT%choice_Ts_model'
    END SELECT

    ! Correction of the ice surface temperature in earlier times with the temperature difference
    ! Changed BdB (10/2009), put on second 'switch', only maximum of Tsurface is freezing for ice-covered grid points.
    Ts = Ts + gT_offset

!    WHERE(Ts > T0) Ts = T0  ! Maximum ice surface temperature is 0 degrees Celcius
    WHERE((Ts > T0) .AND. (Hi > C%Hi_min)) Ts = T0  ! Maximum ice surface temperature is 0 degrees Celcius if ice covered.


  END SUBROUTINE ice_surface_temperature

  SUBROUTINE Fortuin_parameterization_Ts(Elevation, Ts)
    ! Surface temperature parameterization of Fortuin and Oerlemans (1990).
    ! The surface temperature Ts is parameterized by taking a latitude dependent
    ! temperature for the sealevel level, minus a correction for the altitude:
    !   Ts = Ts(at sealevel) - lapserate*elevation
    !      = 307.61 - 0.68775 * ABS(latitude) - lapserate * elevation
    ! Where lapserate*elevation above and below 1500 m altitude differs.
    ! Due to the ABS(), this parameterisation for the lattitude is valid for the 
    ! Southern and Northern Hemisphere. (The NP is taken +90 degree, the SP -90 degree.)
    USE parameters_main_module, ONLY:  Atmos_lapserate_low, Atmos_lapserate_high
    USE reference_fields_ant_module, ONLY: Latitude

    ! Input variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Elevation ! Surface height [m].
    
    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: Ts        ! Surface temperature [K].

    WHERE(Elevation < 1500.0_dp)
     Ts = 307.61_dp - 0.68775_dp * ABS(Latitude) - Atmos_lapserate_low * Elevation
    ELSEWHERE
     Ts = 307.61_dp - 0.68775_dp * ABS(latitude) - Atmos_lapserate_low  *  1500.0_dp &
                                                       - Atmos_lapserate_high * (Elevation - 1500.0_dp)   
    END WHERE
  END SUBROUTINE Fortuin_parameterization_Ts
  
  SUBROUTINE Lipzig_parameterization_Ts(Elevation, Ts)
    ! Surface temperature parameterization of Lipzig's LAM model.
    USE parameters_main_module, ONLY:  Atmos_lapserate_low, Atmos_lapserate_high
    USE reference_fields_ant_module, ONLY: Hs_ref, Ts_ref           ! Hs_ref = Elevation_pd, Ts_ref = Ts_pd
   
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Elevation    ! Surface height [m]
    
    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: Ts           ! Surface temperature [K]
    
    ! Local variables:
    INTEGER                                             :: i, j
    REAL(dp)                                            :: Alr, h15, hmin, hmax ! used to calcute the 
                                                                                ! Atmospheric lapse rate (Alr)

    ! Set the ice-surface temperature Ts = Ts(Elevation(i,j), latitude_at_ij(i,j))
    ! h15 is the altitude above which the lapse rate changes in the Fortuin parameterization
    h15 = 1500._dp             
    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
       hmin = MIN(Elevation(i,j), Hs_ref(i,j))
       hmax = MAX(Elevation(i,j), Hs_ref(i,j))
       IF (hmax <= h15) THEN
        Alr = Atmos_lapserate_low
       ELSE IF (hmin >= h15) THEN
        Alr = Atmos_lapserate_high
       ELSE ! hmin < h15 < hmax
        Alr = ((h15-hmin)* Atmos_lapserate_low + (hmax-h15) * Atmos_lapserate_high) / (hmax - hmin)
       END IF
       Ts(i,j) = Ts_ref(i,j) - Alr * (Elevation(i,j) - Hs_ref(i,j))
    END DO
    END DO
  END SUBROUTINE Lipzig_parameterization_Ts


  SUBROUTINE Lipzig_parameterization_monthly_Ts(mask, Hs, sealevel, Ts, T_monthly)
    ! This is the Lipzig_parameterization_Ts subroutine but adjusted for monthly fields 
    ! used in the Bintanja Surface MB routine
    ! Added by BdB (04/2010)

    USE parameters_main_module, ONLY:  Atmos_lapserate_low, Atmos_lapserate_high
    USE reference_fields_ant_module, ONLY: Hs_ref, Tmon_ref
   
    ! Input variables:
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: mask            ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hs              ! Height of surface                              
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: sealevel        ! sea level relative to PD
    
    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Ts           ! Surface temperature [K]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(OUT) :: T_monthly    ! Monthly 2-meter air temperature [K]

    ! Local variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )              :: change_lapse ! change in elevation correction for temperature
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )              :: Elevation ! Surface height [m]
    INTEGER                                                :: i,j,m
    REAL(dp)                                               :: Alr, h15, hmin, hmax ! used to calcute the 
                                                                                   ! Atmospheric lapse rate (Alr)
    ! h15 is the altitude above which the lapse rate changes in the Fortuin parameterization
    h15 = 1500._dp             
   
    DO j = 1, C%NY_ant    
    DO i = 1, C%NX_ant
      IF (mask(i,j) /= C%type_shelf) THEN
        Elevation(i,j) = Hs(i,j)
      ELSE
        Elevation(i,j) = Hs(i,j) - sealevel(i,j)
      END IF

      hmin = MIN(Elevation(i,j), Hs_ref(i,j))
      hmax = MAX(Elevation(i,j), Hs_ref(i,j))
      IF (hmax <= h15) THEN
        Alr = Atmos_lapserate_low
      ELSE IF (hmin >= h15) THEN
        Alr = Atmos_lapserate_high
      ELSE ! hmin < h15 < hmax
        Alr = ((h15-hmin)* Atmos_lapserate_low + (hmax-h15) * Atmos_lapserate_high) / (hmax - hmin)
      END IF
    
      change_lapse(i,j) = Alr * (Elevation(i,j) - Hs_ref(i,j))
      
      ! Monthly temperature
      DO m = 1,12
        T_monthly(i,j,m) = Tmon_ref(i,j,m) - change_lapse(i,j)
      END DO

      ! Mean over the year
      Ts(i,j)          = SUM(T_monthly(i,j,1:12)) / 12._dp

    END DO
    END DO
  
  END SUBROUTINE Lipzig_parameterization_monthly_Ts


  SUBROUTINE Constant_lapserate_monthly_Ts(mask, Hs, sealevel, Ts, T_monthly)
    ! Temperatures are adjusted with a constant lapse rate, used for the NH ice sheets.
    ! used in the Bintanja Surface MB routine
    ! Added by BdB (02/2011)

    USE parameters_main_module, ONLY:  Atmos_lapserate_NH
    USE reference_fields_ant_module, ONLY: Hs_ref, Tmon_ref, Tmon_clim
   
    ! Input variables:
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: mask            ! mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: Hs              ! Height of surface                              
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(IN)  :: sealevel        ! sea level relative to PD
    
    ! Output variable:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   ), INTENT(OUT) :: Ts              ! Surface temperature [K]
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,12), INTENT(OUT) :: T_monthly       ! Monthly 2-meter air temperature [K]

    ! Local variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant   )              :: change_lapse    ! change in elevation correction for temperature
    INTEGER                                                :: i,j,m, step

    ! Calculate change in temperature due to the change in elevation,
    ! for grid points where the surface is ocean, include changes in sea level
    WHERE (mask /= C%type_shelf) 
      change_lapse = Atmos_lapserate_NH * (Hs - Hs_ref)
    ELSE WHERE
      change_lapse = Atmos_lapserate_NH * (Hs - sealevel - Hs_ref)
    END WHERE
    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant    
      ! Monthly temperature

      ! When using climate fields as forcing
!      IF(C%choice_forcing == C%climate_model_forcing) THEN
!       step = 12 * INT(C%climate_fields_length/12 * (T_ANT%time - C%start_time_of_run + 0.01_dp) / (C%end_time_of_run - C%start_time_of_run))
!       DO m = 1,12
!          T_monthly(i,j,m) = Tmon_clim(i,j,m+step) - change_lapse(i,j)
!        END DO
!      ELSE  
        DO m = 1,12
          T_monthly(i,j,m) = Tmon_ref(i,j,m) - change_lapse(i,j)
        END DO
!      END IF

      ! Mean over the year
      Ts(i,j)            = SUM(T_monthly(i,j,1:12)) / 12._dp
    END DO
    END DO
    
  END SUBROUTINE Constant_lapserate_monthly_Ts
END MODULE ice_surface_temperature_ant_module
