MODULE gradient_ant_module
    USE configuration_main_module, ONLY: dp, C
    USE configuration_ant_module, ONLY: C_ANT
    IMPLICIT NONE

CONTAINS
  SUBROUTINE derivatives_of_Hi_and_Hs(single_GL_points, mask, Hi, dHi_dx, dHi_dy, Hs, dHs_dx, dHs_dy)
    ! This subroutine calculates all necessary derivatives of Hi and Hs
    ! With the option to handle single groundline points seperate.
    ! Taking in the single groundline case the largest 1st order gradient in that x or y direction,
    ! Hi and Hs are independent handled at such a single groundline point, so it is possible
    ! that the dHi and dHs differ from sign. 

    ! Input variables:
    LOGICAL,                        INTENT(IN)  :: single_GL_points ! Special treatment of single ground line points
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: mask
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hi
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(IN)  :: Hs
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: dHi_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: dHi_dy
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: dHs_dx
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant), INTENT(OUT) :: dHs_dy
    
    ! Local variables:
    INTEGER                                     :: i, j

    ! Special treatment of single ground line points:
    IF(single_GL_points .eqv. .TRUE.) THEN
     ! Loop over whole domain but not the domain boundary:

     DO j = 2, C%NY_ant - 1
     DO i = 2, C%NX_ant - 1
       IF(mask(i,j) == C%type_groundline) THEN
        ! For single groundline points between four shelf points:
        IF(mask(i-1,j) == C%type_shelf .AND. mask(i+1,j) == C%type_shelf .AND. &
           mask(i,j-1) == C%type_shelf .AND. mask(i,j+1) == C%type_shelf) THEN
         
         ! X-direction, concerning Hi:
         IF(ABS(Hi(i,j) - Hi(i-1,j)) >= ABS(Hi(i+1,j) - Hi(i,j))) THEN
          ! If the left gradient is the largest, the left gradient is choosen (1st order).
          dHi_dx(i,j) = (Hi(i,j) - Hi(i-1,j)) / C%dx_ant
         ELSE
          ! If the right gradient is the largest, the right gradient is choosen (1st order).
          dHi_dx(i,j) = (Hi(i+1,j) - Hi(i,j)) / C%dx_ant
         END IF 
         ! Y-direction, concerning Hi:
         IF(ABS(Hi(i,j) - Hi(i,j-1)) >= ABS(Hi(i,j+1) - Hi(i,j))) THEN
          ! If the down gradient is the largest, the down gradient is choosen (1st order).
          dHi_dy(i,j) = (Hi(i,j) - Hi(i,j-1)) / C%dy_ant
         ELSE
          ! If the up gradient is the largest, the up gradient is choosen (1st order).
          dHi_dy(i,j) = (Hi(i,j+1) - Hi(i,j)) / C%dy_ant
         END IF 
         
         ! X-direction, concerning Hs (so the sign of dHs is independent of dHi):
         IF(ABS(Hs(i,j) - Hs(i-1,j)) >= ABS(Hs(i+1,j) - Hs(i,j))) THEN
          ! If the left gradient is the largest, the left gradient is choosen (1st order).
          dHs_dx(i,j) = (Hs(i,j) - Hs(i-1,j)) / C%dx_ant
         ELSE
          ! If the right gradient is the largest, the right gradient is choosen (1st order).
          dHs_dx(i,j) = (Hs(i+1,j) - Hs(i,j)) / C%dx_ant
         END IF 
         ! Y-direction, concerning Hs:
         IF(ABS(Hs(i,j) - Hs(i,j-1)) >= ABS(Hs(i,j+1) - Hs(i,j))) THEN
          ! If the down gradient is the largest, the down gradient is choosen (1st order).
          dHs_dy(i,j) = (Hs(i,j) - Hs(i,j-1)) / C%dy_ant
         ELSE
          ! If the up gradient is the largest, the up gradient is choosen (1st order).
          dHs_dy(i,j) = (Hs(i,j+1) - Hs(i,j)) / C%dy_ant
         END IF 

        ELSE
         ! For other groundline points:
         dHi_dx(i,j)  = C_ANT%a_x             * Hi(i-1,j  ) + C_ANT%c_x             * Hi(i+1,j  )
         dHi_dy(i,j)  = C_ANT%a_y             * Hi(i  ,j-1) + C_ANT%c_y             * Hi(i  ,j+1)
         
         dHs_dx(i,j)  = C_ANT%a_x             * Hs(i-1,j  ) + C_ANT%c_x             * Hs(i+1,j  )
         dHs_dy(i,j)  = C_ANT%a_y             * Hs(i  ,j-1) + C_ANT%c_y             * Hs(i  ,j+1)
        END IF 
       ELSE
        ! Sheet and shelf points:
        dHi_dx(i,j)  = C_ANT%a_x             * Hi(i-1,j  ) + C_ANT%c_x             * Hi(i+1,j  )
        dHi_dy(i,j)  = C_ANT%a_y             * Hi(i  ,j-1) + C_ANT%c_y             * Hi(i  ,j+1)

        dHs_dx(i,j)  = C_ANT%a_x             * Hs(i-1,j  ) + C_ANT%c_x             * Hs(i+1,j  )
        dHs_dy(i,j)  = C_ANT%a_y             * Hs(i  ,j-1) + C_ANT%c_y             * Hs(i  ,j+1)
       END IF 
     END DO
     END DO

    ! No special treatment of single ground line points (just a very common central difference scheme):
    ELSE

     DO j = 2, C%NY_ant - 1
     DO i = 2, C%NX_ant - 1
        ! Sheet and shelf points:
        dHi_dx(i,j)  = C_ANT%a_x             * Hi(i-1,j  ) + C_ANT%c_x             * Hi(i+1,j  )
        dHi_dy(i,j)  = C_ANT%a_y             * Hi(i  ,j-1) + C_ANT%c_y             * Hi(i  ,j+1)
        
        dHs_dx(i,j)  = C_ANT%a_x             * Hs(i-1,j  ) + C_ANT%c_x             * Hs(i+1,j  )
        dHs_dy(i,j)  = C_ANT%a_y             * Hs(i  ,j-1) + C_ANT%c_y             * Hs(i  ,j+1)
     END DO
     END DO


    END IF
    
    ! Giving the domain boundary the value of the direct interior neighbour:
    dHi_dx(2:C%NX_ant-1,       1)  = dHi_dx(2:C%NX_ant-1,         2)
    dHi_dx(2:C%NX_ant-1,C%NY_ant)  = dHi_dx(2:C%NX_ant-1,C%NY_ant-1)
    dHi_dx(           1,       :)  = dHi_dx(           2,         :)
    dHi_dx(  C%NX_ant,         :)  = dHi_dx(  C%NX_ant-1,         :)
    dHi_dy(2:C%NX_ant-1,       1)  = dHi_dy(2:C%NX_ant-1,         2)
    dHi_dy(2:C%NX_ant-1,C%NY_ant)  = dHi_dy(2:C%NX_ant-1,C%NY_ant-1)
    dHi_dy(           1,       :)  = dHi_dy(           2,         :)
    dHi_dy(    C%NX_ant,       :)  = dHi_dy(  C%NX_ant-1,         :)

    dHs_dx(2:C%NX_ant-1,        1)  = dHs_dx(2:C%NX_ant-1,         2)
    dHs_dx(2:C%NX_ant-1, C%NY_ant)  = dHs_dx(2:C%NX_ant-1,C%NY_ant-1)
    dHs_dx(           1,        :)  = dHs_dx(       2,             :)
    dHs_dx(    C%NX_ant,        :)  = dHs_dx(  C%NX_ant-1,         :)
    dHs_dy(2:C%NX_ant-1,        1)  = dHs_dy(2:C%NX_ant-1,         2)
    dHs_dy(2:C%NX_ant-1, C%NY_ant)  = dHs_dy(2:C%NX_ant-1,C%NY_ant-1)
    dHs_dy(           1,        :)  = dHs_dy(           2,         :)
    dHs_dy(    C%NX_ant,        :)  = dHs_dy(  C%NX_ant-1,         :)

  END SUBROUTINE derivatives_of_Hi_and_Hs


  SUBROUTINE xy_gradients(NX, NY, dx, dy, single_GL_points, mask, Va, Vb, dVa_dx, dVb_dy)
    ! This routine computes the x- and y- gradients of a variable 'V'
    ! dVa_dx and dVb_dy are the computed derivatives of Va and Vb

    ! Input variables:
    INTEGER ,                   INTENT(IN)  :: NX
    INTEGER ,                   INTENT(IN)  :: NY
    REAL(dp),                   INTENT(IN)  :: dx
    REAL(dp),                   INTENT(IN)  :: dy
    LOGICAL,                    INTENT(IN)  :: single_GL_points ! Special treatment of single ground line points
    INTEGER,  DIMENSION(NX,NY), INTENT(IN)  :: mask
    REAL(dp), DIMENSION(NX,NY), INTENT(IN)  :: Va
    REAL(dp), DIMENSION(NX,NY), INTENT(IN)  :: Vb
    
    ! Output variables:
    REAL(dp), DIMENSION(NX,NY), INTENT(OUT) :: dVa_dx
    REAL(dp), DIMENSION(NX,NY), INTENT(OUT) :: dVb_dy
    
    ! Local variables:
    INTEGER                                 :: i, j

    ! Special treatment of single ground line points:
    IF(single_GL_points .eqv. .TRUE.) THEN 
     ! Loop over whole domain but not the domain boundary:
     DO j = 2, NY - 1
     DO i = 2, NX - 1
       !IF(.FALSE.) THEN ! this is using the original way
       IF(mask(i,j) == C%type_groundline) THEN
        ! For single groundline points between four shelf points:
        IF(mask(i-1,j) == C%type_shelf .AND. mask(i+1,j) == C%type_shelf .AND. &
           mask(i,j-1) == C%type_shelf .AND. mask(i,j+1) == C%type_shelf) THEN
         ! X-direction:
         IF(ABS(Va(i,j) - Va(i-1,j)) >= ABS(Va(i+1,j) - Va(i,j))) THEN
          ! If the left gradient is the largest, the left gradient is choosen (1st order).
          dVa_dx(i,j) = (Va(i,j) - Va(i-1,j)) / dx
         ELSE
          ! If the right gradient is the largest, the right gradient is choosen (1st order).
          dVa_dx(i,j) = (Va(i+1,j) - Va(i,j)) / dx
         END IF 
         ! Y-direction:
         IF(ABS(Vb(i,j) - Vb(i,j-1)) >= ABS(Vb(i,j+1) - Vb(i,j))) THEN
          ! If the down gradient is the largest, the down gradient is choosen (1st order).
          dVb_dy(i,j) = (Vb(i,j) - Vb(i,j-1)) / dy
         ELSE
          ! If the up gradient is the largest, the up gradient is choosen (1st order).
          dVb_dy(i,j) = (Vb(i,j+1) - Vb(i,j)) / dy
         END IF 
        ELSE
         ! For other groundline points:
         dVa_dx(i,j) = C_ANT%a_x * Va(i-1,j) + C_ANT%c_x * Va(i+1,j)
         dVb_dy(i,j) = C_ANT%a_y * Vb(i,j-1) + C_ANT%c_y * Vb(i,j+1)
        END IF 
       ELSE
        ! Sheet and shelf points:
        dVa_dx(i,j) = C_ANT%a_x * Va(i-1,j) + C_ANT%c_x * Va(i+1,j)
        dVb_dy(i,j) = C_ANT%a_y * Vb(i,j-1) + C_ANT%c_y * Vb(i,j+1)
       END IF 
     END DO
     END DO
    ! No special treatment of single ground line points (just a very common central difference scheme):
    ELSE
     DO j = 2, NY - 1
     DO i = 2, NX - 1
        ! Sheet and shelf points:
        dVa_dx(i,j) = C_ANT%a_x * Va(i-1,j) + C_ANT%c_x * Va(i+1,j)
        dVb_dy(i,j) = C_ANT%a_y * Vb(i,j-1) + C_ANT%c_y * Vb(i,j+1)
     END DO
     END DO
    END IF
    
    ! Giving the domain boundary the value of the direct interior neighbour:
    dVa_dx( 1, :) = dVa_dx(   2,   :)
    dVa_dx(NX, :) = dVa_dx(NX-1,   :)
    dVa_dx( :, 1) = dVa_dx(   :,   2)
    dVa_dx( :,NY) = dVa_dx(   :,NY-1)
    dVb_dy( 1, :) = dVb_dy(   2,   :)
    dVb_dy(NX, :) = dVb_dy(NX-1,   :)
    dVb_dy( :, 1) = dVb_dy(   :,   2)
    dVb_dy( :,NY) = dVb_dy(   :,NY-1)
  END SUBROUTINE xy_gradients



  SUBROUTINE xy_gradients_grounded_area(NX, NY, dx, dy, mask, Va, Vb, dVa_dx, dVb_dy)
    ! This routine computes the x- and y- gradients of a variable 'V':
    ! So dVa_dx and dVb_dy are the computed derivatives  of Va and Vb.
    ! On the sheet the left and right neighbour values of V are used
    ! to determine the gradient at point (i,j): central difference scheme.
    ! At the ground line for most points a 2nd order but one sided (inward)
    ! scheme is used, only for the exeptional points,  
    ! If, let say, the right neighbour of point (i,j) is a shelf point,
    ! the two nearest points at the left site of (i,j) are used, to 
    ! calculate the gradient in (i,j) by a Taylor aproximation from 
    ! these two left neighbours points. For other shelf-neighbours, 
    ! left, above or under, the aproach is analogous.
    ! But in case of only two groundline points next to each other, 
    ! the gradient is determined from only V(i,j) itself and this other
    ! single groundline neighbour point.
    ! If both left and right neighbour points are shelf points the derivative is set to zero. 
    !Dit laatste zou beter worden als het zoals bij de 1e routine in deze module wordt

    ! Input variables:
    INTEGER,                    INTENT(IN)  :: NX, NY
    REAL(dp),                   INTENT(IN)  :: dx, dy
    REAL(dp), DIMENSION(NX,NY), INTENT(IN)  :: Va, Vb
    INTEGER,  DIMENSION(NX,NY), INTENT(IN)  :: mask
    
    ! Output variables:
    REAL(dp), DIMENSION(NX,NY), INTENT(OUT) :: dVa_dx, dVb_dy
    
    ! Local variables:
    INTEGER                                 :: i, j

    ! Loop over whole domain but not the domain boundary:
    DO j = 2, NY - 1
    DO i = 2, NX - 1
       IF(mask(i,j)==C%type_sheet) THEN
        dVa_dx(i,j) = (Va(i+1,j  )-Va(i-1,j  )) / (2._dp * dx)
        dVb_dy(i,j) = (Vb(i  ,j+1)-Vb(i  ,j-1)) / (2._dp * dy)
       ELSE IF(mask(i,j)==C%type_groundline) THEN
        ! Calculation of the x-derivative:     
        IF((mask(i-1,j)/=C%type_shelf) .AND. (mask(i+1,j)/=C%type_shelf)) THEN
         ! In case that none of the neighbours is shelf:
         dVa_dx(i,j) = (Va(i+1,j)-Va(i-1,j)) / (2._dp * dx)
        ! Otherwise at least one of the neighbours is shelf:
        ELSE IF(mask(i-1,j)==C%type_shelf) THEN
         ! In case the left neighbour is shelf:
         IF(mask(i+1,j)==C%type_shelf) THEN
          ! In this case both neighbours are shelf:
          dVa_dx(i,j) = 0.
         ! Otherwise the left neighbour is shelf, but the right neighbour is not shelf:
         ELSE IF(mask(i+2,j)/=C%type_shelf) THEN
          dVa_dx(i,j) = (3._dp*Va(i,j)-4._dp*Va(i+1,j)+Va(i+2,j)) / (-2._dp * dx)
         ELSE
           ! Otherwise only two groundline points next to each other:
           dVa_dx(i,j) = (Va(i,j)-Va(i+1,j)) / (-dx)
         END IF
        ELSE
         ! Otherwise the left neighbour is sheet or groundline, whereas the right must be shelf:
         IF(mask(i-2,j)/=C%type_shelf) THEN
          dVa_dx(i,j) = (3._dp*Va(i,j)-4._dp*Va(i-1,j)+Va(i-2,j)) / (2._dp * dx)
         ELSE
           ! Otherwise only two groundline points next to each other:
           dVa_dx(i,j) = (Va(i,j)-Va(i-1,j)) / dx
         END IF
        END IF
        ! Calculation of the y-derivative:
        IF((mask(i,j-1)/=C%type_shelf) .AND. (mask(i,j+1)/=C%type_shelf)) THEN
         !  In case that none of the neighbours is shelf:
         dVb_dy(i,j) = (Vb(i,j+1)-Vb(i,j-1)) / (2._dp * dy)
        ! Otherwise at least one of the neighbours is shelf:
        ELSE IF(mask(i,j-1)==C%type_shelf) THEN
         ! In case the lower neighbour is shelf:
         IF(mask(i,j+1)==C%type_shelf) THEN
          ! In this case both neighbours are shelf:
          dVb_dy(i,j) = 0.
         ! Otherwise the lower neighbour is shelf, but the upper neighbour is not shelf:
         ELSE IF(mask(i,j+2)/=C%type_shelf) THEN
          dVb_dy(i,j) = (3._dp*Vb(i,j)-4._dp*Vb(i,j+1)+Vb(i,j+2)) / (-2._dp * dy)
         ELSE
           ! Otherwise only two groundline points next to each other:
           dVb_dy(i,j) = (Vb(i,j)-Vb(i,j+1)) / (-dy)
         END IF
        ELSE
         ! Otherwise the lower neighbour is sheet or groundline, whereas the upper must be shelf:
         IF(mask(i,j-2)/=C%type_shelf) THEN
          dVb_dy(i,j) = (3._dp*Vb(i,j)-4._dp*Vb(i,j-1)+Vb(i,j-2)) / (2._dp * dy)
         ELSE
           ! Otherwise only two groundline points next to each other:
           dVb_dy(i,j) = (Vb(i,j)-Vb(i,j-1)) / dy
         END IF
        END IF
        ! Gradjent for shelf points:
       !ELSE ! if shelf:
         !! Giving shelf points an unrealistic value, to check if 
         !! they are never used, which should be the case.
         !dVa_dx(i,j) = Huge(1.0_dp)
         !dVb_dy(i,j) = Huge(1.0_dp)
        !dVa_dx(i,j) = (Va(i+1,j)-Va(i-1,j)) / (2._dp * dx)
        !dVb_dy(i,j) = (Vb(i,j+1)-Vb(i,j-1)) / (2._dp * dy)
       END IF
    END DO
    END DO
    
     ! Giving the domain boundary the value of the direct interior neighbour:
    dVa_dx( 1,:) = dVa_dx(   2,:)
    dVa_dx(NX,:) = dVa_dx(NX-1,:)
    dVa_dx(:, 1) = dVa_dx(:,   2)
    dVa_dx(:,NY) = dVa_dx(:,NY-1)
    dVb_dy( 1,:) = dVb_dy(   2,:)
    dVb_dy(NX,:) = dVb_dy(NX-1,:)
    dVb_dy(:, 1) = dVb_dy(:,   2)
    dVb_dy(:,NY) = dVb_dy(:,NY-1)
  END SUBROUTINE xy_gradients_grounded_area
END MODULE gradient_ant_module
