MODULE configuration_ant_module
    USE configuration_main_module, ONLY: dp, C
    IMPLICIT NONE
    
    ! BdB 03/2010: contains the model constant only used for the 
    ! Antarctic Ice Sheet (not read from config file!)
    ! Constant are determined and initialized here

      ! SHELF
      ! =====
      LOGICAL            :: use_dHs_at_shelf_config        = .FALSE.
      LOGICAL            :: no_shelf_config                = .FALSE.

      ! MASS BALANCE
      ! ============
      INTEGER            :: choice_MB_basal_model_config   = 4
      REAL(dp)           :: basal_shelf_melt_config        = 2._dp

      ! TEMPERATURE
      ! ===========
      INTEGER            :: choice_Ts_model_config         = 10

      ! SLIDING
      INTEGER            :: choice_sliding_method_config   = 2

    ! TYPE DEFENITIONS
    !=================

      ! This TYPE contains all the information for the Antarctic ice-sheet
      ! these variables will never change during the run of the program 
      TYPE constants_ant_type                


                                           ! GRID DIFFERENCES IN DISCRETIZATION
                                           ! ==================================
        REAL(dp)                            :: a_x 
        REAL(dp)                            :: c_x
        REAL(dp)                            :: a_y
        REAL(dp)                            :: c_y
        REAL(dp)                            :: a_xx
        REAL(dp)                            :: b_xx
        REAL(dp)                            :: c_xx
        REAL(dp)                            :: a_yy
        REAL(dp)                            :: b_yy
        REAL(dp)                            :: c_yy
                                            
        REAL(dp), DIMENSION(:), ALLOCATABLE :: a_zeta
        REAL(dp), DIMENSION(:), ALLOCATABLE :: b_zeta
        REAL(dp), DIMENSION(:), ALLOCATABLE :: c_zeta
        REAL(dp), DIMENSION(:), ALLOCATABLE :: a_zetazeta
        REAL(dp), DIMENSION(:), ALLOCATABLE :: b_zetazeta
        REAL(dp), DIMENSION(:), ALLOCATABLE :: c_zetazeta

        REAL(dp)                            :: a_N
        REAL(dp)                            :: b_1

        REAL(dp)                            :: z_x_minus
        REAL(dp)                            :: a_x_minus
        REAL(dp)                            :: b_x_minus

        REAL(dp)                            :: b_x_plus
        REAL(dp)                            :: c_x_plus
        REAL(dp)                            :: d_x_plus

        REAL(dp)                            :: z_y_minus
        REAL(dp)                            :: a_y_minus
        REAL(dp)                            :: b_y_minus

        REAL(dp)                            :: b_y_plus
        REAL(dp)                            :: c_y_plus
        REAL(dp)                            :: d_y_plus

        REAL(dp), DIMENSION(:), ALLOCATABLE :: z_zeta_minus
        REAL(dp), DIMENSION(:), ALLOCATABLE :: a_zeta_minus
        REAL(dp), DIMENSION(:), ALLOCATABLE :: b_zeta_minus
                                                
        REAL(dp), DIMENSION(:), ALLOCATABLE :: b_zeta_plus
        REAL(dp), DIMENSION(:), ALLOCATABLE :: c_zeta_plus
        REAL(dp), DIMENSION(:), ALLOCATABLE :: d_zeta_plus
                                                
        REAL(dp)                            :: b_t
        REAL(dp)                            :: s_t

        REAL(dp)                            :: a_zrzr
        REAL(dp)                            :: b_zrzr
        REAL(dp)                            :: c_zrzr

        REAL(dp)                            :: b_zr_plus
        REAL(dp)                            :: c_zr_plus
        REAL(dp)                            :: d_zr_plus

        REAL(dp)                            :: z_zr_minus
        REAL(dp)                            :: a_zr_minus
        REAL(dp)                            :: b_zr_minus

                                ! SHELF
                                ! =====
        LOGICAL                  :: use_dHs_at_shelf
        LOGICAL                  :: no_shelf

                                ! MASS BALANCE
                                !===============================
        INTEGER                  :: choice_MB_basal_model
        REAL(dp)                 :: basal_shelf_melt

                                ! TEMPERATURE
                                ! ===========
        INTEGER                  :: choice_Ts_model

                                ! SLIDING
                                ! ===========
        INTEGER                  :: choice_sliding_method
      END TYPE constants_ant_type
      
      ! C_ANT is the 'struct' containing all the Constants for the Antarctic ice sheet
      TYPE(constants_ant_type), SAVE :: C_ANT

      TYPE time_ant_type
        ! This type contains two global variables, treat them with highest care!
        REAL(dp) :: dt          ! time step of the model, can be changed when time stepping is dynamical
        REAL(dp) :: time        ! time (in years) of the run, increases every time step
        REAL(dp) :: year        ! summing a year for time steps lower than 1 year
      END TYPE time_ant_type
      
      ! T_ANT is the 'struct' containing dt and time
      TYPE(time_ant_type), SAVE :: T_ANT

CONTAINS

  SUBROUTINE initialize_ant_constants()

    ! This routine puts all the constants which will never change during the run 
    ! into a special constant 'struct', for Antarctica this is C_ANT 

    ! Choosing between using the dHi and the dHs in the shelf scheme:
    C_ANT%use_dHs_at_shelf = use_dHs_at_shelf_config
    C_ANT%no_shelf = no_shelf_config

    ! Mass balance choices
    C_ANT%choice_MB_basal_model    = choice_MB_basal_model_config
    C_ANT%basal_shelf_melt         = basal_shelf_melt_config

    ! There are a few options to adjust the surface temperature due to elevation change
    ! For the monthly temperature there are 2:
    !  9 --> the Lipzig parameterization, used for Antarctica, using 2 lapse rates above and below 1.5 km
    ! 10 --> a constant lapse rate of 8 K/km used for all NH ice sheets
    C_ANT%choice_Ts_model          = choice_Ts_model_config

    ! There are a two options to calculate the basal sliding velocity
    !  0 --> no sliding
    !  1 --> Weertman type sliding law, calculated in velocity_ant_module
    !  2 --> SSA as sliding, calculated in shelf_ant_module
    C_ANT%choice_sliding_method    = choice_sliding_method_config

    ! Initialization of the discretization coefficients
    ! uses C%NZ,  C%zeta, C%dt_thermo from configuration_main_module.f90
    CALL initialize_discretization()

  END SUBROUTINE initialize_ant_constants  


  SUBROUTINE initialize_discretization()
    ! See table 3 and table 1.
    ! To be called in anice, not only usefull for uvw-mumps, but also for temperature
    ! Calculating the discretization coefficents for a Arakawa A-grid discretization
    
    ! Local variables:
    INTEGER                       :: k
    REAL(dp), DIMENSION(2:C%NZ)   :: a_k
    REAL(dp), DIMENSION(1:C%NZ-1) :: b_k
    REAL(dp), DIMENSION(3:C%NZ)   :: c_k
    REAL(dp), DIMENSION(1:C%NZ-2) :: d_k

    ALLOCATE(C_ANT%a_zeta(2:C%NZ-1))
    ALLOCATE(C_ANT%b_zeta(2:C%NZ-1))
    ALLOCATE(C_ANT%c_zeta(2:C%NZ-1))
    ALLOCATE(C_ANT%a_zetazeta(2:C%NZ-1))
    ALLOCATE(C_ANT%b_zetazeta(2:C%NZ-1))
    ALLOCATE(C_ANT%c_zetazeta(2:C%NZ-1))
    
    ALLOCATE(C_ANT%z_zeta_minus(3:C%NZ))
    ALLOCATE(C_ANT%a_zeta_minus(3:C%NZ))
    ALLOCATE(C_ANT%b_zeta_minus(3:C%NZ))
    ALLOCATE(C_ANT%b_zeta_plus(1:C%NZ-2))
    ALLOCATE(C_ANT%c_zeta_plus(1:C%NZ-2))
    ALLOCATE(C_ANT%d_zeta_plus(1:C%NZ-2))
  
    C_ANT%a_x  = -1._dp / (2._dp * C%dx_ant)
    C_ANT%c_x  =  1._dp / (2._dp * C%dx_ant)
    C_ANT%a_y  = -1._dp / (2._dp * C%dy_ant)
    C_ANT%c_y  =  1._dp / (2._dp * C%dy_ant)
    C_ANT%a_xx =  1._dp / C%dx_ant**2
    C_ANT%b_xx = -2._dp / C%dx_ant**2
    C_ANT%c_xx =  1._dp / C%dx_ant**2
    C_ANT%a_yy =  1._dp / C%dy_ant**2
    C_ANT%b_yy = -2._dp / C%dy_ant**2
    C_ANT%c_yy =  1._dp / C%dy_ant**2
    
    DO k = 2, C%NZ
     a_k(k) = C%zeta(k)   - C%zeta(k-1)
    END DO
    DO k = 1, C%NZ-1
     b_k(k) = C%zeta(k+1) - C%zeta(k)
    END DO
    DO k = 3, C%NZ
     c_k(k) = C%zeta(k)   - C%zeta(k-2)
    END DO
    DO k = 1, C%NZ-2
     d_k(k) = C%zeta(k+2) - C%zeta(k)
    END DO
    C_ANT%a_N = a_k(C%NZ)
    C_ANT%b_1 = b_k(1)
    
    DO k = 2, C%NZ-1
     C_ANT%a_zeta(k)     =         - b_k(k)  / (a_k(k) * (a_k(k) + b_k(k))) 
     C_ANT%b_zeta(k)     = (b_k(k) - a_k(k)) / (a_k(k) *  b_k(k))  
     C_ANT%c_zeta(k)     =           a_k(k)  / (b_k(k) * (a_k(k) + b_k(k)))
     C_ANT%a_zetazeta(k) =           2._dp   / (a_k(k) * (a_k(k) + b_k(k)))
     C_ANT%b_zetazeta(k) =         - 2._dp   / (a_k(k) *  b_k(k))       
     C_ANT%c_zetazeta(k) =           2._dp   / (b_k(k) * (a_k(k) + b_k(k)))
    END DO

    ! Not all of these are in use:
    DO k = 1, C%NZ-2
      C_ANT%b_zeta_plus(k) = -(b_k(k) + d_k(k)) / (b_k(k) * d_k(k))
      C_ANT%c_zeta_plus(k) =            d_k(k)  / (b_k(k) * (d_k(k) - b_k(k)))   
      C_ANT%d_zeta_plus(k) =            b_k(k)  / (d_k(k) * (b_k(k) - d_k(k)))          
    END DO

    ! Not all of these are in use:
    DO k = 3, C%NZ
      C_ANT%z_zeta_minus(k) =           a_k(k)  / (c_k(k) * (c_k(k) - a_k(k)))   
      C_ANT%a_zeta_minus(k) =           c_k(k)  / (a_k(k) * (a_k(k) - c_k(k)))   
      C_ANT%b_zeta_minus(k) = (a_k(k) + c_k(k)) / (a_k(k) * c_k(k))          
    END DO
    
    C_ANT%z_x_minus =  1._dp / (2._dp * C%dx_ant)
    C_ANT%a_x_minus = -2._dp /          C%dx_ant
    C_ANT%b_x_minus =  3._dp / (2._dp * C%dx_ant)

    C_ANT%b_x_plus = -3._dp / (2._dp * C%dx_ant)
    C_ANT%c_x_plus =  2._dp /          C%dx_ant
    C_ANT%d_x_plus = -1._dp / (2._dp * C%dx_ant)

    C_ANT%z_y_minus =  1._dp / (2._dp * C%dy_ant)
    C_ANT%a_y_minus = -2._dp /          C%dy_ant
    C_ANT%b_y_minus =  3._dp / (2._dp * C%dy_ant)

    C_ANT%b_y_plus = -3._dp / (2._dp * C%dy_ant)
    C_ANT%c_y_plus =  2._dp /          C%dy_ant
    C_ANT%d_y_plus = -1._dp / (2._dp * C%dy_ant)
    
    ! Discretization coefficients in time
    C_ANT%b_t = -1._dp / C%dt_thermo                      ! See equations 14.38
    C_ANT%s_t =  1._dp / C%dt_thermo                      ! See equations 14.39
    
  END SUBROUTINE initialize_discretization
END MODULE configuration_ant_module
