MODULE check_timestep_size_ant_module
  USE configuration_main_module, ONLY: dp, C
  USE configuration_ant_module, ONLY: T_ANT
  IMPLICIT NONE

CONTAINS
  SUBROUTINE check_timestep_size(D_2D, Us, Vs, U, V, mask, critical_time_step)
    ! This subroutine checks the time step with the CFK-criterium
    ! and can be used to change the time step. 
    ! Methods are similar to those in PISM (see also Bueler  et al., 2007)

    TYPE value_at_location_ant                
      REAL(dp):: value
      INTEGER :: i_value
      INTEGER :: j_value
      INTEGER :: k_value
    END TYPE value_at_location_ant

    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: D_2D
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Us
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: Vs
    INTEGER,  DIMENSION(C%NX_ant,C%NY_ant     ), INTENT(IN)  :: mask
    
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: U
    REAL(dp), DIMENSION(C%NX_ant,C%NY_ant,C%NZ), INTENT(IN)  :: V

    ! Output variable:
    REAL(dp),                                    INTENT(OUT) :: critical_time_step
    
    ! Local variables:
    INTEGER                                     :: i,j,k
    TYPE(value_at_location_ant)                 :: D_2D_max
    TYPE(value_at_location_ant)                 :: V_3D_SIA_max
    TYPE(value_at_location_ant)                 :: V_2D_SSA_max
    REAL(dp)                                    :: D_2D_dt   
    REAL(dp)                                    :: V_3D_SIA_dt
    REAL(dp)                                    :: V_2D_SSA_dt

    D_2D_max%value     = 0._dp
    V_3D_SIA_max%value = 0._dp
    V_2D_SSA_max%value = 0._dp

    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      ! As in PISM we check the three 'domains' of diffusivity (SIA), the grounded 3-D SIA velocities
      ! and the shelf velocities (SSA). Equations are as in Bueler et al. (JoG, 2007)
      
      IF(mask(i,j) /= C%type_shelf) THEN

!!--    diffusivity (vertical averaged SIA)
        IF ( -D_2D(i,j) > D_2D_max%value ) THEN
          D_2D_max%value   = -D_2D(i,j)
          D_2D_max%i_value = i
          D_2D_max%j_value = j
        END IF  

!!--    3-D velocities, SIA  and Sliding velocities
        DO k = 1, C%NZ
          IF ( (ABS(U(i,j,k)) + ABS(V(i,j,k))) > V_3D_SIA_max%value ) THEN
            V_3D_SIA_max%value   = ABS(U(i,j,k)) + ABS(V(i,j,k))
            V_3D_SIA_max%i_value = i
            V_3D_SIA_max%j_value = j
            V_3D_SIA_max%k_value = k
          END IF  
        END DO

      ELSE

!!--    Shelf velocities, based on SSA calculation
        IF ( (ABS(Us(i,j)) + ABS(Vs(i,j))) > V_2D_SSA_max%value ) THEN
          V_2D_SSA_max%value   = ABS(Us(i,j)) + ABS(Vs(i,j))
          V_2D_SSA_max%i_value = i
          V_2D_SSA_max%j_value = j
        END IF

      END IF
    END DO
    END DO

    ! PISM: 
    ! dt/2 * (1/dx**2 + 1/dy**2) * D-2d_max le 0.12
    ! dt * max(abs(U/dx) + abs(V/dy) + abs(W/dz)) le 1    ! in 3-D!
    
    ! Determine the critical time step per 'domain' and find the minimum time step required 
    ! according to the CFL condition, for which T_ANT%dt should be smaller than the critical_time_step
    ! NOTE!: Only using C%dx not C%dy (which should stay the same!!)

    D_2D_dt     = C%dx_ant * C%dx_ant / (6._dp * D_2D_max%value)    ! same as in original model
    V_3D_SIA_dt =            C%dx_ant / V_3D_SIA_max%value          ! in original model this is divided by 2 (but added 2 velocities so..)
    V_2D_SSA_dt =            C%dx_ant / V_2D_SSA_max%value

    ! Set time step to an arbitrary number when velocities or diffusivity are zero, as long as it is larger than C%dt_max
    IF (D_2D_max%value     == 0._dp ) D_2D_dt     = C%dt_climate_record
    IF (V_3D_SIA_max%value == 0._dp ) V_3D_SIA_dt = C%dt_climate_record
    IF (V_2D_SSA_max%value == 0._dp ) V_2D_SSA_dt = C%dt_climate_record
    
     critical_time_step = MIN(C%dt_max, D_2D_dt,V_3D_SIA_dt,V_2D_SSA_dt)

  IF (.FALSE.) THEN
  
    IF (critical_time_step >= C%dt_max) THEN
      WRITE(UNIT=C%stdlog, FMT='(A,F8.2,A,F12.4)')          &
       'Maximum step size for ANT is C%dt_max  = ', 1._dp,  & 
       '%, no ice or stepsizes are larger than the maximum time step at time ',T_ANT%time
  
    ELSE
      IF ( D_2D_dt < V_3D_SIA_dt .AND. D_2D_dt < V_2D_SSA_dt ) WRITE(UNIT=C%stdlog, FMT='(A,F8.2,A,3I5,A,F12.4)') &
       'Maximum step size for ANT is dt(D_2D)      = ', 100._dp * T_ANT%dt / D_2D_dt, & 
       '%, at location: ',D_2D_max%i_value, D_2D_max%j_value, mask(D_2D_max%i_value,D_2D_max%j_value), ', at time ',T_ANT%time

      IF ( V_3D_SIA_dt < D_2D_dt .AND. V_3D_SIA_dt < V_2D_SSA_dt ) WRITE(UNIT=C%stdlog, FMT='(A,F8.2,A,3I5,A,F12.4)') &
       'Maximum step size for ANT is dt(V_3D(SIA)) = ', 100._dp * T_ANT%dt / V_3D_SIA_dt, & 
       '%, at location: ',V_3D_SIA_max%i_value, V_3D_SIA_max%j_value, V_3D_SIA_max%k_value, ', at time ',T_ANT%time

      IF ( V_2D_SSA_dt < D_2D_dt .AND. V_2D_SSA_dt < V_3D_SIA_dt ) WRITE(UNIT=C%stdlog, FMT='(A,F8.2,A,3I5,A,F12.4)') &
       'Maximum step size for ANT is dt(V_2D(SSA)) = ', 100._dp * T_ANT%dt / V_2D_SSA_dt, & 
       '%, at location: ',V_2D_SSA_max%i_value, V_2D_SSA_max%j_value, mask(V_2D_SSA_max%i_value,V_2D_SSA_max%j_value), &
       ', at time ',T_ANT%time
    END IF
    
  END IF
    
    CALL FLUSH(C%stdlog)
    
  END SUBROUTINE check_timestep_size
    

  SUBROUTINE determine_if_action_is_required(dt_of_action, time_since_last_action, do_action)
    ! Determine if the 'action' has to be repeated this time step or not
    REAL(dp), INTENT(IN)       :: dt_of_action                 ! time step of the routine in months
    REAL(dp), INTENT(INOUT)    :: time_since_last_action       ! time step in months since last call
    LOGICAL,  INTENT(OUT)      :: do_action                    ! do the actions
    
    ! Input variable:
    ! Decide which actions will be done this time iteration:
    IF(time_since_last_action >= dt_of_action ) THEN
      time_since_last_action = T_ANT%dt
      do_action = .TRUE.
    ELSE
      time_since_last_action = time_since_last_action + T_ANT%dt
      do_action = .FALSE.
    END IF

  END SUBROUTINE determine_if_action_is_required
END MODULE check_timestep_size_ant_module
