MODULE ant_grice_module
  ! Contains the main subroutines of the ANICE Ice-Sheet Model 
  ! (grice version: without mumps shelf routines) for Antarctica.

  USE configuration_main_module, ONLY: dp, C, Model_Time, Called_Selen, Ocean_Area_Earth
  ! C contains the global parameters, used for all ice sheets

  USE configuration_ant_module, ONLY: C_ANT, T_ANT, initialize_ant_constants
  ! C_ANT containt the constant for Antarctica (use_dHs_at_shelf, no_shelf)
  
  IMPLICIT NONE
  
  INTEGER,                       SAVE :: recording_recordnr
  INTEGER,                       SAVE :: restart_recordnr

  REAL(dp),                      SAVE :: time_since_last_shelf_calculation
  REAL(dp),                      SAVE :: time_since_last_thermodynamical_calculation
  REAL(dp),                      SAVE :: time_since_last_bedrock_adjustment
  REAL(dp),                      SAVE :: time_since_last_written_restart_file
  REAL(dp),                      SAVE :: time_since_last_written_record_file

CONTAINS
  SUBROUTINE main_ant_grice_model(temperature_offset, dHb_dt_SLE, deltaS_SLE, Slev_change, MeanIso, SlevIso, Load_Hi_ant)

  USE parameters_main_module, ONLY: seawater_density, ice_density, T0
  USE gradient_ant_module, ONLY: derivatives_of_Hi_and_Hs
  USE zeta_derivatives_ant_module, ONLY: calculate_zeta_derivatives, dzeta
  USE restart_file_ant_module, ONLY: netcdf_file_type_ant, open_netcdf_file, inquire_restart_file, &
        read_restart_file, write_restart_file, write_help_fields_file, close_restart_file, &
        close_help_fields_file, create_restart_file, create_help_fields_file, close_netcdf_file

  USE time_recording_ant_module, ONLY: recording_file_type_ant, create_recording_fields_file, write_recording_fields_file, &
        close_recording_file
  USE shelf_ant_module, ONLY: ssa_velocity, basal_yield_stress
  USE surface_height_ant_module, ONLY: surface_height, adapt_Hi_and_Hs_for_no_shelf
  USE ice_surface_temperature_ant_module, ONLY: ice_surface_temperature
  USE reference_fields_ant_module, ONLY: read_reference_fields, Tmon_ref, Hb_ref, Heatflux, vaf_ref, slev_ref
  USE massbalance_ant_module, ONLY: surface_massbalance, basal_massbalance
  USE ice_thickness_ant_module, ONLY: ice_thickness
  USE thermodynamics_ant_module, ONLY: bottom_frictional_heating, temperature
  USE velocity_ant_module, ONLY: calculate_D_uv_3D, sheet_velocity_2D, grounded_area_velocity_3d
  USE bedrock_ant_module, ONLY: initialize_bedrock, bed
  USE mask_ant_module, ONLY: determine_mask
  USE check_timestep_size_ant_module, ONLY: check_timestep_size, determine_if_action_is_required
  USE write_ascii_output_ant_module, ONLY: write_means_and_mb_icesheet
  USE isotopes_ant_module, ONLY: oxygen_isotope_icesheet

  ! Input variables:
  REAL(dp),                                 INTENT(IN)  :: temperature_offset ! relative to PD temperature (in Kelvin, mean over 40-80N)

  ! Input from SELEN, change in bedrock (per time step) and change in sea level geoid height (relative to PD):
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant),  INTENT(IN)  :: dHb_dt_SLE
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant),  INTENT(IN)  :: deltaS_SLE      ! when one value of the eustatic sea level is needed, deltaS_SLE(1,1) is used

  ! Output variables:
  REAL(dp),                                 INTENT(OUT) :: MeanIso         ! mean d18O of the ice sheet (VPDB)
  REAL(dp),                                 INTENT(OUT) :: SlevIso         ! Ice Volume (in m s.e.) used to calculate MeanIso 
  REAL(dp),                                 INTENT(OUT) :: Slev_change     ! Total contribution to sea level change (mseq)

  ! Output for Selen, ice loading on land
  REAL(dp), DIMENSION(C%NX_ant * C%NY_ant), INTENT(OUT) :: Load_Hi_ant
  
  ! Local types and variables:
  TYPE(netcdf_file_type_ant)                            :: initial_file          ! Contains all fields necessary to start a run
  INTEGER                                               :: final_time_loop_nr    ! end of ice-sheet model loop (default is 100 years)
  INTEGER                                               :: i, j, jj
  LOGICAL                                               :: do_shelf_velocity
  LOGICAL                                               :: do_thermodynamics
  LOGICAL                                               :: do_write_restart_file     = .FALSE.
  LOGICAL                                               :: do_write_recording_output = .FALSE.
  LOGICAL                                               :: do_bedrock_adjustment

  REAL(dp)                                              :: tdum                   ! Start time [years ago]
  REAL(dp)                                              :: domain_leaving_flux_per_dt
  REAL(dp)                                              :: mass_conservation      ! In case all ice volume is conserved within the area (2:C%NY_ant-1,2:C%NX_ant-1) this variable is zero
  REAL(dp)                                              :: critical_time_step
  REAL(dp)                                              :: VolIso                 ! Ice Volume used to calculate MeanIso 

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!    ALLOCATE AND SAVE THE INITIAL FIELDS OF THE ICE-SHEET    !!!!
!!!!        same fields as are read from initial_file            !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: Hi_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: Hb_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: Hs_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: dHi_dt_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: dHb_dt_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: U_ssa_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: V_ssa_mem
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: bmelt_mem
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: Ti_mem
  INTEGER,  DIMENSION(:,:),   ALLOCATABLE, SAVE :: mask_mem
  INTEGER,  DIMENSION(:,:),   ALLOCATABLE, SAVE :: cmask_mem  
  INTEGER,  DIMENSION(:,:),   ALLOCATABLE, SAVE :: bmask_mem
  REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: T2m_mem

  ! because not calculated every time step
  REAL(dp), DIMENSION(:,:),   ALLOCATABLE, SAVE :: icedis_field        ! field of ice discharge, sum should be the same as total_icedis in output

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!           END OF ALLOCATE AND SAVE INITIAL FIELDS           !!!!
!!!!             remaining fields are only allocated             !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hi                  ! Ice thickness [m] at time step t
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hs                  ! Surface height [m], relative to present day sea level
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hb                  ! Bedrock height [m], relative to present day sea level
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHi_dt              ! Time derivative of the ice thickness [m yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHb_dt              ! Time derivative of the bedrock height [m yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Us                  ! Vertically averaged velocity in the x-direction [m yr^-1] at time step t  
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Vs                  ! Vertically averaged velocity in the y-direction [m yr^-1] at time step t
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: bottom_melt         ! basal melting underneath the ice sheet (m yr^-1) - CURRENTLY NOT CALCULATED
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: Ti                  ! Temperature field in the ice part [K] at time step t
  INTEGER,  DIMENSION(C%NX_ant, C%NY_ant      ) :: mask                ! Distinguishes the sheet, the shelf and the boundary (= groundline) points by a mask
  INTEGER,  DIMENSION(C%NX_ant, C%NY_ant      ) :: mask_binary         ! Distinguishes the sheet and the shelf (0) from the ocean (1) points by a mask !CVC

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hi_new              ! Ice thickness [m] at time step t+dt
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hb_new              ! Bedrock height [m], relative to present day sea level
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHb              ! CvC Bedrock height change [m], relative to present day sea level

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: Ti_new              ! Temperature field in the ice part [K] at time step t+dt

  ! SIA and SSA velocities solved seperately
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: U_SIA               ! Vertically averaged velocity in the x-direction for grounded ice
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: V_SIA               ! Vertically averaged velocity in the y-direction for grounded ice
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: U_SSA               ! SSA solution in the x direction
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: V_SSA               ! SSA solution in the y direction
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: U_base              ! basal velocities in the x direction
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: V_base              ! basal velocities in the y direction
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: tau_yield           ! basal yield stress (Pa)
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Hi_load             ! Ice loading on land for output
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: total_load_ant      ! Ice loading on land and oceanload !CvC
  ! REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: RSL      ! Ice loading on land and oceanload !CvC
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Init_ice_load      ! Ice loading on land and oceanload !CvC


  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHb_dt_tot          ! Total change in bedrock since first time step

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHi_dx              ! Horizontal derivative in x-direction of the ice thickness
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHi_dy              ! Horizontal derivative in y-direction of the ice thickness
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHs_dx              ! Horizontal derivative in x-direction of the surface height
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: dHs_dy              ! Horizontal derivative in y-direction of the surface height
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: shelf_dHs_dx        ! Horizontal derivative in x-direction of the surface height used in the shelf-velocity calculation
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: shelf_dHs_dy        ! Horizontal derivative in y-direction of the surface height used in the shelf-velocity calculation

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: MB_surface          ! Surface Mass Balance [mieq = meter ice equivalent]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: MB_basal            ! Basal Mass Balance [mieq = meter ice equivalent]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: MB_applied_per_dt   ! Applied Mass Balance per T_ANT%dt [mieq = meter ice equivalent per T_ANT%dt]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Accumulation        ! Accumulation from surface_massbalance_Bintanja routine  
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Ablation            ! Ablation from surface_massbalance_Bintanja routine
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Refreezing          ! Refreezing

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: IsoIce              ! d18O of the ice-sheet
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant,   12) :: T2m_mon             ! monthly 2m air temperature (Kelvin)          
  REAL(dp)                                      :: Mean_Tsurf          ! mean temperature (Kelvin)

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant,    6) :: fluxes_per_dt       ! All fluxes (from 2:NX-1,2:NY-1) In meter per T_ANT%dt   >= 0
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: Fr                  ! Geothermal heat flux
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: Ti_pmp              ! The melting temperature, depending on pressure [K]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: A_flow              ! The flow parameter at each x,y,zeta point
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: A_flow_mean         ! The flow parameter at each x,y point
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: Cpi                 ! The specific heat capacity of ice at each x,y,zeta point [J kg^-1 K^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: Ki                  ! The conductivity of ice at each x,y,zeta point [J m^-1 K^-1 yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: sliding_coverage    ! 1 if a sliding point, 0 else
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: frictional_heating  ! The frictional heating flux
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: U                   ! The 3D velocity field in the x-direction [m yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: V                   ! The 3D velocity field in the y-direction [m yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: W                   ! The 3D velocity field in the z(eta)-direction [m yr^-1]
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ) :: D_uv_3D             ! The shared part of the U and V velocity, before vertically averaging.
  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: D_2D                ! Vertically averaged diffusion coefficient, an help variable
  INTEGER,  DIMENSION(C%NX_ant, C%NY_ant      ) :: calve_mask          ! The mask with a calving line

  REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: lambda_pore_water   ! scaling of pore water pressure, identifing saturation of till
  !REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ) :: bottom_heat_ratio   ! ratio of bottom layer temperature relative to pmp

  REAL(dp), DIMENSION(C%NX_ant,C%NY_ant       ) :: transmis            ! transmissivity of the atmosphere


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!        ONLY INITIALIZE AT THE FIRST MODEL TIME STEP         !!!!
!!!!              allocation every model time step               !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  IF (Model_Time == C%start_time_of_run) THEN

    WRITE(UNIT=*,FMT='(a)') ' Program grice_ant starts calculating......'
    WRITE(UNIT=*,FMT='(A)') ' Initializing...'

    ! Initialization of the struct  C_ANT% :
    CALL initialize_ant_constants()
    
    ! Added BdB 11/2009: write data to ascii file, for plotting with gnuplot
    OPEN (UNIT=4001, FILE=TRIM(C%output_dir)//'ANICE4_output_antarctica'//TRIM(C%filename_extension), STATUS='replace', ACTION='write')

    ! Added BdB 11/2011: write data to ascii file for the total mass change
    OPEN (UNIT=4011, FILE=TRIM(C%output_dir)//'ANICE4_output_ant_total_mass_change'//TRIM(C%filename_extension), STATUS='replace', ACTION='write')

    ! added DoE 24/10
    OPEN (UNIT=27111992, FILE=TRIM(C%output_dir)//'relaxation_time_caps'//TRIM(C%filename_extension), STATUS='replace', ACTION='write')
    OPEN (UNIT=27111993, FILE=TRIM(C%output_dir)//'wavelength_caps'//TRIM(C%filename_extension), STATUS='replace', ACTION='write')
    ! added CvC 06/11
	! OPEN (UNIT=27111993, FILE=TRIM(C%output_dir)//'Hb_new'//TRIM(C%filename_extension), STATUS='replace', ACTION='write')


    WRITE(4001, '(A2,A)')         '%%','Output for Antarctica'
    WRITE(4001, '(A2,6A12,5A18)') '%%', 'Time (ka)', 'Sea level', 'T offset', 'Mean Tsurf', 'Mean Iso', 'Slev contr.',       &
                                  'Ice Volume', 'Vol. Land', 'Vol Shelf', 'Area Land', 'Area Shelf'

    WRITE(4011, '(A2,A)')               '%% ',' Total mass changes for Antarctica in Gt per year'
    WRITE(4011, '(A2,A12,8A14,A8,3A10)') '%%','Time (ka)', 'dHi per yr', 'MB per yr', 'IceDis per yr', 'BMB per yr',         &
                                        'SMB per yr', 'Acc per yr','Abl per yr', 'Rfr per yr', 'fac.','weight','DdT','DQ80S'
  
  END IF ! end of first time-step
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!       END INITIALIZATION AT THE FIRST MODEL TIME STEP       !!!!
!!!!              allocation every model time step               !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  ! Allocation of the components of the type dzeta in  zeta_derivatives_module
  ALLOCATE(dzeta%t                (C%NX_ant, C%NY_ant, C%NZ ))
  ALLOCATE(dzeta%x                (C%NX_ant, C%NY_ant, C%NZ ))
  ALLOCATE(dzeta%y                (C%NX_ant, C%NY_ant, C%NZ ))
  ALLOCATE(dzeta%z                (C%NX_ant, C%NY_ant       ))

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!        ONLY INITIALIZE AT THE FIRST MODEL TIME STEP         !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  IF (Model_Time == C%start_time_of_run) THEN

    ! Allocate the memory of the initial fields
    ALLOCATE(Hi_mem                 (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Hb_mem                 (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Hs_mem                 (C%NX_ant, C%NY_ant       ))
    ALLOCATE(dHi_dt_mem             (C%NX_ant, C%NY_ant       ))
    ALLOCATE(dHb_dt_mem             (C%NX_ant, C%NY_ant       ))
    ALLOCATE(U_ssa_mem              (C%NX_ant, C%NY_ant       ))
    ALLOCATE(V_ssa_mem              (C%NX_ant, C%NY_ant       ))  
    ALLOCATE(bmelt_mem              (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Ti_mem                 (C%NX_ant, C%NY_ant, C%NZ ))
    ALLOCATE(mask_mem               (C%NX_ant, C%NY_ant       ))
    ALLOCATE(cmask_mem              (C%NX_ant, C%NY_ant       ))  
    ALLOCATE(bmask_mem              (C%NX_ant, C%NY_ant       ))
    
    ALLOCATE(T2m_mem                (C%NX_ant, C%NY_ant, 12   ))

    ALLOCATE(icedis_field           (C%NX_ant, C%NY_ant       ))    

    ! Read the essential initial fields:  Hi, Hb, Ti, Us, Vs  from the initial netcdf file:
    ! Output: initial_file 
    CALL open_netcdf_file(C%initial_ant_filename, initial_file) 

    ! Output: file ID and its variable ID's which are global defined in restart_file_ant_module.f90
    CALL inquire_restart_file(initial_file)

    ! Output: Hi_mem, Hb_mem, Hs_mem, Ti_mem, U_ssa_mem, V_ssa_mem, tdum
    CALL read_restart_file(initial_file, C%starting_recordnr, Hi_mem, Hb_mem, Hs_mem, U_ssa_mem, V_ssa_mem, bmelt_mem, Ti_mem, tdum)
    
    ! In/output: initial_file
    CALL close_netcdf_file(initial_file)

    ! Initialize the reference fields (e.g. present day fields), used for the MB_surface scaling 
    ! for obtaining the reference load needed for scaling the bedrock adjustment):
    CALL read_reference_fields(Hi_mem, Hb_mem)

    ! Initialize the reference load with which the bedrock adjustment is scaled:
    CALL initialize_bedrock()

    ! initialize
    frictional_heating = 0._dp
    !bottom_heat_ratio  = 0._dp
    dHi_dt_mem = 0._dp
    dHb_dt_mem = 0._dp
   
    IF(ANY(Hi_mem < C%Hi_min)) THEN
     WRITE(UNIT=*,FMT='(A)')  ' There are Hi values detected lower then C%Hi_min, this might cause problems with the dzeta calculation' 
     WRITE(UNIT=*,FMT='(4A)') ' Check the initial file: ', TRIM(C%initial_ant_filename)
    END IF

    IF(C%enable_restart_file) THEN
     ! Create the restart file: 
     ! Output: restart_file; Optional: -
     CALL create_restart_file(C%NX_ant, C%NY_ant, C%dx_ant, C%dy_ant, C%restart_ant_filename)
     restart_recordnr = 1
    END IF

    IF(C%enable_help_fields_file) THEN
     ! Create the help_fields file: 
     CALL create_help_fields_file(C%NX_ant, C%NY_ant, C%dx_ant, C%dy_ant, C%help_fields_ant_filename)
    END IF

    ! Initialization of recording file (which is overwritten if existing):
    IF(C%enable_recording_file) THEN
     CALL create_recording_fields_file(C%recording_ant_filename)
     recording_recordnr = 1
    END IF

    ! Output: mask
    CALL determine_mask(Hi_mem, Hb_mem, deltaS_SLE, mask_mem, cmask_mem)

    bmask_mem = mask_mem
    !CvC make mask_binary
    ! WHERE(mask_mem /= C%type_ocean) mask_binary = 0
    ! WHERE(mask_mem == C%type_ocean) mask_binary = 1
    WHERE(mask_mem == 1)
        bmask_mem = 0
    END WHERE
    WHERE(bmask_mem /= 0)
        bmask_mem = 1
    END WHERE
    
    open(75,file="/home/username/HetGroteKoppelScript/ANICE/results/bmask_mem.dat",status="replace")
        do i = 1,141
            write(75,750) (bmask_mem(i,j), j = 1,141)
        enddo
    close(75)	 
750 format(141(I1),' ')

    ! In/Output: Hi_mem, Hs_mem, dHi_dt_mem, U_ssa_mem, V_ssa_mem
    IF(C_ANT%no_shelf)  CALL adapt_Hi_and_Hs_for_no_shelf(mask_mem, deltaS_SLE, Hi_mem, Hs_mem, dHi_dt_mem, U_ssa_mem, V_ssa_mem)

    ! Initialize monthly temperatures with the reference field
    T2m_mon = Tmon_ref

    ! This initializations cause 'action' at the first time step, all times are in months
    time_since_last_shelf_calculation           = C%dt_shelf_velocity
    time_since_last_thermodynamical_calculation = C%dt_thermo
    time_since_last_bedrock_adjustment          = C%dt_bedrock
    time_since_last_written_restart_file        = C%dt_restart
    time_since_last_written_record_file         = C%dt_recording

    T_ANT%dt   = C%dt
    T_ANT%year = 0._dp

  END IF ! end of first time-step
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!       END INITIALIZATION AT THE FIRST MODEL TIME STEP       !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  
    ! TIME LOOP: (ADJUSTED TO A 100 YEAR TIME-LOOP WITH TIME STEP OF T_ANT%dt years)

    final_time_loop_nr = INT(C%dt_climate_record / C%dt)
    critical_time_step = C%dt
    T_ANT%time         = Model_Time
    
    ! Load the saved initial fields in the model used arrays for:
    Hi          = Hi_mem
    Hb          = Hb_mem
    Hs          = Hs_mem 
    dHi_dt      = dHi_dt_mem
    dHb_dt      = dHb_dt_mem
    U_base      = U_ssa_mem
    V_base      = V_ssa_mem
    bottom_melt = bmelt_mem
    Ti          = Ti_mem
    mask        = mask_mem
    calve_mask  = cmask_mem 
    mask_binary = bmask_mem
    T2m_mon     = T2m_mem

    WRITE (0,*) 'ANT_grice: time = ', T_ANT%time

    DO WHILE((T_ANT%time - Model_Time + 0.01_dp) < C%dt_climate_record)
      ! Output: do_shelf_velocity; In/Output: time_since_last_shelf_calculation
      CALL determine_if_action_is_required(C%dt_shelf_velocity, time_since_last_shelf_calculation, do_shelf_velocity)

      ! Output: do_thermodynamics; In/Output: time_since_last_thermodynamical_calculation
      CALL determine_if_action_is_required(C%dt_thermo, time_since_last_thermodynamical_calculation, do_thermodynamics)

      ! Output: do_bedrock_adjustment; In/Output: time_since_last_bedrock_adjustment
      CALL determine_if_action_is_required(C%dt_bedrock, time_since_last_bedrock_adjustment, do_bedrock_adjustment)

      ! The routines that are not CALLed every time step are set to be called every first new loop:
      IF(T_ANT%time == Model_Time) THEN
        do_thermodynamics = .TRUE.
        do_shelf_velocity = .TRUE.

        ! Also reset the time_since variables
        time_since_last_shelf_calculation           = T_ANT%dt
        time_since_last_thermodynamical_calculation = T_ANT%dt
      END IF

      ! CvC : start 3D GIA routine
      IF(do_bedrock_adjustment) THEN
          open (91, file = '/home/username/HetGroteKoppelScript/ANICE/input_fields/antarctica/fem_deflection_to_ANICE.dat', status = 'old')
           do i = 1,141
             read(91,*) (dHb(i,j),j=1,141) !! Is this the right way of reading it?? The output below is correct using j,i instead of i,j
           enddo
          close(91)
          
            open(94,file="/home/username/HetGroteKoppelScript/ANICE/results/dHb.dat",status="replace")
             do i = 1,141
               write(94,940) (dHb(i,j), j = 1,141)
             enddo
             close(94)	 
940         format(141(e15.4))
          
          dHb_dt = dHb / C%dt_bedrock
            open(95,file="/home/username/HetGroteKoppelScript/ANICE/results/dHb_dt.dat",status="replace")
             do i = 1,141
               write(95,950) (dHb_dt(i,j), j = 1,141)
             enddo
             close(95)	 
950         format(141(e15.4))

          Hb_new = Hb + dHb

          dHb_dt_tot = Hb_new - Hb_ref
          open(96,file="/home/username/HetGroteKoppelScript/ANICE/results/dHb_dt_tot.dat",status="replace")
           do i = 1,141
             write(96,960) (dHb_dt_tot(i,j), j = 1,141)
           enddo
          close(96)	 
960       format(141(e15.4))

      END IF
          ! HB UPDATE: spread out Hb change over the time step
          Hb = Hb + dHb_dt * T_ANT%dt
		  ! CvC: End 3D GIA routine

    ! DERIVATIVES
      ! Choose one of next cases, if using a parameterizised shelf choose the shelf independent gradient:

      ! Output: dHi_dx, dHi_dy, dHs_dx, dHs_dy
      CALL derivatives_of_Hi_and_Hs(.TRUE., mask, Hi, dHi_dx, dHi_dy, Hs, dHs_dx, dHs_dy)
      ! (Note: dHs_dx is generaly not dHi_dx + dHb_dx, because at shelf Hs = sea level + (1 - rhoi/rhow)H,
      ! and at shelf-GL this causes a jump for GL gradients in case such a dHs_dx would be used only for 
      ! grounded points (this is quite tricky!).

    ! ZETA DERIVATIVES
      IF(do_thermodynamics) THEN  ! As long dzeta is only used in the thermodynamics.
       ! Updates the global struct  dzeta  in zeta_derivatives_module
       CALL calculate_zeta_derivatives(Hi,                                          &                                         
                                       dHi_dt,                                      &                                     
                                       dHi_dx, dHi_dy,                              &  
                                       dHi_dt + dHb_dt,                             &   ! = dHs_dt                           
                                       dHs_dx, dHs_dy)

      END IF ! End: do_thermodynamics

    ! TI_PMP, FLOW PARAMETER OF ICE, SPECIFIC HEAT CAPACITY OF ICE, CONDUCTIVITY OF ICE
      ! Output: Ti_pmp, A_flow, A_flow_mean, Cpi, Ki                                                                                                         
      CALL temperature_parameters(Hi, Ti, mask, Ti_pmp, A_flow, A_flow_mean, Cpi, Ki)                                                                        

    ! 2D VELOCITY
      ! Output: D_uv_3D[sheet,groundline]
      CALL calculate_D_uv_3D(Hi, dHs_dx, dHs_dy, Hb, deltaS_SLE, Ti(:,:,C%NZ), Ti_pmp(:,:,C%NZ), C%m_enh_sia * A_flow, mask /= C%type_shelf, D_uv_3D)

      ! Output: Us[sheet,groundline], Vs[sheet,groundline], D_2D[sheet,groundline]
      CALL sheet_velocity_2D(Hi, dHs_dx, dHs_dy, D_uv_3D, mask /= C%type_shelf, U_SIA, V_SIA, D_2D)

        ! Save SIA velocities in the complete velocity field
        IF (C_ANT%choice_sliding_method /= 2) THEN 
         WHERE(mask /= C%type_shelf) 
          Us = U_SIA
          Vs = V_SIA
         END WHERE
         U_base = Us
         V_base = Vs
        END IF
      IF(do_shelf_velocity .AND. (C_ANT%choice_sliding_method == 2 .OR. (.NOT. C_ANT%no_shelf)) ) THEN
       ! From a proper derivation it appears that the dHs used for the shelf-velocities should be the dHs as we 
       ! use everywhere, synthetic experiments confirm this. However with our antarctic experiment we still need
       ! the old method (as in Huybrechts), probably this has to do with an (unknown) equilibrium run we use as
       ! initilization. This is (one of) the most striking inconsequences left in this ice code.
       shelf_dHs_dx = dHs_dx
       shelf_dHs_dy = dHs_dy

       IF(C_ANT%use_dHs_at_shelf .eqv. .FALSE.) THEN                           !!!
        WHERE(mask==C%type_shelf)                                          !!!  
         shelf_dHs_dx = (1._dp - ice_density / seawater_density) * dHi_dx  !!!  original ANICE / Huybrechts way
         shelf_dHs_dy = (1._dp - ice_density / seawater_density) * dHi_dy  !!!  original ANICE / Huybrechts way
        END WHERE                                                          !!!  
       END IF

       ! Output: tau_yield
        CALL basal_yield_stress(Hb, mask, deltaS_SLE, tau_yield, lambda_pore_water)

      ! Output: U_SSA, V_SSA
        CALL ssa_velocity(calve_mask, shelf_dHs_dx, shelf_dHs_dy, C%m_enh_ssa * A_flow_mean, & 
                          tau_yield, U_base, V_base, U_SSA, V_SSA)

        ! update basal (SSA) velocities with new field
        U_base = U_SSA
        V_base = V_SSA
      END IF  ! end do_shelf_velocity

      IF (C_ANT%choice_sliding_method /= 2) THEN
       WHERE(mask /= C%type_shelf)
         U_base = 0._dp
         V_base = 0._dp
       END WHERE  
      END IF

      ! Combine SIA and basal velocities (in 2-D)
      Us = U_SIA + U_base
      Vs = V_SIA + V_base      

      ! set velocities to zero when ice thickness is zero
      WHERE (Hi == C%Hi_min)
       Us = 0._dp
       Vs = 0._dp
      END WHERE  

      IF(do_thermodynamics) THEN

    ! TI                                                                                                                            
       ! Calculation of the ice temperature (which is for the shelf rather simple):                                                                                      
       SELECT CASE(C%choice_temperature_model)
        CASE(1)
    ! TS, T2m_mon
         CALL ice_surface_temperature(mask, Hs, deltaS_SLE, Hi, temperature_offset, Ti(:,:,1), T2m_mon)
         
         ! Calculation of the 3D velocity field for grounded points, this velocity field is only used in the thermodynamics:
         ! Output: U, V, W  [sheet,groundline]                                                                                             
         CALL grounded_area_velocity_3d(Hi, dHb_dt, dHi_dx, dHi_dy, dHs_dx, dHs_dy, D_uv_3D, U_base, V_base, mask, U, V, W)

         IF(C%include_sliding_heatflux) THEN
          ! Output: frictional_heating                                                                                                      
          CALL bottom_frictional_heating(Hi, dHs_dx, dHs_dy, tau_yield, U(:,:,C%NZ), V(:,:,C%NZ), Ti(:,:,C%NZ), Ti_pmp(:,:,C%NZ), mask, frictional_heating)
         END IF             
         ! Use constant prescribed geothermal heat flux (see equation (8.50)):
         IF (C%monotonic_geothermal_heat_flux) THEN
           Fr = C%geothermal_heat_flux
         ! Or use the Heat flux field
         ELSE
           Fr = Heatflux * C%sec_per_year
         END IF
         ! Output: Ti_new                                                                                                        
         CALL temperature(Hi, dHs_dx, dHs_dy, Ti, Ti_pmp, Cpi, Ki, U, V, W, frictional_heating, mask, Fr, Ti_new) 
         Ti = Ti_new                                                                                                             

        CASE(2)
         ! In case of complete 3D temperatate ice: 
         Ti = Ti_pmp + C%dTi_below_T_pmp

        CASE(3)
         ! A fixed equal temperature for all points: 
         Ti = C%Ti_fixed
        CASE(4)

         ! Keep Ti at its initial value
         Ti = Ti_mem

         CALL ice_surface_temperature(mask, Hs, deltaS_SLE, Hi, temperature_offset, Ti(:,:,1), T2m_mon)

        CASE DEFAULT
         STOP ' STOP: Non valid C%choicetemperature_model'
       END SELECT
       
      END IF ! End: do_thermodynamics

    ! MB_surface and MB_basal
        ! Output: MB_surface (added MB terms to arguments; BdB 10/2009)
        CALL surface_massbalance(T2m_mon, Hi, mask, MB_surface, Accumulation, Ablation, Refreezing, Mean_Tsurf, transmis)
             
      ! Output: MB_basal
      IF (C_ANT%no_shelf) THEN
        MB_basal = 0._dp
      ELSE
        CALL basal_massbalance(mask, Hi, Hb, Hs, deltaS_SLE, temperature_offset, MB_basal)
      END IF

    ! HI
      ! Calculate new ice thickness:
      
      ! When spinning up the thermodynamics do not calculate Hi change
      IF (C%do_temperature_spin_up) THEN
        ! The first 100 years, evolve Hi on land
        IF (Model_Time < -249900._dp) THEN
          ! Output: Hi_new
          MB_surface = 0._dp
          MB_basal   = 0._dp
          CALL ice_thickness(Hi, D_2D, MB_surface + MB_basal, Hs, U_base, V_base, mask, Hi_new, MB_applied_per_dt, &
                             domain_leaving_flux_per_dt, mass_conservation, fluxes_per_dt)
 
          WHERE(mask==C%type_shelf) Hi_new = Hi
        ! All other time steps Hi is fixed for all grid points
        ELSE
          Hi_new = Hi
        END IF
      ELSE
        ! Output: Hi_new
        CALL ice_thickness(Hi, D_2D, MB_surface + MB_basal, Hs, U_base, V_base, mask, Hi_new, MB_applied_per_dt, &
                           domain_leaving_flux_per_dt, mass_conservation, fluxes_per_dt)
      END IF
      
      dHi_dt = (Hi_new - Hi) / T_ANT%dt
      
    ! IsoIce and MeanIso (Added BdB 11/2009)
      CALL oxygen_isotope_icesheet(Ti(:,:,1), Hi, Hi_new, Hs, calve_mask, IsoIce, MeanIso, VolIso, fluxes_per_dt)
                                   
    ! HI UPDATE:
      Hi = Hi_new
 
    ! HS
      ! Update ice surface height using new sea level:
      ! Output: Hs
      CALL surface_height(deltaS_SLE, Hi, Hb, Hs)

    ! MASK
      ! Output: mask; Optional output: calve_mask   
      CALL determine_mask(Hi, Hb, deltaS_SLE, mask, calve_mask)
      
      !CvC make mask_binary
      ! mask_binary = mask
      mask_binary = 1._dp
      WHERE(mask == 1) 
        mask_binary = 0
      END WHERE
      WHERE(mask == 2) 
        mask_binary = 0
      END WHERE
      ! WHERE(mask_binary /= 0)
        ! mask_binary = 1
      ! END WHERE
    
    ! Adaption in case of no shelf
      ! In/Output: Hi, Hs, dHi_dt, Us, Vs
      IF(C_ANT%no_shelf)  CALL adapt_Hi_and_Hs_for_no_shelf(mask, deltaS_SLE, Hi, Hs, dHi_dt, Us, Vs)

    ! TIME UPDATE
      ! The time is updated here, so with the new Hi the new Hb can be calculated,
      ! If we calculate for the new time step the new sea level, we can calculate with
      ! the new Hi a consistent new Hb. Then with the new sea level, the new Hi, and 
      ! the new Hb, we can calculate the new Hs and the new mask. The result is that
      ! we have before writing a new consistent set of Hi, Hb, Hs and mask, being
      ! consistent with the buoyancy criterium as well. The consistent Hi-Hb-Hs is 
      ! also important for the linear back scaling of vertical profiles. For all that
      ! reasons the time update is done here.
      T_ANT%time = T_ANT%time + T_ANT%dt
      T_ANT%year = T_ANT%year + T_ANT%dt
      IF ( (T_ANT%year + 0.5_dp * T_ANT%dt) >= 1._dp) T_ANT%year = 0._dp  ! half a time step is added for rounding numbers

    ! Calculate the sea level equivalent total ice volume for grounded ice used for the Isotope calculation
    SlevIso = -ice_density * VolIso / (seawater_density * Ocean_Area_Earth)

    IF (C%use_selen) THEN
      ! The total contribution of ice volume on land to sea level, no bedrock correction since this is included in Selen
      Slev_change = (slev_ref - (ice_density / seawater_density) * SUM(Hi, MASK = (mask /= C%type_shelf))) * C%dx_ant * C%dy_ant / Ocean_Area_Earth
    ELSE
      ! The total contribution of the Antarctic ice sheet to sea level, volume above flotation, 
      ! corrected for bedrock below sea level and changes in bedrock elevation for ocean grid points:
      Slev_change = (- SUM((ice_density / seawater_density) * Hi, MASK = (mask /= C%type_shelf)) & 
                     - SUM(min(0._dp,Hb),                         MASK = (mask /= C%type_shelf)) &
                     + SUM(min(0._dp,Hb)) + vaf_ref) * C%dx_ant * C%dy_ant / Ocean_Area_Earth

    END IF

    ! WRITING

      ! Ice loading on land for output in help fields
      Hi_load = 0._dp
      WHERE (mask /= C%type_shelf) Hi_load = Hi - C%Hi_min
     
      ! Write the total changes in mass over the ice sheet every 100 years
      IF ( (T_ANT%time + 0.01_dp) >= (Model_Time + C%dt_climate_record) ) & 
        CALL write_means_and_mb_icesheet(deltaS_SLE(1,1), temperature_offset, MeanIso, Mean_Tsurf, Slev_change, calve_mask, & 
                                         Hi, dHi_dt, MB_surface, MB_basal, Accumulation, Ablation, Refreezing, fluxes_per_dt, icedis_field)

      IF(C%enable_recording_file) THEN
       ! Output: do_write_recording_output; In/Output: time_since_last_written_record_file
       CALL determine_if_action_is_required(C%dt_recording, time_since_last_written_record_file, do_write_recording_output)
       ! Writting during the last time step:
       IF(T_ANT%time + T_ANT%dt >= C%end_time_of_run) do_write_recording_output = .TRUE.
      END IF

      ! Write the time recorded variables to a netcdf file:
      IF(do_write_recording_output) THEN
       ! Look at timestep size:
       ! Output: critical_time_step
       CALL check_timestep_size(D_2D, Us, Vs, U, V, mask, critical_time_step)

!       WRITE(0, '(A,I6,A)') 'MESSAGE ANT: Writing data to time recording file (recnr=',recording_recordnr,').'

       ! Input to write data points as function of time (BdB 11/2009
       CALL write_recording_fields_file(recording_recordnr        , &
                                        T_ANT%time                , &
                                        critical_time_step        , &
                                        MB_surface                , &
                                        MB_basal                  , &
                                        MB_applied_per_dt         , &
                                        Hi                        , &
                                        domain_leaving_flux_per_dt, &
                                        mass_conservation         , &
                                        mask                      , &
                                        deltaS_SLE(1,1)           , &
                                        temperature_offset)
                                        
       recording_recordnr = recording_recordnr + 1

      END IF ! end of do_write_recording_output

      ! BdB: adjust timing of writing restart file, only the first and last
      IF(T_ANT%time-C%dt == C%start_time_of_run) THEN 
        do_write_restart_file = .TRUE.
      ELSE IF(T_ANT%time == C%end_time_of_run) THEN
        do_write_restart_file = .TRUE.
      ELSE
        do_write_restart_file = .FALSE.
      END IF

      ! Write data to the restart netCDF file:
      IF(do_write_restart_file) THEN

       IF (C%enable_restart_file) THEN
        WRITE(0, '(A,I6,A,F12.3)') 'MESSAGE ANT: Writing data to restart file (recnr=',restart_recordnr,'), time = ',T_ANT%time

        ! Output: -
        CALL write_restart_file(C%NX_ant, C%NY_ant, restart_recordnr, Hi, Hb, Hs, U_base, V_base, bottom_melt, Ti)
       END IF

       IF(C%enable_help_fields_file) THEN

        ! CvC read initial ice above floatation thickness
        open(16,file="/home/username/HetGroteKoppelScript/ANICE/input_fields/antarctica/Hi_AF_ref.dat",status="old")
         do i = 1,141
          read(16,*) (Hi_AF_ref(i,j), j = 1,141)
         enddo
        close(16)
        
		! CvC read initial grounded ice thickness
        open(17,file="/home/username/HetGroteKoppelScript/ANICE/input_fields/antarctica/Initial_ice_load.dat",status="old")
         do i = 1,141
          read(17,*) (Initial_ice_load(i,j), j = 1,141)
         enddo
        close(17)


        total_load_ant = 0._dp
        Hi_AF = 0._dp

        WHERE (Hi_load > 0 .OR. Initial_ice_load > 0)
          ocean_depth_spatial = 0._dp
          ocean_depth_spatial = -Hb + deltaS_SLE(1,1)

      
          WHERE (ocean_depth_spatial < 0)
            ocean_depth_spatial = 0
          END WHERE
          
          Hi_AF = Hi_load - seawater_density / ice_density * ocean_depth_spatial
          total_load_ant = Hi_AF - Hi_AF_ref
        END WHERE
        
        print *, 'The change in geoid is ', deltaS_SLE(1,1)
       
	   ! CvC write grounded ice above floatation minus initial grounded ice above floatation
        open(15,file="/home/username/HetGroteKoppelScript/ANICE/results/total_load_ant.dat",status="replace")
           do i = 1,141
             write(15,150) (total_load_ant(i,j), j = 1,141)
           enddo
        close(15)	 
150     format(141(e19.8))


        ! Making the sliding coverage of those points being in the domain: Ti(C%NZ) > Ti_pmp - melt_range:
        sliding_coverage = 0._dp
        IF (C_ANT%choice_sliding_method == 1) THEN
          WHERE(Ti(:,:,C%NZ) >= Ti_pmp(:,:,C%NZ) - C%melt_range) sliding_coverage = 1._dp
        ELSE IF (C_ANT%choice_sliding_method == 2) THEN
          ! Making the sliding coverage where basal velocities exceed those of SIA:
          WHERE ( SQRT(U(:,:,C%NZ)**2 + V(:,:,C%NZ)**2) >= SQRT( (Us-U(:,:,C%NZ))**2 + (Vs-V(:,:,C%NZ))**2) ) sliding_coverage = 1._dp
          WHERE ( mask == C%type_shelf .OR. Hi == C%Hi_min) sliding_coverage = 0._dp
        END IF
        

        ! Writing the help_fields_file:
        ! Output: -
        CALL write_help_fields_file(restart_recordnr     , &
                                    MB_surface           , &
                                    MB_basal             , &
                                    mask_binary           , & !CvC
                                    calve_mask           , &
                                    dHs_dx               , &
                                    dHs_dy               , & 
                                    D_2D                 , &
                                    A_flow_mean          , &
                       !  fluxes_per_dt(:,:,5) / T_ANT%dt , & ! dummy1_2D: advective fluxes per year
                                    T2m_mon              , &
                            MB_applied_per_dt / T_ANT%dt , & ! dummy2_2D: applied mass balance per year MB_applied_per_dt / T_ANT%dt
                                    SQRT(Us**2 + Vs**2)  , & ! dummy3_2D:
                                    Hi_load              , & ! dummy4_2D: Hi_load
                                    icedis_field         , & ! dummy5_2D: Ts relative to PD 
                                    !transmis             , & ! dummy6_2D: 
                                    total_load_ant          , & ! dummy6_2D: Difference between Hi_load and initial Hi load
                                    D_uv_3D              , &
                                    A_flow               , &
                                    U                    , &
                                    V                    , &
                                    W                    , &
                                    Cpi                  , &
                                    Ki                   , &
                                    U                    , & ! U      
                                    V                    , & ! V       
                                    W                    , & ! W      
                                    dzeta%t              , & ! dzeta%t
                                    dzeta%x              , & ! dzeta%x
                                    dzeta%y)
       END IF



      restart_recordnr = restart_recordnr + 1
      END IF ! End do_write_restart_file

!!-- ADJUST THE TIME STEP WHEN NEEDED AT THE END OF THE LOOP TO DETERMINE THE NEXT TIME STEP
       IF ( C%dynamic_time_step_adaption .AND. T_ANT%year == 0._dp .AND. T_ANT%time /= C%start_time_of_run) THEN
        ! TAKE CARE: changing the time step T_ANT%dt is only possible if the complete code
        ! is consistent. Changing any parameter in the C% struct violates the icedyn
        ! policy, but: THIS IS THE SINGLE EXCEPTION, namely for reasons dealing with
        ! the correct adaption of mass balances and time derivatives. Be aware that
        ! other time derivatives are not adapted (see C%b_t and C%s_t which are 
        ! related to C%dt_thermo).
        ! Only adapt when a year is completed, so the Mass balance routine can be determined 
        ! for the whole year, without adjustable time step
        ! Output: critical_time_step
        CALL check_timestep_size(D_2D, Us, Vs, U, V, mask, critical_time_step)

        ! If the critical time step is below a year, the time step is adjusted to a fixed part of the year
        IF (critical_time_step * C%time_step_scaling <= C%dt ) THEN 
          T_ANT%dt = MAX( 1._dp / C%dt_max, 1._dp / DBLE(1 + INT(1._dp / (critical_time_step * C%time_step_scaling))) )
        ! If the critical time step is larger than a year, the model time step is choosen as a whole year,
        ! with a maximum value of C%dt_max
        ELSE 
          T_ANT%dt = MIN(C%dt_max, DBLE(INT(critical_time_step * C%time_step_scaling)))
        END IF 

!!!         ! As a test only adjust time step to larger than 1 year values
!!!         T_ANT%dt = MAX(1._dp, MIN(C%dt_max, DBLE(INT(critical_time_step * C%time_step_scaling))))
          
         ! adjust the time step when time is larger than dt_climate_record, 
         ! which can only occure when the time step is larger than one year
         IF ( (T_ANT%time - Model_Time + T_ANT%dt) > C%dt_climate_record .AND. (T_ANT%time - Model_Time + 0.01_dp) < C%dt_climate_record ) &
           T_ANT%dt = Model_Time + C%dt_climate_record - T_ANT%time

       END IF


    END DO ! End time loop
!

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!              FILL INITIAL FIELDS WITH NEW DATA              !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Deallocate the memory first.
    DEALLOCATE(Hi_mem       )
    DEALLOCATE(Hb_mem       )
    DEALLOCATE(Hs_mem       )
    DEALLOCATE(dHi_dt_mem   )
    DEALLOCATE(dHb_dt_mem   )
    DEALLOCATE(U_ssa_mem    )
    DEALLOCATE(V_ssa_mem    )
    DEALLOCATE(bmelt_mem    )
    DEALLOCATE(Ti_mem       )
    DEALLOCATE(mask_mem     )
    DEALLOCATE(cmask_mem    )
    DEALLOCATE(bmask_mem    )
    DEALLOCATE(T2m_mem      )
        
    ! Allocate the memory for the next time step
    ALLOCATE(Hi_mem           (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Hs_mem           (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Hb_mem           (C%NX_ant, C%NY_ant       ))
    ALLOCATE(dHi_dt_mem       (C%NX_ant, C%NY_ant       ))
    ALLOCATE(dHb_dt_mem       (C%NX_ant, C%NY_ant       ))
    ALLOCATE(U_ssa_mem        (C%NX_ant, C%NY_ant       ))
    ALLOCATE(V_ssa_mem        (C%NX_ant, C%NY_ant       ))  
    ALLOCATE(bmelt_mem        (C%NX_ant, C%NY_ant       ))
    ALLOCATE(Ti_mem           (C%NX_ant, C%NY_ant, C%NZ ))
    ALLOCATE(mask_mem         (C%NX_ant, C%NY_ant       ))
    ALLOCATE(cmask_mem        (C%NX_ant, C%NY_ant       ))
    ALLOCATE(bmask_mem        (C%NX_ant, C%NY_ant       ))
    ALLOCATE(T2m_mem          (C%NX_ant, C%NY_ant, 12   ))  
    
    ! Fill memory arrays with newly modelled data:
    Hi_mem      = Hi
    Hb_mem      = Hb
    Hs_mem      = Hs
    dHi_dt_mem  = dHi_dt
    dHb_dt_mem  = dHb_dt
    U_ssa_mem   = U_base
    V_ssa_mem   = V_base
    bmelt_mem   = bottom_melt
    Ti_mem      = Ti
    mask_mem    = mask
    cmask_mem   = calve_mask
    bmask_mem   = mask_binary
    T2m_mem     = T2m_mon

    ! Set loading of ice thickness on land as output for the SELEN model !CvC commented here
    jj = 0
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      jj = jj + 1
      IF (mask(i,j) /= C%type_shelf) THEN 
        Load_Hi_ant(jj) = Hi(i,j) - C%Hi_min
      ELSE
        Load_Hi_ant(jj) = 0._dp
      END IF
    END DO
    END DO

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!      MODELLED DATA IS DEALLOCATED FOR THE NEXT TIME STEP    !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
    DEALLOCATE(dzeta%t)
    DEALLOCATE(dzeta%x)
    DEALLOCATE(dzeta%y)
    DEALLOCATE(dzeta%z)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!  CLOSE NETCDF FILES AND WRITE STATEMENS AT FINAL TIME STEP  !!!!
!!!!              deallocation every model time step             !!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  IF (T_ANT%time == C%end_time_of_run) THEN
  
    ! Finish and closing writing output:
!    IF(C%enable_restart_file)     CALL close_netcdf_file(restart_file)
    IF(C%enable_help_fields_file) CALL close_help_fields_file()
    IF(C%enable_recording_file)   CALL close_recording_file()
    IF(C%enable_restart_file)     CALL close_restart_file()

    ! Close the ascii output in time_recording_module
    CLOSE (4001)
    CLOSE (4011)
    CLOSE (27111992)
    CLOSE (27111993)
	! CLOSE (27111994)


    WRITE(UNIT=*, FMT='(A)') ' Program grice_ant has finished!'

  END IF ! end of last time step
  
  END SUBROUTINE main_ant_grice_model



  SUBROUTINE temperature_parameters(Hi, Ti, mask, Ti_pmp, A_flow, A_flow_mean, Cpi, Ki)
    ! In this subroutine the three-dimensonal parameters depending on the temperature are calcultated.
    
    USE parameters_main_module, ONLY: T0, CC, SMT, vertical_average 
    
    ! Input variables:
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ), INTENT(IN)  :: Hi          ! The ice thickness [m]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ), INTENT(IN)  :: Ti          ! Ice temperature [K]
    INTEGER,  DIMENSION(C%NX_ant, C%NY_ant      ), INTENT(IN)  :: mask        ! mask
    
    ! Output variables:
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ), INTENT(OUT) :: Ti_pmp      ! The pressure melting point temperature [K]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ), INTENT(OUT) :: A_flow      ! Flow parameter [Pa^-3 y^-1]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant      ), INTENT(OUT) :: A_flow_mean ! Vertically mean of the flow parameter [Pa^-3 y^-1]
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ), INTENT(OUT) :: Cpi         ! Specific heat capacity of ice [J kg^-1 K^-1].
    REAL(dp), DIMENSION(C%NX_ant, C%NY_ant, C%NZ), INTENT(OUT) :: Ki          ! Conductivity of ice [J m^-1 K^-1 yr^-1].

    ! Used for calculation of flow parameter only:
    REAL(dp), PARAMETER         :: A_low_temp  = 1.14E-05_dp   ! [Pa^-3 yr^-1] The constant a in the Arrhenius relationship
    REAL(dp), PARAMETER         :: A_high_temp = 5.47E+10_dp   ! [Pa^-3 yr^-1] The constant a in the Arrhenius relationship
    REAL(dp), PARAMETER         :: Q_low_temp  = 6.0E+04_dp    ! [J mol^-1] Activation energy for creep in the Arrhenius relationship
    REAL(dp), PARAMETER         :: Q_high_temp = 13.9E+04_dp   ! [J mol^-1] Activation energy for creep in the Arrhenius relationship
    REAL(dp), PARAMETER         :: R_gas       = 8.314_dp      ! Gas constant [J mol^-1 K^-1] 

    ! Local variables:
    INTEGER                                          :: i, j, k
    REAL(dp)                                         :: Ti_mean     ! Mean ice temperature at the shelf [K]

    ! All variables are initialized at zero, Ti_pmp is caclculated everywhere
    A_flow      = 0._dp
    A_flow_mean = 0._dp
    Cpi         = 0._dp
    Ki          = 0._dp
    
    DO j = 1, C%NY_ant
    DO i = 1, C%NX_ant
      ! Calculate the pressure melting point temperature (= the maximum temperature) for each depth (see equation (11.2)):
      Ti_pmp(i,j,:) = T0 - CC * Hi(i,j) * C%zeta(:)
      
      IF(mask(i,j) /= C%type_shelf) THEN
       DO k = 1, C%NZ

         ! Calculation of the flow parameter at the sheet and groundline as a function of the ice temperature 
         ! the Arrhenius relationship (see equation (11.10), Huybrechts (4.6)):
         IF(Ti(i,j,k) < 263.15_dp) THEN
          A_flow(i,j,k) = A_low_temp  * EXP(-Q_low_temp  / (R_gas * Ti(i,j,k)))  
         ELSE
          A_flow(i,j,k) = A_high_temp * EXP(-Q_high_temp / (R_gas * Ti(i,j,k)))  
         END IF
         
         ! Calculation of the parameterization of the specific heat capacity of ice, based on Pounder (1965):
         !Cpi(i,j,k) = 2098.1_dp + 6.97928_dp * (Ti(i,j,k) - T0)  ! See equation (11.9)
         Cpi(i,j,k) = 2115.3_dp + 7.79293_dp * (Ti(i,j,k) - T0)  ! See equation (11.9)
         
         ! Calculation of the parameterization of the thermal conductivity of ice, based on Ritz (1987):
         Ki(i,j,k)  = 3.101E+08_dp * EXP(-0.0057_dp * Ti(i,j,k)) ! See equation (11.5), Huybrechts (4.40)
       END DO

       ! Calculation of the vertical average flow parameter at the sheet and groundline
       A_flow_mean(i,j) = vertical_average(A_flow(i,j,:))
      ELSE
       ! Calculation of the flow parameter at the shelf as a function of the ice temperature 
       ! the Arrhenius relationship (see equation (11.10), Huybrechts (4.6)):
       Ti_mean = (Ti(i,j,1) + SMT) / 2._dp
       IF(Ti_mean < 263.15_dp) THEN
        A_flow_mean(i,j) = A_low_temp  * EXP(-Q_low_temp  / (R_gas * Ti_mean))  
       ELSE
        A_flow_mean(i,j) = A_high_temp * EXP(-Q_high_temp / (R_gas * Ti_mean))  
       END IF
      END IF  
    END DO
    END DO
  END SUBROUTINE temperature_parameters




END MODULE ant_grice_module
