MODULE read_geoid_module


USE configuration_main_module, ONLY: dp, C
USE netcdf, ONLY: nf90_max_var_dims, nf90_create, nf90_close, nf90_clobber, nf90_share, nf90_unlimited , &
nf90_enddef, nf90_put_var, nf90_sync, nf90_def_var, nf90_int, nf90_put_att, nf90_def_dim, &
nf90_open, nf90_write, nf90_inq_dimid, nf90_inquire_dimension, nf90_inquire, nf90_double, &
nf90_inq_varid, nf90_inquire_variable, nf90_get_var, nf90_noerr, nf90_strerror


IMPLICIT NONE

REAL(dp), DIMENSION(:,:  ), ALLOCATABLE, SAVE :: geoid_ref
CHARACTER(LEN=*), PARAMETER :: name_geoid                = 'dummy5_2D'
CHARACTER(LEN=*), PARAMETER :: name_x                     = 'X'
CHARACTER(LEN=*), PARAMETER :: name_y                     = 'Y'
CHARACTER(LEN=*), PARAMETER :: name_t                     = 't'


TYPE netcdf_file_type_geoid
    PRIVATE
    ! id for netCDF file:
    INTEGER :: ncid
    ! id's for variables:
    INTEGER :: var_x
    INTEGER :: var_y
    INTEGER :: var_t
    INTEGER :: var_geoid
END TYPE netcdf_file_type_geoid



CONTAINS

SUBROUTINE inquire_geoid(reference)
    ! Check if the right dimensions are present in the reference file, and if the variables have the right
    ! dimension

    ! Input/output variable:
    TYPE(netcdf_file_type_geoid), INTENT(INOUT) :: reference

    ! Local variables:
    INTEGER                               :: dim_x
    INTEGER                               :: dim_y
    INTEGER                               :: dim_t

    ! Inquire dimensions id's. Check that all required dimensions exist and their lengts are correct:
        CALL inquire_dim(reference, name_x, C%NX_ant,           dim_x)
        CALL inquire_dim(reference, name_y, C%NY_ant,           dim_y)
        CALL inquire_dim(reference, name_t, nf90_unlimited, dim_t)


            ! Inquire variable id's. Make sure that each variable has the correct dimensions:
    CALL inquire_double_var(reference, name_geoid,     (/ dim_x, dim_y, dim_t /), reference%var_geoid)

END SUBROUTINE inquire_geoid



SUBROUTINE load_geoid(reference,geoid)
        ! Input variables:
        TYPE(netcdf_file_type_geoid),     INTENT(IN)  :: reference ! The file from which to read.

        ! Output variables:
        REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, INTENT(OUT) :: geoid
        ALLOCATE(geoid (C%NX_ant,C%NY_ant,121)   )


        ! Read the variables:

        CALL handle_error(nf90_get_var(reference%ncid, reference%var_geoid,  geoid))
!start=(/1, 1, 1/)

END SUBROUTINE load_geoid


SUBROUTINE get_geoid(geoid,record_nr,geoid_out)
        REAL(dp), DIMENSION(:,:,:), INTENT(IN) :: geoid
        INTEGER,                    INTENT(IN) :: record_nr

        REAL(dp), DIMENSION(:,:), ALLOCATABLE, INTENT(OUT) :: geoid_out

        geoid_out = geoid(:,:,record_nr)

END SUBROUTINE


SUBROUTINE open_netcdf_file(filename, netcdf)
    IMPLICIT NONE

    ! Input variable:
    CHARACTER(LEN=*), INTENT(IN)        :: filename

    ! Output variable:
    TYPE(netcdf_file_type_geoid), INTENT(OUT) :: netcdf

    ! Open netCDF file:
    CALL handle_error(nf90_open(filename, IOR(nf90_write,nf90_share), netcdf%ncid))

END SUBROUTINE open_netcdf_file



SUBROUTINE handle_error(stat, message)
    USE netcdf, ONLY: nf90_noerr, nf90_strerror
    IMPLICIT NONE

    ! Input variables:
    INTEGER,                    INTENT(IN) :: stat
    CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: message

    IF(stat /= nf90_noerr) THEN
        IF(PRESENT(message)) THEN
        WRITE(0,'(A,A,A,A)') 'ERROR: netCDF failed (in restart_file_ant_module) because: ', TRIM(nf90_strerror(stat)), ' concerning: ', message
        ELSE
        WRITE(0,'(A,A)')     'ERROR: netCDF failed (in restart_file_ant_module) because: ', TRIM(nf90_strerror(stat))
        END IF
        STOP
    END IF
END SUBROUTINE handle_error




SUBROUTINE inquire_double_var(netcdf, var_name, id_dims, id_var)
! Inquire the id of a variable and check that the dimensions of the variable match the dimensions given by the user and
! that the variable is of type nf90_DOUBLE.
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type_geoid), INTENT(IN)    :: netcdf
    CHARACTER(LEN=*),           INTENT(IN)    :: var_name
    INTEGER, DIMENSION(:),      INTENT(IN)    :: id_dims

    ! Output variables:
    INTEGER,                INTENT(OUT)   :: id_var

    ! Local variables:
    INTEGER                               :: xtype, ndims
    INTEGER, DIMENSION(nf90_max_var_dims) :: actual_id_dims

    CALL handle_error(nf90_inq_varid(netcdf%ncid, var_name, id_var))
    CALL handle_error(nf90_inquire_variable(netcdf%ncid, id_var, xtype=xtype,ndims=ndims,dimids=actual_id_dims))
    IF(xtype /= nf90_double) THEN
        WRITE(0,'(3A)') 'ERROR: Actual type of variable "',var_name,'" is not nf90_DOUBLE.'
        STOP
    END IF
    IF(ndims /= SIZE(id_dims)) THEN
        WRITE(0,'(A,I5,3A,I5,A)') 'ERROR: Actual number of dimensions(', &
    ndims,') of variable "',var_name,'": does not match required number of dimensions (',SIZE(id_dims),').'
        STOP
    END IF
    IF(ANY(actual_id_dims(1:ndims) /= id_dims)) THEN
        WRITE(0,'(3A)') 'ERROR: Actual dimensions of variable "',var_name,'" does not match required dimensions.'
        STOP
    END IF
END SUBROUTINE inquire_double_var


SUBROUTINE inquire_dim(netcdf, dim_name, length, id_dim)
! Inquire the id of a dimension and check that the length matches the length given by the user.
    IMPLICIT NONE

    ! Input variables:
    TYPE(netcdf_file_type_geoid), INTENT(IN)  :: netcdf
    CHARACTER(LEN=*),           INTENT(IN)  :: dim_name
    INTEGER,                    INTENT(IN)  :: length

    ! Output variables:
    INTEGER,                    INTENT(OUT) :: id_dim

    ! Local variables:
    INTEGER                             :: actual_length
    INTEGER                             :: dim_unlimited

    CALL handle_error(nf90_inq_dimid(netcdf%ncid,dim_name,id_dim))
    IF(length /= nf90_unlimited) THEN
        CALL handle_error(nf90_inquire_dimension(netcdf%ncid, id_dim, len=actual_length))
        IF(length /= actual_length) THEN
            WRITE(0,'(A,I5,3A,I5,A)') 'ERROR: Actual length (',actual_length, &
            ' of dimension "',dim_name,'" does not match required length (',length,').'
        STOP
        END IF
    ELSE
        CALL handle_error(nf90_INQUIRE(netcdf%ncid, unlimitedDimId=dim_unlimited))
        IF(dim_unlimited /= id_dim) THEN
            WRITE(0,'(3A)') 'ERROR: Dimension "',dim_name,'" is not unlimited as required.'
        STOP
        END IF
    END IF
END SUBROUTINE inquire_dim






END MODULE read_geoid_module
