PROGRAM inverse_coupling_program
!   Main program to couple multiple ice-sheet models with the Bintanja/Van de Wal Inverse routine (Nature 05,08)
!   Revised and updated by Bas de Boer a2009/2010

!   Optional are 4 ice-sheets:
!   - Eurasia       (in src_eas)
!   - North America (in src_nam)
!   - Antarctica    (in src_ant)
!   - Greenland     (in src_grl) 

    USE configuration_main_module, ONLY: dp, C, read_main_config_file, initialize_main_constants,                   &
                                         finalize_constants, Model_Time, Called_Selen, Insolation_Adjusted, Ocean_Area_Earth, Ocean_Depth_Earth

    USE parameters_main_module, ONLY: gamma_dw, pdmiso_ant, pdmiso_grl, hsliso_ant_PD, hsliso_grl_PD, ESOA, Docean

    USE forcing_main_module, ONLY: read_forcing_main_records, benthic_d18O_forcing, sealevel_temp_forcing, &
                                   inverse_model_forcing, forcing_main

    USE insolation_forcing_module, ONLY: initialize_insolation_globe, read_insolation_forcing_globe,       &
                                         insolation_data

    USE eas_grice_module, ONLY: main_eas_grice_model
    USE nam_grice_module, ONLY: main_nam_grice_model
    USE ant_grice_module, ONLY: main_ant_grice_model
    USE grl_grice_module, ONLY: main_grl_grice_model

    USE selen_main_module, ONLY: main_ice2selen    

!DoE 22/06/2017
    USE read_geoid_module,      ONLY: inquire_geoid,load_geoid,get_geoid,netcdf_file_type_geoid,open_netcdf_file



IMPLICIT NONE


    ! Local variables:
    INTEGER     :: j,i
    REAL(dp)     :: geoid_divider
    INTEGER     :: time_loop_nr,time_loop_end   ! time-loop counter, total number of 100-year time-steps
    INTEGER     :: record_nr                    ! location of time-point in the climate record
    
    REAL(dp)    :: Delta_Temp                   ! relative to PD temperature (in Kelvin, mean over 40-80N)
    REAL(dp)    :: Delta_Slev                   ! relative to PD global mean (eustatic) sea level (in meters)
    REAL(dp)    :: Tdw_change                   ! relative to PD deep water temperature
    REAL(dp)    :: Dtemp_inverse                ! relative to PD temperature from the inverse routine
    REAL(dp)    :: Temp_areaNH                  ! additional temperature offset due to NH ice growth 

    REAL(dp)    :: time_mod                     ! time of the model, saved in Model_Time (in years before present)
    REAL(dp)    :: scaling_factor               ! scaling factor for winds, either benthic d18O or temperature

    REAL(dp)    :: d18O_observed                ! benthic d18O at the Climate Record time point
    REAL(dp)    :: d18O_t_plus_one              ! benthic d18O at the next time step (for inverse routine)
    
    ! Local variables of the ice sheets
    REAL(dp)    :: hsl_eas                      ! ice volume for Eurasia (in meters sea level equivalent, mseq)
    REAL(dp)    :: hsl_nam                      ! ice volume for North America (mseq)
    REAL(dp)    :: hsl_ant                      ! ice volume for Antarctica (mseq)
    REAL(dp)    :: hsl_grl                      ! ice volume for Greenland (mseq)

    REAL(dp)    :: area_eas                     ! area of land ice of Eurasia (m2)
    REAL(dp)    :: area_nam                     ! area of land ice of North America (m2)

    REAL(dp)    :: meaniso_eas                  ! ice-sheet mean d18O for Eurasia (VPDB)
    REAL(dp)    :: meaniso_nam                  ! ice-sheet mean d18O for North America (VPDB)
    REAL(dp)    :: meaniso_ant                  ! ice-sheet mean d18O for Antactica (VPDB)
    REAL(dp)    :: meaniso_grl                  ! ice-sheet mean d18O for Greenland (VPDB)

    REAL(dp)    :: hsliso_eas                   ! ice-sheet d18O volume for Eurasia (mseq)
    REAL(dp)    :: hsliso_nam                   ! ice-sheet d18O volume for North America (mseq)
    REAL(dp)    :: hsliso_ant                   ! ice-sheet d18O volume for Antactica (mseq)
    REAL(dp)    :: hsliso_grl                   ! ice-sheet d18O volume for Greenland (mseq)

    ! Lists of the ice-sheet variables
    REAL(dp), DIMENSION(4)  :: Ice_sealevel     ! Ice volumes of the four ice sheets (mseq)
    REAL(dp), DIMENSION(4)  :: Ice_meaniso      ! mean d18O of the four ice sheets (VPDB)
    REAL(dp), DIMENSION(4)  :: Ice_voliso       ! Ice volumes used for d18O calculations (mseq)

    REAL(dp), DIMENSION(9)  :: Contrib          ! list of contributions (defined below)
    REAL(dp)                :: d18O_deviation   ! Difference between modelled and observed d18O
    REAL(dp)                :: RMS_error        ! Average difference over all time steps

    ! Ice loading passed from ANICE to SELEN
    REAL(dp), DIMENSION(  :), ALLOCATABLE :: Load_Hi_nam
    REAL(dp), DIMENSION(  :), ALLOCATABLE :: Load_Hi_eas
    REAL(dp), DIMENSION(  :), ALLOCATABLE :: Load_Hi_ant
    REAL(dp), DIMENSION(  :), ALLOCATABLE :: Load_Hi_grl
    
    ! Bedrock change and regional sea level passed from SELEN to ANICE
    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: dHb_dt_eas
    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: deltaS_eas

    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: dHb_dt_nam
    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: deltaS_nam

    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: dHb_dt_ant
    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: deltaS_ant

    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: dHb_dt_grl
    REAL(dp), DIMENSION(:,:), ALLOCATABLE :: deltaS_grl

    ! Area and depth of the worlds oceans from SELEN:
    REAL(dp)                              :: ocean_area
    REAL(dp)                              :: ocean_depth    

    ! Correction for global sea level
    REAL(dp), PARAMETER     :: perc = 0.85_dp   ! Only used for NaIS+EuIS (for 'which_icesheets(1:2)' = 'NE')


!DoE 22/06/2017

    REAL(dp), DIMENSION(:,:,:), ALLOCATABLE, SAVE :: geoid_file
    REAL(dp), DIMENSION(:,:), ALLOCATABLE, SAVE :: geoid_instance
    REAL(dp), DIMENSION(:,:), ALLOCATABLE, SAVE :: geoid_instance_past
    REAL(dp), DIMENSION(:,:), ALLOCATABLE, SAVE :: geoid_instance_future

    TYPE(netcdf_file_type_geoid)     :: geoid_file_ref
    ALLOCATE(geoid_file (C%NX_ant,C%NY_ant,121))



    ! Read some of the variables in the configuration_main_module from the configuration file:
    CALL read_main_config_file()
    ! Initialization of the struct  C%: 
    CALL initialize_main_constants()
    
    ! Reading the forcing record into a struct:
    IF(C%choice_forcing == C%d18O_forcing .OR. C%choice_forcing == C%relative_slev_temp_forcing) & 
      CALL read_forcing_main_records()




    ! Reading the geoid model into a struct
    !   DoE 22/06/2017
    CALL open_netcdf_file(C%geoid_ant_filename, geoid_file_ref)

    CALL inquire_geoid(geoid_file_ref)

    CALL load_geoid(geoid_file_ref, geoid_file)



    ! Initialize insolation forcing for the Bintanja Mass balance routine
    CALL initialize_insolation_globe()




    ! Initialize variables:
    RMS_error       = 0._dp 

    Ice_sealevel(1) = 0._dp
    Ice_sealevel(2) = 0._dp
    Ice_sealevel(3) = 0._dp
    Ice_sealevel(4) = 0._dp
    
    Ice_meaniso(1)  = 0._dp
    Ice_meaniso(2)  = 0._dp
    IF (C%paleo_reference_noice) THEN
     Ice_meaniso(3)  = 0._dp
     Ice_meaniso(4)  = 0._dp
    ELSE
     Ice_meaniso(3)  = pdmiso_ant
     Ice_meaniso(4)  = pdmiso_grl
    END IF

    Ice_voliso(1)   = 0._dp
    Ice_voliso(2)   = 0._dp
    IF (C%paleo_reference_noice) THEN
     Ice_voliso(3)   = 0._dp
     Ice_voliso(4)   = 0._dp
    ELSE
     Ice_voliso(3)   = hsliso_ant_PD
     Ice_voliso(4)   = hsliso_grl_PD
    END IF
    
    Delta_Temp  = 0._dp
    Delta_Slev  = 0._dp
    Temp_areaNH = 0._dp

    ! Allocate fields from SELEN to ANICE: bedrock change, geoid and relative sea level
    ALLOCATE(dHb_dt_nam(C%NX_nam,C%NY_nam))
    ALLOCATE(deltaS_nam(C%NX_nam,C%NY_nam))

    ALLOCATE(dHb_dt_eas(C%NX_eas,C%NY_eas))
    ALLOCATE(deltaS_eas(C%NX_eas,C%NY_eas))

    ALLOCATE(dHb_dt_ant(C%NX_ant,C%NY_ant))
    ALLOCATE(deltaS_ant(C%NX_ant,C%NY_ant))

    ALLOCATE(dHb_dt_grl(C%NX_grl,C%NY_grl))
    ALLOCATE(deltaS_grl(C%NX_grl,C%NY_grl))
    
    ! Allocate field from ANICE to SELEN: ice loading on the ANICE grid points
    ALLOCATE(Load_Hi_nam(C%NX_nam * C%NY_nam))
    ALLOCATE(Load_Hi_eas(C%NX_eas * C%NY_eas))
    ALLOCATE(Load_Hi_ant(C%NX_ant * C%NY_ant))
    ALLOCATE(Load_Hi_grl(C%NX_grl * C%NY_grl))

    dHb_dt_eas   = 0._dp
    dHb_dt_nam   = 0._dp
    dHb_dt_ant   = 0._dp
    dHb_dt_grl   = 0._dp

    deltaS_eas   = 0._dp
    deltaS_nam   = 0._dp
    deltaS_ant   = 0._dp
    deltaS_grl   = 0._dp
    
    Load_Hi_nam  = 0._dp
    Load_Hi_eas  = 0._dp
    Load_Hi_ant  = 0._dp
    Load_Hi_grl  = 0._dp
!
! -- HERE READ IN BEDROCK AND SEA-SURFACE LEVEL CHANGE WHEN DOING A RESTART WITH SELEN.
!
!   Restart selen so load in change in topography and sea surface height from the previous time slice
    IF (C%restart_model .and. C%use_selen) THEN
!      North America
      IF(C%which_icesheets(1:1) == 'N') THEN 
        open (1015,file=C%initial_selen_nam_filename,status='old')
        do j = 1, C%NY_nam
        do i = 1, C%NX_nam
          read (1015,'(2F15.5)') dHb_dt_nam(i,j),deltaS_nam(i,j)
        end do
        end do
        close(1015)
      END IF
!     Eurasia
      IF(C%which_icesheets(2:2) == 'E') THEN 
        open (1016,file=C%initial_selen_eas_filename,status='old')
        do j = 1, C%NY_eas
        do i = 1, C%NX_eas
          read (1016,'(2F15.5)') dHb_dt_eas(i,j),deltaS_eas(i,j)
        end do
        end do
        close(1016)
      END IF
!     Greenland
      IF(C%which_icesheets(3:3) == 'G') THEN 
        open (1017,file=C%initial_selen_grl_filename,status='old')
        do j = 1, C%NY_grl
        do i = 1, C%NX_grl
          read (1017,'(2F15.5)') dHb_dt_grl(i,j),deltaS_grl(i,j)
        end do
        end do
        close(1017)
      END IF
!     Antarctica
      IF(C%which_icesheets(4:4) == 'A') THEN 
        open (1018,file=C%initial_selen_ant_filename,status='old')
        do j = 1, C%NY_ant
        do i = 1, C%NX_ant
          read (1018,'(2F15.5)') dHb_dt_ant(i,j),deltaS_ant(i,j)
        end do
        end do
        close(1018)
      END IF
    END IF

    ! Initialize constants
    Called_Selen        = .FALSE.
    Insolation_Adjusted = .FALSE.
    Ocean_Area_Earth    = ESOA
    Ocean_Depth_Earth   = Docean
        
    IF(C%choice_forcing == C%constant_d18O_forcing) THEN
      WRITE (0,'(A)') 'Use constant (present day) benthic d18O as forcing for the model'
    ELSE IF(C%choice_forcing == C%relative_slev_temp_forcing) THEN
      WRITE (0,'(A)') 'Use varying slev/temp as forcing for the model'
      WRITE (0,'(2A)') 'From file: ',C%forcing_filename
    ELSE IF(C%choice_forcing == C%d18O_forcing) THEN
      WRITE (0,'(A)') 'Use benthic d18O as forcing for the model'
      WRITE (0,'(2A)') 'From file: ',C%forcing_filename
    ELSE IF(C%choice_forcing == C%climate_model_forcing) THEN
      WRITE (0,'(A)') 'Use temperature and precipitation fields from a climate model as forcing for the model'
    END IF    
       

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                       START MAIN TIME LOOP                            !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    
    ! Open standard output file
    OPEN (UNIT=1111,FILE=C%main_standard_filename,STATUS='replace',ACTION='write')

    ! Open isotopes output file
    OPEN (UNIT=1112,FILE=C%main_isotope_filename,STATUS='replace',ACTION='write')
    
    ! Open SELEN output file
    IF(C%use_selen) OPEN (UNIT=1113,FILE=C%main_selen_filename,STATUS='replace',ACTION='write')

    ! Initialize the time of the model
    time_mod = C%start_time_of_run
    Model_Time = time_mod
    
    IF(C%choice_forcing == C%d18O_forcing .OR. C%choice_forcing == C%relative_slev_temp_forcing) THEN
      ! Initialize the record number in the input data
      DO i = 1, C%size_climate_record
        IF(Model_Time == forcing_main%time2(i)) record_nr = i + 1  ! add 1 since runing the models for the next dt_climate_record
      END DO
    ELSE
      ! constant forcing, record_nr starts at 1:
      record_nr = 1
    END IF

    ! Determine the end of the time-loop in steps of the climate record (default = 100 years) 
    time_loop_end = INT((C%end_time_of_run - C%start_time_of_run) / C%dt_climate_record)
    IF( (C%end_time_of_run - C%start_time_of_run) < C%dt_climate_record ) time_loop_end = 1
    
  WRITE (0,*) 'loop: ',C%start_time_of_run, C%end_time_of_run, time_loop_end







    DO time_loop_nr = 1, time_loop_end  ! Time in steps of dt_climate_record




      IF(MOD(time_loop_nr,1000) == 0) &
      WRITE(0,'(A,F6.1,A)') 'Calculation at ',100._dp * dble(time_loop_nr)/dble(time_loop_end),' %'
      
      ! Get benthic d18O value from the input (climate record or constant value)
      IF(C%choice_forcing == C%d18O_forcing .OR. C%choice_forcing == C%constant_d18O_forcing) THEN
        CALL benthic_d18O_forcing(record_nr, d18O_observed, d18O_t_plus_one)

      ! Get temperature and sealevel from record
      ELSE IF(C%choice_forcing == C%relative_slev_temp_forcing) THEN
        CALL sealevel_temp_forcing(record_nr, Delta_Slev, Delta_Temp)

      END IF


!!---- Read insolation from file
    IF(C%choice_insolation == C%varying_insolation .AND. MOD((time_loop_nr-1),10) == 0 ) THEN
      Insolation_Adjusted = .TRUE.
      CALL read_insolation_forcing_globe()
    ELSE
      Insolation_Adjusted = .FALSE.
    END IF

!!--- Run the ice-sheet models  for 'dt_climate_record' years (= 100 yr)
      ! There are thee main options for 'which_icesheets' (in total 7 including the 4 seperate models):
      ! NEGA - all ice sheets: NaIS+EuIS+GrIS+AIS
      ! NEFF - only NaIS+EuIS ice sheets (BW08 set up)
      ! FFGA - pre NH run, only including GrIS+AIS

      ! Do a spin-up of temperature, start with no varying forcing
      IF(C%do_temperature_spin_up) THEN
        IF(Model_Time < -125000._dp) THEN
         Delta_Temp = 0._dp
        END IF
         Delta_Slev = 0._dp
      END IF

      IF(C%choice_forcing == C%constant_d18O_forcing) THEN
        scaling_factor = d18O_observed
      ELSE IF(C%choice_forcing == C%relative_slev_temp_forcing) THEN
        scaling_factor = Delta_Temp
      ELSE IF(C%choice_forcing == C%d18O_forcing) THEN
        scaling_factor = d18O_observed
      ELSE IF(C%choice_forcing == C%climate_model_forcing) THEN
        scaling_factor = d18O_observed
      END IF    




      ! Run the North American Ice Sheet (NaIS)
      IF(C%which_icesheets(1:1) == 'N') THEN 
        IF(.NOT. C%use_selen) THEN
          ! bedrock response is calculated with ELRA model
          dHb_dt_nam = 0._dp
          ! use eustatic sea level
          deltaS_nam = Delta_Slev
        END IF
        
        ! As a test while usig Selen set the sea-level change in the ice-sheet model to eustatic
        IF (C%use_anice_with_eustatic) deltaS_nam = Delta_Slev
        
        WRITE (0,*) 'Inverse_coupling_program: CALL North America grice routine'         
        CALL main_nam_grice_model(scaling_factor, Delta_Temp, dHb_dt_nam, deltaS_nam, hsl_nam, meaniso_nam, hsliso_nam, area_nam, Load_Hi_nam)

        ! Every time step, store the ice-sheet variables for EAS and NAM in the list,
        ! which are output of the ANICE model runs (hsl should be negative):
        Ice_sealevel(2) = hsl_nam
        Ice_meaniso(2)  = meaniso_nam
        Ice_voliso(2)   = hsliso_nam
      END IF

      ! Run the Eurasian Ice Sheet (EuIS), output is ice-sheet sea level and d18O_mean
      IF(C%which_icesheets(2:2) == 'E') THEN 
        IF(.NOT. C%use_selen) THEN
          ! bedrock response is calculated with ELRA model
          dHb_dt_eas = 0._dp
          ! use eustatic sea level
          deltaS_eas = Delta_Slev
        END IF  

        ! As a test while usig Selen set the sea-level change in the ice-sheet model to eustatic
        IF (C%use_anice_with_eustatic) deltaS_eas = Delta_Slev

        WRITE (0,*) 'Inverse_coupling_program: CALL Eurasia grice routine'         
        CALL main_eas_grice_model(Delta_Temp, dHb_dt_eas, deltaS_eas, hsl_eas, meaniso_eas, hsliso_eas, area_eas, Load_Hi_eas)

        ! Every time step, store the ice-sheet variables for EAS and NAM in the list,
        ! which are output of the ANICE model runs (hsl should be negative):
        Ice_sealevel(1) = hsl_eas
        Ice_meaniso(1)  = meaniso_eas
        Ice_voliso(1)   = hsliso_eas
      END IF

      ! The NH area also have influence on their regional climate. The determined Delta_Temp from 
      ! the inverse routine already incoorporates this difference. Therefore this offset, which is 
      ! about 3 K for the LGM (average of PMIP2 models) is added tp the temperature forcing 
      ! for the Antarctic ice sheet.
      ! So Max value is 3 K at the LGM, for an area of 1E13 m2 of the EuIS and 1.4E13m2 for the NaIS
      IF(C%which_icesheets(1:4) == 'NEGA') &
        Temp_areaNH = 1.5_dp * MIN(2._dp, area_eas / 1.E13_dp + area_nam / 1.4E13_dp)

      ! Run the Greenland Ice Sheet (GRL)
      IF(C%which_icesheets(3:3) == 'G') THEN
        ! If SELEN is not used, set dhb_dt to zero (this is caluclated internally with the ELRA model) and 
        ! the sea level change to the eustatic calculated sea level derived from ice volume
        IF(.NOT. C%use_selen) THEN
          ! bedrock response is calculated with ELRA model
          dHb_dt_grl = 0._dp
          ! use eustatic sea level
          deltaS_grl = Delta_Slev
        END IF  

        ! As a test while usig Selen set the sea-level change in the ice-sheet model to eustatic
        IF (C%use_anice_with_eustatic) deltaS_grl = Delta_Slev

        WRITE (0,*) 'Inverse_coupling_program: CALL Greenland grice routine'
        CALL main_grl_grice_model(Delta_Temp, dHb_dt_grl, deltaS_grl, hsl_grl, meaniso_grl, hsliso_grl, Load_Hi_grl)

        ! Every time step, store the ice-sheet variables for ANT and GRL in the list,
        ! which are output of the ANICE model runs (hsl should be negative):
        Ice_sealevel(4) = hsl_grl 
        Ice_meaniso(4)  = meaniso_grl
        Ice_voliso(4)   = hsliso_grl
      END IF





      ! Run the Antarctic Ice Sheet (ANT)
      IF(C%which_icesheets(4:4) == 'A') THEN
        IF(.NOT. C%use_selen) THEN
          ! bedrock response is calculated with ELRA model
          dHb_dt_ant = 0._dp
          ! use eustatic sea level

        !DoE 23/06/17
        !original:

        IF (.NOT. C%use_geoid_forcing) THEN
        deltaS_ant = Delta_Slev

        ELSE IF (C%use_geoid_forcing)   THEN
        !new year
        !every 1000 year the geoid gets called

            IF(MOD(time_loop_nr-1,10) == 0) THEN
                !CALL get_geoid(geoid_file,(time_loop_nr+9)/10,geoid_instance)
                CALL get_geoid(geoid_file,(time_loop_nr+9)/10,geoid_instance_past)
                IF (((time_loop_nr+9)/10)+1 < 122) THEN
                    CALL get_geoid(geoid_file,((time_loop_nr+9)/10)+1,geoid_instance_future)
                ENDIF
            END IF
            geoid_divider = mod(time_loop_nr-1,10)
            geoid_divider = geoid_divider/10
            IF (((time_loop_nr+9)/10)+1 < 122) THEN
                geoid_instance = ((1-geoid_divider)*geoid_instance_past) + (geoid_divider*geoid_instance_future)
            ELSE
                geoid_instance = geoid_instance_past
            END IF
            deltaS_ant = geoid_instance
        END IF



        END IF

        ! As a test while usig Selen set the sea-level change in the ice-sheet model to eustatic
        IF (C%use_anice_with_eustatic) deltaS_ant = Delta_Slev
        WRITE (0,*) 'Inverse_coupling_program: CALL Antarctica grice routine'

!DoE 31/05/2017

        CALL main_ant_grice_model(Delta_Temp + Temp_areaNH, dHb_dt_ant, deltaS_ant, hsl_ant, meaniso_ant, hsliso_ant, Load_Hi_ant)


        ! Every time step, store the ice-sheet variables for ANT and GRL in the list,
        ! which are output of the ANICE model runs (hsl should be negative):

        Ice_sealevel(3) = hsl_ant ! total sea level contribution in msl-eq
        Ice_meaniso(3)  = meaniso_ant
        Ice_voliso(3)   = hsliso_ant

      END IF

      ! Update the model time and climate record (output is written after 100 year)
      ! time loop of the ice-sheet models
      time_mod  = time_mod  + C%dt_climate_record
      record_nr = record_nr + 1

      ! Save the time in Model_Time, a global values saved in configuration_main_module.f90
      Model_Time = time_mod

!!--- Use SELEN to calculate the geoid: the regional sea-level field relative to the center of mass of the Earth

      ! Output: bedrock change U and geoid N(=U=S) relative to the first time step
      IF(C%use_selen .AND. (MOD((time_loop_nr-1),C%steps_per_selen) == 0 .OR. time_loop_nr == time_loop_end) ) THEN
        WRITE (0,*) 'CALL SELEN: ',Model_Time,time_loop_nr
        Called_Selen = .TRUE.

        CALL main_ice2selen(time_loop_nr,                                    &
                             Load_Hi_nam, dHb_dt_nam, deltaS_nam,            &
                             Load_Hi_eas, dHb_dt_eas, deltaS_eas,            &
                             Load_Hi_grl, dHb_dt_grl, deltaS_grl,            &
                             Load_Hi_ant, dHb_dt_ant, deltaS_ant, ocean_area, ocean_depth )
        Ocean_Area_Earth = ocean_area
        Ocean_Depth_Earth = ocean_depth
      ELSE
        Called_Selen = .FALSE.
      END IF


!!--- Use inverse routine to compute the temperature offset and contributions to d18O

      ! For d18O forcing, put sea level from the GRICE models in the variable Delta_Slev
      ! Here, Ice_sealevel is negative, including 
      IF(C%choice_forcing <= C%d18O_forcing) THEN
        IF(C%which_icesheets(1:4) == 'NEFF' ) THEN 
          Delta_Slev = SUM(Ice_sealevel(1:2)) / perc
        ELSE
          Delta_Slev = SUM(Ice_sealevel(1:4))
        END IF
      ELSE IF (C%choice_forcing == c%climate_model_forcing) THEN
        Delta_Slev = SUM(Ice_sealevel(1:4))
      END IF  

      ! Output: Dtemp_inverse, Contrib(1:9), for time = time + dt_climate_record
      CALL inverse_model_forcing(record_nr, time_loop_nr, time_loop_end, Delta_Slev, Ice_voliso,    & 
                                 Ice_meaniso, d18O_t_plus_one, Dtemp_inverse, Tdw_change, Contrib)

      ! The list Contrib(9) contains all variables related to d18O variations/contributions:
      ! Contrib(1) = d18O_modelled <-- is the modelled benthic d18O
      ! Contrib(2) = d18O_seawater          Contrib(6) = d18O_seawater_eas
      ! Contrib(3) = d18O_temp              Contrib(7) = d18O_seawater_nam
      ! Contrib(4) = d18O_seawater_relPD    Contrib(8) = d18O_seawater_ant
      ! Contrib(5) = d18O_temp_relPD        Contrib(9) = d18O_seawater_grl
      
!!--- Calculate difference between modelled and observed d18O
      IF(C%choice_forcing <= C%d18O_forcing) THEN
        d18O_deviation  = Contrib(1) - d18O_observed
        RMS_error = RMS_error + d18O_deviation**2 
      ELSE 
        d18O_observed = Contrib(1)
      END IF
   
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                       OUTPUT OF DATA IN ASCII                         !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!!      WRITE (0,'(A,F10.2)') 'Writing output data to file at time ',Model_Time
      
     IF(C%which_icesheets(3:3) == 'F' .and. C%which_icesheets(4:4) == 'F') THEN ! so for NEFF, NFFF,FEFF

       IF( time_loop_nr == 1 ) WRITE (1111,'(A)') '%% Output for Eurasian and North American Ice Sheets'    

       IF( time_loop_nr == 1 ) WRITE (1111,102) '%%','Time (ka)','Sea level','T offset',        &
            'Tdw offset','d18O obs','d18O model','d18O_sw','d18O_temp','Hsl EAS','Hsl NAM','Q 65N Jun'
        
       WRITE (1111,112) Model_Time/1000._dp, Delta_Slev, Delta_Temp, Tdw_change,                &
            d18O_observed - C%PD_benthic_isotope, Contrib(1) - C%PD_benthic_isotope,            &
            Contrib(4), Contrib(5), (Ice_sealevel(j),j=1,2), insolation_data%Jun65N

     ELSE ! for TTTT, FFTT, FFTF, FFFT

       IF( time_loop_nr == 1 ) WRITE (1111,'(A)') '%% Output for all four Ice Sheets'   

       IF( time_loop_nr == 1 ) WRITE (1111,101) '%%','Time (ka)','Sea level','T offset',        &
            'Tdw offset','d18O obs','d18O model','d18O_sw','d18O_temp',                         &
            'Hsl EAS','Hsl NAM','Hsl ANT','Hsl GRL','T off AIS','T areaNH','Q65N Jun','Ocean Area'
        
       WRITE (1111,111) Model_Time/1000._dp, Delta_Slev, Delta_Temp, Tdw_change,                &
            d18O_observed - C%PD_benthic_isotope, Contrib(1) - C%PD_benthic_isotope,            &
            (Contrib(j),j=4,5), (Ice_sealevel(j),j=1,4),                                        & 
            Delta_Temp + Temp_areaNH, Temp_areaNH, insolation_data%Jun65N, Ocean_Area_Earth*1.E-12_dp
     END IF
     ! Output for when either Greenland and Antarctic are inactive
     IF(C%which_icesheets(3:3) == 'F' .and. C%which_icesheets(4:4) == 'F') THEN ! so for NEFF, NFFF,FEFF

        IF(time_loop_nr == 1) WRITE (1112,'(A)') '%% Output for Eurasian and North American Ice Sheets'

        IF(time_loop_nr == 1) WRITE (1112,104) '%%','Time (ka)','d18O obs','d18O model',        &
            'delta_w','delta_T','d18O_sw EAS','d18O_sw NAM','d18Oice EAS','d18Oice NAM'
        
        WRITE (1112,114) Model_Time/1000._dp, d18O_observed, (Contrib(j),j=1,3),                &
                         (Contrib(j),j=6,7), (Ice_meaniso(j),j=1,2)

     ! Output for when greenland and/or antarctica are active
     ELSE

       IF(time_loop_nr == 1) WRITE (1112,'(A)') '%% Output for all four Ice Sheets'  

       IF(time_loop_nr == 1) WRITE (1112,103) '%%','Time (ka)','d18O obs','d18O model',         &
           'delta_w','delta_T','d18O_sw EAS','d18O_sw NAM','d18O_sw ANT','d18O_sw GRL',         &
           'd18Oice EAS','d18Oice NAM','d18Oice ANT','d18Oice GRL'
        
       WRITE (1112,113) Model_Time/1000._dp, d18O_observed, (Contrib(j),j=1,3),                 &
                        (Contrib(j),j=6,9), (Ice_meaniso(j),j=1,4)

      END IF
      
      CALL FLUSH(1111)
      CALL FLUSH(1112)
      
      ! Put temperature from the inverse routine in the variable Delta_Temp 
      ! In case of temp/slev forcing, this is read from a file
      IF(C%choice_forcing <= C%d18O_forcing) Delta_Temp = Dtemp_inverse




    END DO




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!                       END OF MAIN LOOP                                !!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Calculate the average difference between modelled and observed d18O
    IF(C%choice_forcing <= C%d18O_forcing) THEN
     RMS_error = SQRT( RMS_error / dble(time_loop_end) )
     WRITE (UNIT=C%stdlog, FMT='(1x,a12,f8.4)') 'RMS error of d18O = ', RMS_error
    END IF 

!!--- Formats for write-statements

      101 format (a2,a10,13a14,a10,a14)
      111 format (f12.3,3f14.3,4f14.4,6f14.3,f10.2,e14.5)
      
      102 format (a2,a10,9a14,a10)
      112 format (f12.3,3f14.3,4f14.4,2f14.3,f10.2)
      
      103 format (a2,a10,12a14)
      113 format (f12.3,8f14.4,4f14.3)
      
      104 format (a2,a10,8a14)
      114 format (f12.3,6f14.4,2f14.3)

      CALL finalize_constants()



END PROGRAM inverse_coupling_program
