MODULE insolation_forcing_module
  USE configuration_main_module, ONLY: C, dp, Model_Time
  IMPLICIT NONE
  
  ! This TYPE contains the insolation record for one time interval, which is 
  ! updated every 1000 years with read_insolation_forcing_globe
  ! initialization of the data is done before the model starts
  ! BdB (04/2010)
  
  TYPE insolation_data_type
      REAL(dp), DIMENSION(12,181)     :: Qins    ! Monthly insolation (W/m2) for all latitude
      REAL(dp), DIMENSION(181)        :: Phi     ! List of all latitudes
      REAL(dp)                        :: Jun65N  ! 65 degrees North June insolation (W/m2)
      REAL(dp)                        :: Jan80S  ! 80 degrees South January insolation (W/m2)
  END TYPE insolation_data_type
  
  ! the 'struct' insolation_data contains the data and is saved
  TYPE(insolation_data_type), SAVE         :: insolation_data

CONTAINS
  SUBROUTINE initialize_insolation_globe()
    ! Initialize the inlation data with the present day distribution
    ! only changed when using varying insolation (from the CONFIG file)
  
    ! Local variables:
    INTEGER                         :: j2,m

    REAL(dp)                        :: rlat      ! latitude
    REAL(dp)                        :: instime   ! time of insolation
    REAL(dp), DIMENSION(12)         :: Qmon      ! Monthly insolation (W/m2)
      
!!---- Read present day insolation

      ! Read insolation for all latitudes
      OPEN(UNIT=3000,FILE=C%insolation_filename)
      DO j2 = 1, 181
        READ(UNIT=3000,FMT=*) instime,rlat,(Qmon(m),m=1,12)
        insolation_data%Phi(j2) = rlat

        DO m = 1, 12
          insolation_data%Qins(m,j2) = Qmon(m)
        END DO
        
        ! Put insolation in variables, for writing (65N) and computing varying melt (80S)
        IF (j2 == 156) insolation_data%Jun65N = Qmon(6)
        IF (j2 == 11)  insolation_data%Jan80S = Qmon(1)  
      END DO

      CLOSE(UNIT=3000)
  END SUBROUTINE initialize_insolation_globe

 
  SUBROUTINE read_insolation_forcing_globe()
    ! Only calculate new insolation: Q_at_TOA every 1000 years
    ! Always use a file consisting of 6 Myr points for every 1000 years, 
    ! with 181x12 points per time step

    ! Local variables:
    INTEGER                         :: j1,j2,m

    REAL(dp)                        :: rlat      ! latitude
    REAL(dp)                        :: instime   ! time of insolation
    REAL(dp), DIMENSION(12)         :: Qmon      ! Monthly insolation (W/m2)

 
    ! Read insolation for all latitudes
    OPEN(UNIT=3000,FILE=C%insolation_filename)
    
    DO j1 = 1, 411 !6001
    DO j2 = 1, 181
      READ(UNIT=3000,FMT=*) instime,rlat,(Qmon(m),m=1,12)

      IF ( 1000._dp*instime == Model_Time ) THEN
        DO m = 1, 12
          insolation_data%Qins(m,j2) = Qmon(m)
        END DO

        ! Put insolation in variables, for writing (65N) and computing varying melt (80S)
        IF (j2 == 156) insolation_data%Jun65N = Qmon(6)
        IF (j2 == 11)  insolation_data%Jan80S = Qmon(1)  
      END IF

      ! Exit the loop after reaching the specific time-step
      !IF ( 1000._dp*instime > Model_Time ) EXIT
    END DO
    END DO

    WRITE (0,'(2(A,F14.2))') 'Use varying insolation as forcing at time =', Model_Time, ' June65N = ',insolation_data%Jun65N

    CLOSE(UNIT=3000)

  END SUBROUTINE read_insolation_forcing_globe
END MODULE insolation_forcing_module
