MODULE forcing_main_module
    USE configuration_main_module, ONLY: dp, C, Ocean_Depth_Earth
    IMPLICIT NONE
    
    ! This TYPE contains the input data: time (kyr bp) and benthic d18O value (permille)
    TYPE d18O_data_type
      REAL(dp), DIMENSION(:), ALLOCATABLE :: time2  ! time in kyrs before present (negative)
      REAL(dp), DIMENSION(:), ALLOCATABLE :: d18Ob  ! benthic d18O value (absolute) in permille
      REAL(dp), DIMENSION(:), ALLOCATABLE :: gSlev  ! Global Sea Level offset relative to PD
      REAL(dp), DIMENSION(:), ALLOCATABLE :: gTemp  ! Global Temperature offset, relative to PD
    END TYPE d18O_data_type

    ! forcing_main is the 'struct' containing the forcing records
    TYPE(d18O_data_type), SAVE            :: forcing_main

CONTAINS
  SUBROUTINE read_forcing_main_records()
    ! Reading the climate records from a file, and putting them into a 'struct' 
    ! if the file ends before the whole array is filled, the remainder of 
    ! the array is filled with PD d180 values 
    ! Two options are possible:
    ! 2 - bethic d18O data as input, used in the inverse routine which determines 
    !     Temperature. Sea level is calculated from ice volume of the ice sheets
    !
    ! 3 - Sea level and Temperature data (e.g. BW08 output data), which can be 
    !     directly applied to the ice sheets.
     
    ! Local variables:
    INTEGER :: i
    INTEGER :: ios
    INTEGER :: stat


    ! Choice of forcing is benthic d18O data, used in inverse routine   
    IF(C%choice_forcing == C%d18O_forcing) THEN
    
      ! Allocate space for the struct forcing_main%*
      ALLOCATE(forcing_main%time2(C%size_climate_record),                       &
               forcing_main%d18Ob(C%size_climate_record), STAT=stat)
      IF(stat/=0) STOP 'Could not allocate memory for ''''forcing_main%*''''!'
        
      forcing_main%d18Ob  = C%PD_benthic_isotope

      ! Read the climate records until the whole array is filled or end of file is reached:
      OPEN(UNIT=1500, FILE=C%forcing_filename)
      DO i = 1, C%size_climate_record
        READ(UNIT=1500, FMT=*, IOSTAT=ios) forcing_main%time2(i),               &
                                           forc/uing_main%d18Ob(i)
        IF(ios /= 0) EXIT
      END DO
    
      CLOSE(UNIT=1500)

    ELSE IF(C%choice_forcing == C%relative_slev_temp_forcing) THEN
      ! Allocate space for the struct forcing_main%*
      ALLOCATE(forcing_main%time2(C%size_climate_record),                       &
             forcing_main%gSlev(C%size_climate_record),                         &
             forcing_main%gTemp(C%size_climate_record), STAT=stat)
      IF(stat/=0) STOP 'Could not allocate memory for ''''forcing_main%*''''!'

      forcing_main%gSlev = 0._dp
      forcing_main%gTemp = 0._dp

      ! Read the climate records until the whole array is filled or end of file is reached:
      OPEN(UNIT=1500, FILE=C%forcing_filename)
      DO i = 1, C%size_climate_record
        READ(UNIT=1500, FMT=*, IOSTAT=ios) forcing_main%time2(i),               &
                                           forcing_main%gSlev(i),               &
                                           forcing_main%gTemp(i)
        IF(ios /= 0) EXIT
      END DO
      CLOSE(UNIT=1500)
    END IF  

  END SUBROUTINE read_forcing_main_records


  SUBROUTINE sealevel_temp_forcing(Clim_Record, sealevel, gT_offset)
      ! USED IN THE MAIN ROUTINE: inverse_coupling_program
      
      ! This routine read sealevel and temperature from the climate record, per output time step 
      ! which is set to a default value of 100 years.
      ! The sealevel is relative to present day sealevel [in meter].
      ! The global temperature signal (gT_offset) is relative to present day, in degrees Kelvin.

      ! Input variable:
      INTEGER,  INTENT(IN)  :: Clim_Record   ! record number of the data, only used when reading a record
        
      ! Output variables:
      REAL(dp), INTENT(OUT) :: sealevel      ! sealevel at time t (in meter, relative to present day sealevel)
      REAL(dp), INTENT(OUT) :: gT_offset     ! global temperature signal (in Kelvin, relative to present day)

      IF(Clim_Record < 1 .OR. Clim_Record > C%size_climate_record) THEN
        WRITE(UNIT=*,FMT=*) 'ERROR in forcing_main_module, record number: ', Clim_Record, ' not in climate records!'
        STOP
      END IF

      ! The sealevel and the global temperature signal (gT_offset) are counted for
      ! the corresponding time, by interpolation between the two nearest climate records 
      sealevel  = forcing_main%gSlev(Clim_Record)
      gT_offset = forcing_main%gTemp(Clim_Record)

  END SUBROUTINE sealevel_temp_forcing

    
  SUBROUTINE benthic_d18O_forcing(Clim_Record, d18O_time0, d18O_time1)
      ! USED IN THE MAIN ROUTINE: inverse_coupling_program

      ! Get d18O at the present and at the next time step for 
      ! usage in the inverse routine
      ! input data (default is each 100 years)
      
      ! Input variable:
      INTEGER,  INTENT(IN)  :: Clim_Record     ! record number of the data, only used when reading a record
      
      ! Output variables:
      REAL(dp), INTENT(OUT) :: d18O_time0      ! benthic d18O value at this time step
      REAL(dp), INTENT(OUT) :: d18O_time1      ! benthic d18O value at the next time step (for inverse routine)

      IF(Clim_Record < 1 .OR. Clim_Record > C%size_climate_record) THEN
        WRITE(UNIT=*,FMT=*) 'ERROR in forcing_main_module, record number: ', Clim_Record, ' not in climate records!'
        STOP
      END IF

      IF(C%choice_forcing == C%d18O_forcing) THEN
        ! Applying the variable forcing from a (historical) record:
        d18O_time0 = forcing_main%d18Ob(Clim_Record)
        ! A check is added for the very last time step when Clim_Record = C%size_climate_record.
        d18O_time1 = forcing_main%d18Ob(MIN(C%size_climate_record,Clim_Record+1))
      ELSE IF(C%choice_forcing == C%constant_d18O_forcing) THEN
        ! Applying a constant forcing, with present day is 3.228 permil (LR04)
        d18O_time0 = C%constant_forcing_d18O
        d18O_time1 = C%constant_forcing_d18O
      END IF

    END SUBROUTINE benthic_d18O_forcing    

    
    SUBROUTINE inverse_model_forcing(Clim_Record, teller, end_teller, sealevel, Ice_volice, Ice_meaniso,   &
                                     d18O_obs_next, gT_offset, gT_dw_offset, Contrib)
                                      
      ! USED for benthic d18O forcing only!
      ! input are several variables of the ice-sheets
      ! output is a temperature offset and 
      ! an array of the different contributions to d18O (Contrib(9)
      
      USE parameters_main_module, ONLY: gamma_dw, hsliso_ant_PD, hsliso_grl_PD, pdmiso_ant, pdmiso_grl

      ! Input variables:
      INTEGER,                 INTENT(IN)  :: Clim_Record     ! record number of the data, only used for gTemp as forcing
      INTEGER,                 INTENT(IN)  :: teller          ! time-loop counter
      INTEGER,                 INTENT(IN)  :: end_teller      ! time-loop end

      REAL(dp),                INTENT(IN)  :: sealevel        ! Global sealevel relative to Present Day (m)
      REAL(dp), DIMENSION(4),  INTENT(IN)  :: Ice_volice      ! Ice volumes of the four ice sheets (in mseq)
      REAL(dp), DIMENSION(4),  INTENT(IN)  :: Ice_meaniso     ! Mean d18O of the four ice sheets (VPDB)
      REAL(dp),                INTENT(IN)  :: d18O_obs_next   ! d18O from record (or constant) at time=t+1
      
      ! Output variables:
      REAL(dp),                INTENT(OUT) :: gT_offset       ! Temperature offset relative to Present Day (K)
      REAL(dp),                INTENT(OUT) :: gT_dw_offset    ! Deep-water temperature offset relative to Present Day (K)
      REAL(dp), DIMENSION(9),  INTENT(OUT) :: Contrib         ! list of contributions (defined below)

      ! Correction for isotope contribution, only when using EAS+NAM
      REAL(dp), PARAMETER                  :: lambda = 1.20_dp

      ! Save temperature for calculating mean temperatures
      REAL(dp), DIMENSION(:), ALLOCATABLE, SAVE :: DTemp_mid  ! initial temperature perturbation (gT_offset) 

      REAL(dp)    :: dTmean_climate                           ! mean temperature over 2000 years, for inverse coupling
      REAL(dp)    :: dTmean_deepwater                         ! mean temperature over 3000 years, for deep-water coupling
      INTEGER     :: i,nmid1,nmid2                            ! length of averaging period (20 or 30 time steps)

      ! d18O variables:
      REAL(dp)                       :: d18O_modelled         ! Modelled benthic d18O (VPDB)        
      REAL(dp)                       :: d18O_seawater         ! seawater d18O (delta_w) (VPDB)
      REAL(dp)                       :: d18O_temp             ! seawater d18O (delta_T) (VPDB)
      REAL(dp)                       :: d18O_change           ! change in benc d18O [model(t) - obs(t+1)]
     
      ! Relative to Present Day values:
      REAL(dp)                       :: d18O_seawater_relPD   ! d18O of seawater (VSMOW)
      REAL(dp)                       :: d18O_temp_relPD       ! d18O from temperature (VPDB)
      
      ! Individual contributions of the ice-sheets:
      REAL(dp)                       :: d18O_seawater_eas     ! Relative to PD d18O_seawater from Eurasia (VSMOW)
      REAL(dp)                       :: d18O_seawater_nam     ! Relative to PD d18O_seawater from North America (VSMOW)
      REAL(dp)                       :: d18O_seawater_ant     ! Relative to PD d18O_seawater from Antactica (VSMOW)
      REAL(dp)                       :: d18O_seawater_grl     ! Relative to PD d18O_seawater from Greenland (VSMOW)
      
    ! Memory of gT_offset for calculating temperature means 
    IF(teller == 1) THEN 
      ALLOCATE(DTemp_mid(0:end_teller))
      DTemp_mid(0) = 0._dp
    END IF
    
!!--- Calculate mean temperatures
      nmid1 = MAX(1,MIN(C%averaging_time_climate,   teller))         ! over the past 2 kyr
      nmid2 = MAX(1,MIN(C%averaging_time_deepwater, teller))         ! over the past 3 kyr

      dTmean_climate = 0._dp
      DO i = 1, nmid1
        dTmean_climate = dTmean_climate + DTemp_mid(teller-nmid1+i-1)
      END DO
      dTmean_climate = dTmean_climate / nmid1

      dTmean_deepwater = 0._dp
      DO i = 1, nmid2
        dTmean_deepwater = dTmean_deepwater + DTemp_mid(teller-nmid2+i-1)
      END DO
      dTmean_deepwater = dTmean_deepwater / nmid2

!!--- Caclulating the deep-water temperature contribution

      gT_dw_offset    = C%scale_dw * dTmean_deepwater      ! <-- default scaling is 0.20
      d18O_temp_relPD = gamma_dw * gT_dw_offset            ! <-- default slope is -0.28
      
!!--- Calculate the modelled benthic d18O

!     variables of the ice-sheets are stored in two list:
!       Ice_volice(1:4) contains:       Ice_meaniso(1:4) contains:
!     - Ice_volice(1) = hsliso_eas    - Ice_meaniso(1) = meaniso_eas
!     - Ice_volice(2) = hsliso_nam    - Ice_meaniso(2) = meaniso_nam
!     - Ice_volice(3) = hsliso_ant    - Ice_meaniso(3) = meaniso_ant
!     - Ice_volice(4) = hsliso_grl    - Ice_meaniso(4) = meaniso_grl    

      ! In case of only using the NH ice sheets
      IF(C%which_icesheets(1:4) == 'NEFF') THEN

        ! Sea water d18O contributions
        d18O_seawater_eas = Ice_meaniso(1) * Ice_volice(1) / (Ocean_Depth_Earth + sealevel)
        d18O_seawater_nam = Ice_meaniso(2) * Ice_volice(2) / (Ocean_Depth_Earth + sealevel)
        d18O_seawater_ant = 0._dp
        d18O_seawater_grl = 0._dp

        ! Total of all ice sheets, relative to present day
        d18O_seawater_relPD = lambda * d18O_seawater_eas + d18O_seawater_nam

      ! for all other cases do not include any corrections
      ELSE
        
        ! Sea water d18O contributions
        d18O_seawater_eas = Ice_meaniso(1) * Ice_volice(1) / (Ocean_Depth_Earth + sealevel)
        d18O_seawater_nam = Ice_meaniso(2) * Ice_volice(2) / (Ocean_Depth_Earth + sealevel)

        ! For Paleocene runs, relative to zero ice, total change in d18O is calculated with the actual reference values of the data        
        IF (C%paleo_reference_noice) THEN
          d18O_seawater_ant = Ice_meaniso(3) * Ice_volice(3) / (Ocean_Depth_Earth + sealevel) 
          d18O_seawater_grl = Ice_meaniso(4) * Ice_volice(4) / (Ocean_Depth_Earth + sealevel) 
        ELSE
          d18O_seawater_ant = Ice_meaniso(3) * Ice_volice(3) / (Ocean_Depth_Earth + sealevel) -  &
                              pdmiso_ant     * hsliso_ant_PD /  Ocean_Depth_Earth
          d18O_seawater_grl = Ice_meaniso(4) * Ice_volice(4) / (Ocean_Depth_Earth + sealevel) -  &
                              pdmiso_grl     * hsliso_grl_PD /  Ocean_Depth_Earth
        END IF

        ! Total of all ice sheets, relative to present day
        d18O_seawater_relPD = d18O_seawater_eas + d18O_seawater_nam +                             &
                              d18O_seawater_ant + d18O_seawater_grl
       
      END IF

      ! Total modelled benthic d18O
      d18O_modelled = C%PD_benthic_isotope + d18O_seawater_relPD + d18O_temp_relPD

      ! Absolute values of delta_w and delta_T
      d18O_seawater = d18O_seawater_relPD - 0.28_dp     ! <-- change from VSMOW (which is the same as relPD) to VPDB
      d18O_temp     = d18O_modelled - d18O_seawater
   
!!--- The inverse routine

      ! Only run for d18O forcing, otherwise just calculate d18O
      IF(C%choice_forcing <= C%d18O_forcing) THEN
        d18O_change = d18O_modelled - d18O_obs_next
        gT_offset   = dTmean_climate + C%tuner_one * d18O_change
        DTemp_mid(teller) = gT_offset
        
      ! In case of using temperature and sealevel from as forcing from a file, do calculate 
      ! the benthic d18O values, but do not use the inverse routine to compute gT_offset
      ELSE IF(C%choice_forcing == C%relative_slev_temp_forcing) THEN
        dTemp_mid(teller) = forcing_main%gTemp(Clim_Record)
      ! In case of using a climate model as input, no temperature anomaly is calculated, set to zero
      ELSE IF(C%choice_forcing == C%climate_model_forcing) THEN
        dTemp_mid(teller) = 0._dp
      END IF  

      ! Fill contributions in contributions array Contrib(9)
      Contrib(1) = d18O_modelled
      Contrib(2) = d18O_seawater
      Contrib(3) = d18O_temp
      Contrib(4) = d18O_seawater_relPD
      Contrib(5) = d18O_temp_relPD

      Contrib(6) = d18O_seawater_eas
      Contrib(7) = d18O_seawater_nam
      Contrib(8) = d18O_seawater_ant
      Contrib(9) = d18O_seawater_grl
      
  END SUBROUTINE inverse_model_forcing
END MODULE forcing_main_module
