

#' What does it do?
#' 
#' Process input csv files into a list of model inputs. The csv files must have
#' 5 columns:
#' parameter	subroutine	value	variation	amplitude	increment
#' 
#' > sub routine <- the respective subroutine that the value is used for
#' > value       <- the basevalue
#' > variation   <- either 1 or 0. 1 indicates that the respective value is changed
#' > amplitude   <- the amplitude of the change in % (i.e. -20;20 changes the baseline value with 
#'                  -20 nd + 20%)
#' > increment   <- The incremental steps of change in % (i.e. 5 makes that the model generates 
#'                  the +/-20% change in amplitude in steps of 5%  )
#' 
#' @param path_input_file A character string defining the path to the proper
#'  csv input file. Make sure the csv file was saved as 'CSV (Comma delimited) (*.csv)'
#'  in Microsoft Excel.
#' @return A list of list (model run >  subroutine type) with the proper 
#
input_processor <- function(path_input_file) {

# path_input_file = "./data/mat_ha_2.csv"
  
  raw_input <- read.csv(path_input_file, encoding = "UTF-8")
  
  init <- split(raw_input[ , c("parameter", "value")], raw_input$subroutine)
  init <- lapply(init, function(x){
    tidyr::pivot_wider(x, 
                       names_from = "parameter", 
                       values_from = "value")
  })
  init <-  lapply(init, as.list)
  
  varying_variable <- raw_input[which(raw_input$variation == 1), ]
  
  #addition harmen, otherwise line 32 returns ' non-character argument'
  varying_variable$amplitude <- as.character(varying_variable$amplitude)
  # end addition harmen
  
  amplitude <- unlist(strsplit(varying_variable$amplitude, ";"))
  amplitude <- as.numeric(amplitude)
  names(amplitude) <-  c("lower_bound", "upper_bound")
  
  variation_steps <- seq(from = amplitude["lower_bound"],
                         to = amplitude["upper_bound"], 
                         by = varying_variable$increment)
  
  varying_variable$new_values <- list(
    with(varying_variable, 
         value + (value * variation_steps) / 100)
  )
  
  #addition harmen: without this, the parameter that is supposed to deviate doesn't, and extra, unneeded
  # elements are added to the sub_routine parameters
  varying_variable$parameter <- as.character(varying_variable$parameter)
  #end addition harmen
  
  nb_runs <- length(variation_steps)
  
  input_list <- rep(list(init), nb_runs)
  
  
  update_varying_values <- function(input, new_value){
    input[[varying_variable$subroutine]][[varying_variable$parameter]] <- new_value
    return(input)
  }
  
  input_list <- mapply(FUN = update_varying_values, 
                       input_list, 
                       unlist(varying_variable$new_values),
                       SIMPLIFY = FALSE)
  
  syntactic_variation_steps <- ifelse(variation_steps < 0,
                                      paste0("__m", sub("-", "", variation_steps)),
                                      paste0("__p", variation_steps))
  
  names(input_list) <- paste0(varying_variable$parameter, syntactic_variation_steps)
                              
  
  return(input_list)
  
}

