
#' What does it do? 
#' This script calls the different sub-routines that together make the model work. 
# The scipt generates the key variables we are interested in for the analysis:
#'
#' fw_yield           <- the target yield (kg FW/ha/year) used as an input into the model
#' K_offtake          <- K offtake in kg/ha/year
#' K_requirement      <- K requirement in Kg/ha/year
#' tlu                <- the number of TLU required to produce sufficient manure for 1 ha of banana/year
#' crop_range_ratio   <- The area of rangeland needed to produce enough manure to fertilize 1 hectare of banana 
#' N_balance          <- partial N balance in kg/ha/year 
#' P_balance,         <- partial P balance in kg/ha/year 
#' manure_mop_cost    <- input costs in $/ha/year for manure and MOp combined
#' manure_mop_urea_cost <- inout costs if manure Mop and Urea are combined - if urea is needed for a neutral N balance
#' manure_needed        <- input costs in US$/ha/year if only manure is used to satisfy K requirements

model_core <- function(inputs, K_by_mop){
  
  # Overide K_by_mop value if argument provided
  if(!missing(K_by_mop)){
    inputs$manure_cont$K_by_mop <- K_by_mop
  }
  
  subroutines <- list.files("./model/subroutines/", full.names = TRUE)
  
  purrr::walk(subroutines, source)
  
  # Allow to use a list as argument of function
  out_1 <- do.call(fun_banana_nutrient_requirement, inputs$banana_nut_req)
  out_2 <- do.call(fun_manure_production, inputs$herd_comp)
  out_3 <- do.call(fun_manure_availability, inputs$manure_avb)
  
  inputs$manure_cont$nutrient_req <- out_1[grep("N_offtake|P_offtake|soil_N_supply|soil_P_supply|K_requirement", names(out_1))]
  out_4 <- do.call(fun_manure_fertilizer_cost, inputs$manure_cont)
  
  # Calculations
  tlu_1            <- (out_4$manure_needed/out_2$manure_production_standard_cow)*out_2$TLU_standard_cow
  tlu              <- tlu_1/out_3$partitioning_coefficient
  crop_range_ratio <- tlu/inputs$manure_avb$stocking_density
  
  
  return(list(yield_nutrient_req  = list(fw_yield      = out_1$fw_yield,
                                         K_offtake     = out_1$K_offtake,
                                         K_requirement = out_1$K_requirement),
              
              tlu_parameter      = list(tlu              = tlu,
                                        crop_range_ratio = crop_range_ratio),
              
              
              nutrient_balance   = list(N_balance        = out_4$N_balance, 
                                        P_balance        = out_4$P_balance), 
              
              input_costs        = list(manure_mop_cost      = out_4$manure_mop_cost,
                                        manure_mop_urea_cost = out_4$manure_mop_urea_cost,
                                        manure_needed    =     out_4$manure_needed)
  ))
  
}