
#' 4.  Manure Fertilizer cost
#'
#' WHAT DOES IT DO?
#' 
#' This module calculates:
#'      > the cost of manure used alone and how it changes when fertilizers are used
#'      > Nitrogen and Phosporous partial nutrient balances
#'      
#' The module is set to satisfy k requirements, and allows for different types of inputs,
#'  manure, Mop or urea.
#' 
#'___________________________________________________________________________________________________________ 
#' 
#' INPUT PARAMETERS
#' 
#' Input parameters and the definitions are:											
#'   >  fuso_capacity        - Kg of fresh weight manure per truck
#'   >  N_content_manure     - N content (%) in DM manure  
#'   >  P_content_manure     - P content (%) in DM manure  
#'   >  K_content_manure     - K content (%) in DM manure               
#'   >  manure_price         - Price of manure in $/truck
#'   >  fresh_to_dry_weight  - Conversion of Fresh Weight manure (kg) to Dry Weight manure (kg)
#'   >  K_by_mop             - K supplied by MOP, % of total requirement
#'   >  mop_price            - Price of MOP in $/kg
#'   >  urea_price           - Price of Urea in $/kg
#'   >  nutrient_req         - N,P,K requirement of banana. This argument is an output of sub-routine 1, 
#'                             and is a list. 
#'                             
#'____________________________________________________________________________________________________________
#'  

#'      
#'_________________________________________________________________________________________________________   
#'
#'  OUTCOME VARIABLES
#'  
#'  >  manure_price_kg      <- Price of manure in UGX/kg dry matter manure
#'  >  manure_needed        <- Kg of dry weight manure / ha / year needed to compensate K requirements
#'  >  mop_needed           <- Kg of MOP / ha / year needed to compensate K requirements
#'  >  N_supply             <- N supplied by manure quantity applied, in kg/ha/year
#'  >  P_supply             <- P supplied by manure quantity applied, in kg/ha/year
#'  >  K_supply             <- K supplied by manure and MOP quantity applied, in kg/ha/year
#'  >  N_balance            <- N requirements - N supplied, in kg/ha/year
#'  >  P_balance            <- P requirements - P supplied, in kg/ha/year
#'  >  K_balance            <- K requirements - K supplied, in kg/ha/year
#'  >  manure_cost          <- Total input cost if only manure is used to satisfy K demands, in $/ha/year
#'  >  manure_mop_cost      <- Input cost if a combination of manure and MOP are used to satisfy K demands,
#'                             in $/ha/year
#'____________________________________________________________________________________________________________
#'      
#' FUNCTION

fun_manure_fertilizer_cost <- function(N_content_manure,
                                       P_content_manure,
                                       K_content_manure,
                                       manure_price_kg_FW,
                                       fresh_to_dry_weight,
                                       K_by_mop,
                                       mop_price,
                                       urea_price,
                                       nutrient_req){
  
  
  if(!identical(names(nutrient_req), c("N_offtake","P_offtake","soil_N_supply",
                                        "soil_P_supply","K_requirement"))){
    stop("wrong format in nutrient_req list")}
  
  
  
  # 1. DEFINE FIXED PARAMETERS 
  
  #'   definitions and values are:
  K_content_mop     <- 50.22     # K% in MOP fertilizer  
  N_content_urea    <- 46        # N% in Urea fertilizer
  
  # 2. MANURE PRICE,  MANURE AND MOP NEEDED
  
  # 2.1 Calculate the manure price (UGX/ kg DM manure)     
  
  manure_price_kg_DM <- manure_price_kg_FW / fresh_to_dry_weight 
  
  # 2.2 Calculate the amount of manure needed to satisfy K requirement of banana 
  #     given the % of total K supplied by MOP, and given the yield level
  #    ( kg DM manure needed / ha banana)
  
  manure_needed <-  ((100-K_by_mop)/100) * (nutrient_req$K_requirement / (K_content_manure/100)) 
  
  # 2.3 Calculate the amount of MOP needed to satisfy K requirement of banana 
  #     given the % of total K supplied by MOP, and given the yield level
  #    ( kg MOP needed / ha banana)
  
  mop_needed <- ( nutrient_req$K_requirement*(K_by_mop/100))/(K_content_mop/100)
  
  
  # 3. NUTRIENT SUPPLY and nutrient balance
  
  # 3.1 Calculate the N,P,K supply to banana (kg/ha/year), given:
  #     banana yield, N,P,K content manure, % of total K supplied by MOP 
  
  N_supply_manure     <- manure_needed * (N_content_manure / 100)
  P_supply_manure     <- manure_needed  * (P_content_manure / 100)
  K_supply_total      <- (((100-K_by_mop)/100)* nutrient_req$K_requirement) +
    ((K_by_mop/100)* nutrient_req$K_requirement)
  
  
  # 3.2 Calculate the N,P,K balance (kg/ha/year), given:
  #     N, P, K supply (3.1) and the N,P OFFTAKE, and the K REQUIREMENT
  #     Note that by definition, K balance is 0, since the model is set to balance K requirements
  
  N_balance <- N_supply_manure + nutrient_req$soil_N_supply - nutrient_req$N_offtake 
  
  P_balance <- P_supply_manure + nutrient_req$soil_P_supply - nutrient_req$P_offtake
  
  K_balance <- K_supply_total  -  nutrient_req$K_requirement
  
  # 3.3 Calculate if and how much extra N is needed (only in case N balance is negative) and,
  #     amount of urea needed to get balance from negative to 0. 
  
  urea_needed <- ifelse(N_balance < 0, (abs(N_balance))/(N_content_urea/100), 0)
  
  
  # 4. CALCULATE INPUT COSTS (US$/ha/year)
  
  # 4.1 Define conversion factor for converting UGX to US$
  #    on 28th of May, 2020
  #    source:  https://www.xe.com/currencyconverter/convert/?Amount=1&From=USD&To=UGX 
  
  UGX_USD_conVersion  <- 0.000264077
  
  # 4.2 Input costs if only manure is used to satisfy k requirements. 
  
  manure_cost <- (  manure_price_kg_DM*(nutrient_req$K_requirement/K_content_manure*100) ) * UGX_USD_conVersion
  
  # 5.3 Input costs if manure and MOP are combined to satisfy k requirements.
  #     Note: % of total K requirement supplied by MOP is a user defined variable.   
  
  manure_mop_cost <- ( (manure_needed* manure_price_kg_DM) + (mop_needed*mop_price) ) * UGX_USD_conVersion
  
  # 4.4 Input costs if manure and MOP are combined to satisfy k requirements, and
  #     urea is used to supply N in case the N balance is negative
  #     Note: % of total K requirement supplied by MOP is a user defined variable.
  
  manure_mop_urea_cost <- ((manure_needed* manure_price_kg_DM) + (mop_needed*mop_price) +
                             (urea_needed*urea_price)) *
    UGX_USD_conVersion
  
  # 5. RETURN OUTPUTS
  
  return(list( manure_needed   = manure_needed,
               mop_needed      = mop_needed,
               N_supply_manure = N_supply_manure,
               P_supply_manure = P_supply_manure,
               K_supply_total  = K_supply_total,
               N_balance       = N_balance,
               P_balance       = P_balance,
               K_balance       = K_balance,
               urea_needed     = urea_needed,
               manure_cost     = manure_cost,
               manure_mop_cost = manure_mop_cost,
               manure_mop_urea_cost = manure_mop_urea_cost))  
  
} 




