
#' WHAT DOES IT DO?
#' 
#' This module calculates nutrient offtake and nutrient requirement, 
#' given target yields. indigenous soil supply and the nutrient recovery fraction 
#'___________________________________________________________________________________________________________ 
#' 
#' INPUT PARAMETERS // 
#' 
#' Input parameters and the definitions are:											
#'   >  fw_yield         <- banana yield in kg FW/ha/year
#'   >  N_recovery       <- Fraction of N applied taken up by the crop
#'   >  P_recovery       <- Fraction of P applied taken up by the crop
#'   >  K_recovery       <- Fraction of K applied taken up by the crop
#'   >  soil_K_supply    <- the amount of indegenous soil K supply (kg/ha/year)
#'   >  soil_N_supply    <- the amount of indegenous soil N supply (kg/ha/year)
#'   >  soil_P_supply    <- the amount of indegenous soil P supply (kg/ha/year)
#'   >
#'____________________________________________________________________________________________________________

#'
#'  OUTPUT VARIABLES
#'  
#'  >  N_offtake  <- in kg/ha/year
#'  >  P_offtake  <- in kg/ha/year
#'  >  K_offtake  <- in kg/ha/year
#'  >  N_requirement  <- in kg/ha/year
#'  >  P_requirement  <- in kg/ha/year
#'  >  K_requirement  <- in kg/ha/year
#'  

#function
fun_banana_nutrient_requirement <- function(fw_yield,
                                                 N_recovery,
                                                 P_recovery,
                                                 K_recovery,
                                                 soil_K_supply,
                                                 soil_N_supply,
                                                 soil_P_supply
){
  
  
  
  #'    Fixed parameters
  #'   
  #'   the definitions and values are:
  
  dry_matter_bunch <- 0.153     # Fraction of dry matter in a fresh bunch 
  N_fraction_bunch <- 0.0109    # Fraction of N in bunch, on DM basis
  P_fraction_bunch <- 0.0011    # Fraction of p in bunch, on DM basis
  K_fraction_bunch <- 0.0355    # Fraction of K in bunch, on DM basis
  
  
  yield_dm_ha <- fw_yield * dry_matter_bunch
  
  
  N_offtake <- yield_dm_ha * N_fraction_bunch
  P_offtake <- yield_dm_ha * P_fraction_bunch
  K_offtake <- yield_dm_ha * K_fraction_bunch
  
  N_requirement <- ifelse(N_offtake - soil_N_supply < 0, 
                          0, (N_offtake - soil_N_supply)/N_recovery )
  
  P_requirement <- ifelse(P_offtake - soil_P_supply < 0, 
                          0, (P_offtake - soil_P_supply)/P_recovery )
  
  K_requirement <- ifelse(K_offtake - soil_K_supply < 0, 
                          0, (K_offtake - soil_K_supply)/K_recovery )
  
  return(list( fw_yield  = fw_yield,
               N_offtake = N_offtake,
               P_offtake = P_offtake,
               K_offtake = K_offtake,
               soil_K_supply = soil_K_supply,
               soil_N_supply = soil_N_supply,
               soil_P_supply = soil_P_supply,
               N_requirement = N_requirement,
               P_requirement = P_requirement,
               K_requirement = K_requirement))
  
} 