
library(dplyr)

#'                               DESCRIPTION
#'-------------------------------------------------------------------------
#'
#'
#' This script generates three dataframes. 
#' Each dataframe contains the sensitivity indices for 1 scenario:
#' 
#' 
#'  1) 0  % of total K requirement supplied by mop 
#'  2) 25 % of total K requirement supplied by mop 
#'  3) 50 %  of total K requirement supplied by mop
#'
#' Each dataframe contains the sensitivity index for each sensitiivty parameter, for each of the 6
#' key model outputs
#'

#source output dataframes
#'
#'______________________________________________________________________________________________________
#'
#'
#'       DEFINE FUNCTION TO CALCUALTE SENSITIVITY INDEX
#'


#' create a function that calculates the sensitivity index for a sensitivity parameter.
#' i.e. it expresses what the influence of a certain parameter is on each of the key outputs of the model 
#'
#' use formula that calculates the sensitivty index (SI) as in descheemaeker et al., 2018
#
#'     SI = ((OUT 20%) - (OUT  -20%))
#'   __________________________________             
#'               (OUT base)
#'
#' In which:
#'           > OUT 20% = output variable of interest when sensitivity parameter deviates +20% from base value
#'           > OUT -20% = output variable of interest when sensitivity parameter deviates -20% from base value
#'           > OUT  base = output variable of interest when sensitivity parameter is set at base value 


si_calculator <- function(output_dataframe) {
  
  #order dataframe according to $variable,
  #otherwise SI indices are not assigned in the right order
  
  output_dataframe <- output_dataframe[order(output_dataframe$variable),]
  
  #calculate index
  sens_index <-  (with(output_dataframe, value[variation == 20]) - 
                    with(output_dataframe, value[variation == -20])) /
    with(output_dataframe, value[variation == 0])
  
  sens_index <- abs(sens_index)
  
  return(sens_index = sens_index)
}

#'
#'___________________________________________________________________________________________________
#'
#'    
#'    CALCULATE SENSITIVITY INDICES FOR EACH OF SCENARIOS
#'

#'loop si calculator trough each of the 3 output lists  and create dataframes.
#'these dataframes contain the si for each input parameter and each output variable, 
#' 1 dataframe per mop-scenario
#'
si_mop_0  <-  as.data.frame(lapply(all_output_0_mop, si_calculator)) 
si_mop_25  <-  as.data.frame(lapply(all_output_25_mop, si_calculator)) 
si_mop_50  <-  as.data.frame(lapply(all_output_50_mop, si_calculator)) 

# transpose dataframes
si_mop_0   <- as.data.frame(t(si_mop_0))
si_mop_25  <- as.data.frame(t(si_mop_25))
si_mop_50  <- as.data.frame(t(si_mop_50))


#assign column names to sensitivity index dataframe
output_variable_name <- all_output_0_mop[[1]]
output_variable_name <- levels(output_variable_name$variable)

colnames(si_mop_0) <- output_variable_name
colnames(si_mop_0) <- paste("mop_0", colnames(si_mop_0), sep = "_")

colnames(si_mop_25) <- output_variable_name
colnames(si_mop_25) <- paste("mop_25", colnames(si_mop_25), sep = "_")

colnames(si_mop_50) <- output_variable_name
colnames(si_mop_50) <- paste("mop_50", colnames(si_mop_50), sep = "_")

remove(output_variable_name)

#'
#'_________________________________________________________________________________________________________
#'  
  


#'
#' LIST THE MOST INFLUENTIAL PARAMETERS PER SCENARIO AND PER OUTPUT VARIABLE
#'

# bind the 3 dataframes for the 3 scenarios, and define seperate dataframes per outcome variable
si_all_scenarios <- cbind(si_mop_0, si_mop_25, si_mop_50)

si_manure_mop  <- select(si_all_scenarios, contains("manure_mop_cost"))
si_N_balance   <- select(si_all_scenarios, contains("N_balance"))
si_P_balance   <- select(si_all_scenarios, contains("P_balance"))
si_tlu         <- select(si_all_scenarios, contains("tlu"))

# save dataframes containting the sensitivty indices as csv files and remove dataframes
write.csv(si_manure_mop, "./results/table_6_input_costs.csv")
write.csv(si_N_balance, "./results/table_6_N_balance.csv")
write.csv(si_P_balance, "./results/table_6_P_balance.csv")
write.csv(si_tlu, "./results/table_6_tlu_required.csv")

remove(si_manure_mop, si_N_balance, si_P_balance, si_tlu)

#______________________________________________________________________________________________________