
#' This script calculates the nr of TLU needed for Birere and Rugaaga, given:
#'  > the total banana area in Birere and Rugaaga
#'  > the total nr of TLU present in the 'manure catchment area' for these sub-counties
#'  
#'  for different yield levels, and for different % of total K requirement supplied by MOP

#' Source the relevant scripts, and keep only what is needed for this analysis
library(ggplot2)

rm(list=setdiff(ls(), c("tlu_yield", "manure_needed_yield","target_yields", "my_figure_theme", "rangeland_area",
                        "birere_banana_area", "rugaaga_banana_area", "birere_banana_area",
                       "fun_banana_nutrient_requirement", "fun_manure_availability", "fun_manure_fertilizer_cost",
                       "fun_manure_production", "input_processor", "model_core","output_processor") ))
source("./data_analysis/scripts/my_figure_theme.R")

########################################################################################################
#########################################################################################################


#'             % of total collectable manure that is sold
#'          
#'  SNV TIDE dataset ("banana agronomy/data/SNV/Farm assessment/for analysis/TIDE_manure use)
#'  shows what cattle farmers do with their collectable manure
  
purpose                <- c("pasture", "fodder crop", "arable crop", "sold", "discarded")
perc                   <- c(8,	20,	63, 8,2)
manure_allocation      <- as.data.frame(cbind(purpose, perc) )  
manure_allocation$perc <- as.numeric(as.character(manure_allocation$perc))

fraction_manure_sold <- 0.08

remove(perc,purpose)

#' make barchart showing what cattle farmers do with their manure and save as figure
#' 
collectable_manure_allocation <-  ggplot(manure_allocation, aes(x=purpose, y = perc))+
  geom_bar(stat = "identity")+
  ylab("% of collectable manure allocated")+
  xlab("")+
  my_figure_theme

#save a copy
png(filename = "./results/figure_8.png",
    width = 500,
    height = 400)

print(collectable_manure_allocation)


dev.off()

#save as eps
Fig_8 <- collectable_manure_allocation

ggexport(Fig_8,filename = "./results/figure_8.eps",
         width = 1200,
         height = 495)


########################################################################################################
########################################################################################################


#'                   TLU present in total catchment area and in study subcounties itself
#'                   
#'                   Collectable manure in catchment areas: potential and actual
#'                   
#'                   


#' total TLU  in catchments
total_tlu_catchment <-  aggregate(rangeland_area$total_tlu,
                                  by = list(study_region = rangeland_area$study_region),
                                  FUN = sum)


#'total collectable manure in the catchment areas
total_manure_catchment <- aggregate(rangeland_area$manure_collectable,
                                    by = list(study_region = rangeland_area$study_region),
                                    FUN =  sum)

#total potential manure in catchments
bir_catch_manure_potential <- total_manure_catchment[[1,2]]
rug_catch_manure_potential <- total_manure_catchment[[2,2]]
remove(total_manure_catchment)

#total acual manure in catchments 
bir_catch_manure_actual  <- bir_catch_manure_potential * fraction_manure_sold
rug_catch_manure_actual  <- rug_catch_manure_potential * fraction_manure_sold 


#total tlu in catchments
bir_catch_tot_tlu <- total_tlu_catchment[[1,2]]
rug_catch_tot_tlu <- total_tlu_catchment[[2,2]]
remove(total_tlu_catchment)

#'  total tlu in the sub_counties itself
bir_tot_tlu <- rangeland_area[rangeland_area$sub_county == "birere",]
bir_tot_tlu <- bir_tot_tlu[,"total_tlu"]

rug_tot_tlu <- rangeland_area[rangeland_area$sub_county == "rugaaga",]
rug_tot_tlu <- rug_tot_tlu[,"total_tlu"]


###########################################################################################################
##########################################################################################################

#'                            TLU and manure needed for Birere

bir_reg_analysis                <- tlu_yield
bir_reg_analysis$sub_county     <- "birere" 
bir_reg_analysis                <- merge(bir_reg_analysis, manure_needed_yield, by = c("yield", "manure"))

#rename columns
bir_reg_analysis <- bir_reg_analysis %>% dplyr::rename(variable_tlu    = variable.x,
                                                       tlu_needed      = value.x,
                                                       variable_manure = variable.y,
                                                       manure_needed   = value.y)
#total tlu needed birere
bir_reg_analysis$tot_tlu_needed   <- bir_reg_analysis$tlu_needed * birere_banana_area
bir_reg_analysis$tot_tlu_f_1000   <- bir_reg_analysis$tot_tlu_needed / 1000


#total manure needed birere
bir_reg_analysis$total_manure_needed      <- bir_reg_analysis$manure_needed * birere_banana_area
bir_reg_analysis$total_manure_needed_tons <- bir_reg_analysis$total_manure_needed / 1000 

###########################################################################################################
##########################################################################################################

#'                            TLU and manure needed for Rugaaga


rug_reg_analysis              <- tlu_yield
rug_reg_analysis$sub_county   <- "rugaaga" 
rug_reg_analysis              <- merge(rug_reg_analysis, manure_needed_yield, by = c("yield", "manure"))  

#rename columns
rug_reg_analysis <- rug_reg_analysis %>% dplyr::rename(variable_tlu    = variable.x,
                                                       tlu_needed      = value.x,
                                                       variable_manure = variable.y,
                                                       manure_needed   = value.y)
#total tlu needed rugaaga
rug_reg_analysis$tot_tlu_needed   <- rug_reg_analysis$tlu_needed * rugaaga_banana_area
rug_reg_analysis$tot_tlu_f_1000   <- rug_reg_analysis$tot_tlu_needed / 1000

#total manure needed rugaaga
rug_reg_analysis$total_manure_needed      <- rug_reg_analysis$manure_needed * rugaaga_banana_area
rug_reg_analysis$total_manure_needed_tons <- rug_reg_analysis$total_manure_needed / 1000 


###################################################################################################
###################################################################################################

#prepare dataframe containing the lables for plotting tlu in Birere
label_bir_tlu <- c("TLU Birere", "TLU catchment")
bir_tlu       <- c(bir_tot_tlu, bir_catch_tot_tlu)  
bir_tlu       <-  cbind.data.frame(label_bir_tlu, bir_tlu)

remove(label_bir_tlu, bir_tot_tlu, bir_catch_tot_tlu)

bir_tlu$tlu_f_1000 <- bir_tlu$bir_tlu/1000


#prepare dataframe containing the lables for plotting tlu in Rugaaga
label_rug_tlu <- c("TLU Rugaaga", "TLU catchment")
rug_tlu       <- c(rug_tot_tlu, rug_catch_tot_tlu)  
rug_tlu       <-  cbind.data.frame(label_rug_tlu, rug_tlu)

remove(label_rug_tlu, rug_tot_tlu, rug_catch_tot_tlu)

rug_tlu$tlu_f_1000 <- rug_tlu$rug_tlu/1000


#prepare dataframe containing the lables for plotting collectable manure in Birere
label_bir_manure <- c("potential", "actual")
bir_manure       <- c(bir_catch_manure_potential, bir_catch_manure_actual)  
bir_manure       <-  cbind.data.frame(label_bir_manure, bir_manure)
remove(label_bir_manure, bir_catch_manure_actual, bir_catch_manure_potential)

bir_manure$manure_tonnes <- bir_manure$bir_manure/1000

#prepare dataframe containing the lables for plotting collectable manure in Rugaaga
label_rug_manure <- c("potential", "actual")
rug_manure       <- c(rug_catch_manure_potential, rug_catch_manure_actual)  
rug_manure       <-  cbind.data.frame(label_rug_manure, rug_manure)
remove(label_rug_manure, rug_catch_manure_actual, rug_catch_manure_potential)

rug_manure$manure_tonnes <- rug_manure$rug_manure/1000



#########################################################################################################
########################################################################################################

#' plot total collectable manure in Birere catchment

plot_manure_availability_birere <- 
  ggplot(data = bir_reg_analysis, aes(x= yield/1000, y=total_manure_needed_tons))+
  geom_line(aes(linetype = manure), size = 1)+
  scale_linetype_manual(name  ="% manure used",
                        breaks=c("100", "75", "50"),
                        labels=c("100%", "75%", "50%"), 
                        values = c("solid", "dashed", "dotted"))+
  ylab("Manure required (t DM/year)")+
  xlab("Yield (t FW/ha/year)")+
  #annotate text target yields and add target yield lines
  annotate("text", x = target_yields$yield -1.3, y = 60000, 
           label = target_yields$labels, size = 5)+
  geom_vline(xintercept = target_yields$yield, linetype = "dashed")+
  #annotate text manure present and add lines
  annotate("text", y = bir_manure$manure_tonnes +3000, x =60, 
           label = bir_manure$label_bir_manure, size = 5)+
geom_hline(yintercept = bir_manure$manure_tonnes)+
  my_figure_theme
  
#########################################################################################################
########################################################################################################

#' plot total collectable manure in Rugaaga catchment 

plot_manure_availability_rugaaga <- 
ggplot(data = rug_reg_analysis, aes(x= yield/1000, y=total_manure_needed_tons))+
  geom_line(aes(linetype = manure), size = 1)+
  scale_linetype_manual(name  ="% manure used",
                        breaks=c("100", "75", "50"),
                        labels=c("100%", "75%", "50%"), 
                        values = c("solid", "dashed", "dotted"))+
  ylab("Manure required (t DM/year)")+
  xlab("Yield (t FW/ha/year)")+
   #annotate text target yields and add target yield lines
  annotate("text", x = target_yields$yield -1.3, y = 150000, 
           label = target_yields$labels, size = 5)+
  geom_vline(xintercept = target_yields$yield, linetype = "dashed")+
  #annotate text manure present and add lines
  annotate("text", y = rug_manure$manure_tonnes +6000, x =60, 
           label = rug_manure$label_rug_manure, size = 5)+
  geom_hline(yintercept = rug_manure$manure_tonnes)+
  my_figure_theme


#plot figures manure availabilty at regional scale


regional_manure_availability <- 
  ggarrange(plot_manure_availability_birere, plot_manure_availability_rugaaga,
          labels = c("A", "B"),
          common.legend = T)


png(filename = "./results/figure_9.png",
    width = 1200,
    height = 495)

print(regional_manure_availability)

dev.off()

#Save as EPS
Fig_9 <-regional_manure_availability

ggexport(Fig_9,filename = "./results/figure_9.eps",
         width = 1200,
         height = 495)


#_______________________________________________________________________________________________________________


#'     SOME CALCULATIONS FOR THE TABLE SPATIAL ANALYSIS


#how many TLU are available per ha banana?
bir_tlu$TLU_ha_banana <- bir_tlu$bir_tlu / birere_banana_area
rug_tlu$TLU_ha_banana <- rug_tlu$rug_tlu / rugaaga_banana_area

#' how much manure would potentially be available. 1 TLU produces
collectable_manure_tlu <- ((250^0.75)* 12.58) * 0.54 
#' 427 kg DM collectable manure per year (see subroutines 3 and 4)
#' 
bir_tlu$pot_manure_ha <- bir_tlu$TLU_ha_banana* collectable_manure_tlu
rug_tlu$pot_manure_ha <- rug_tlu$TLU_ha_banana* collectable_manure_tlu                          
                         
                          
#' how much K is available per ha bannaa per year?
#' K content manure is 1.11
bir_tlu$pot_potassium_ha <- bir_tlu$pot_manure_ha * (1.11/100)
rug_tlu$pot_potassium_ha <- rug_tlu$pot_manure_ha * (1.11/100)


##################################################################################################


#'       ADDITIONAL GRPAHS



#plot birere_TLU

plot_birere <-  ggplot(data = bir_reg_analysis, aes(x= yield/1000, y=tot_tlu_f_1000 , group= manure))+
                geom_line()+
                geom_point(aes(shape = manure))+
                ylab("Total TLU needed (x 1000)")+
                xlab("Yield (kg FW/ha/year)")+
                xlim(25,70)+
  #annotate text target yields
                annotate("text", x = target_yields$yield -1.3, y = 200, 
                label = target_yields$labels, size = 4)+
  #annotate text TLU present
                annotate("text", y = bir_tlu$tlu_f_1000 +9, x =60, 
                label = bir_tlu$label_bir, size = 4)+
  # add lines
                geom_vline(xintercept = target_yields$yield, linetype = "dashed")+
                geom_hline(yintercept = bir_tlu$tlu_f_1000)+
  #define theme and further layout
                theme(legend.position = "none")+
                my_figure_theme

#plot rugaaga

plot_rugaaga <-  ggplot(data = rug_reg_analysis, aes(x= yield/1000, y= tot_tlu_f_1000, fill= manure))+
                 geom_line(aes(color = manure))+
                 xlab("Yield (kg FW/ha/year)")+
                 ylab("")+
                 xlim(25,70)+
  #annotate text target yields
                 annotate("text", x = target_yields$yield -1.3, y = 480, 
                 label = target_yields$labels, size = 4)+
  #annotate text TLU present
                 annotate("text", y = rug_tlu$tlu_f_1000 +12, x =60000, 
                 label = rug_tlu$label_rug_tlu, size = 4)+
  # add lines
                 geom_vline(xintercept = target_yields$yield, linetype = "dashed")+
                 geom_hline(yintercept = rug_tlu$tlu_f_1000)+
#define theme and further layout
                 theme(legend.position = "none")+
                 my_figure_theme


#plot the 2 figures + legend
ggarrange(plot_birere, plot_rugaaga, 
          common.legend = T,
          labels = c("A", "B"))







