
#' This script calculates:
#'     >  the total area of the catchment area for Birere and Rugaaga
#'     >  the number of TLU present in the respective catchment areas
#'      

library(dplyr)

# Iimport files

catch_bir           <- read.csv(header = T, "./model/data/catchment_area_birere.csv")
catch_rug           <- read.csv(header = T, "./model/data/catchment_area_rugaaga.csv")
district_stock_dens <- read.csv(header = T, "./model/data/total_catchment_area.csv")

#remove unneeded columns and add column indicating which study region belongs to it
catch_bir <- subset(catch_bir, select = c(3,7,9))
catch_rug <- subset(catch_rug, select = c(3,7,9))

catch_bir$studyregion <- "birere"
catch_rug$studyregion <- "rugaaga"

#merge dataframes and clean: lower cases and understandable columnnames
rangeland_area         <- rbind2(catch_bir, catch_rug)
rangeland_area$ADM1_EN <- tolower(rangeland_area$ADM1_EN)
rangeland_area$ADM3_EN <- tolower(rangeland_area$ADM3_EN) 

colnames(rangeland_area) <- c("district", "sub_county", "area_m2", "study_region")
remove(catch_bir, catch_rug)

#calculate sub_county area in ha
rangeland_area$area_ha <- rangeland_area$area_m2 / 10000


#'   calculate the nr of TLU per sub-county  
#'
#' According to UBOS livestock census (2008), the total nr of cattle per district is (pg 135)
#'   >  Isingiro: 180350
#'   > Kiruhura : 342320
#'   
#'   calculate the area (in ha) for each district, and plug in the total cattle numbers 

district_stock_dens   <- subset(district_stock_dens, district_stock_dens$ADM1_EN != "MBARARA")    

district_stock_dens   <-  aggregate(x = district_stock_dens$area,
                  by = list(district = district_stock_dens$ADM1_EN),
                  FUN = sum)

district_stock_dens$district  <- tolower(district_stock_dens$district)
names(district_stock_dens)[2] <- "area_m2"

total_cattle <- c(180350, 342320)
district_stock_dens <- cbind(district_stock_dens, total_cattle)
remove(total_cattle)

district_stock_dens$area_ha <- district_stock_dens$area_m2 /10000

#' convert cattle numbers to TLU, using  a script in R project banana_manure_model.
#' The function calculates the bodyweight of a standard cow, given herd composition and 
#' breed characteristics.

#' according to UBOS livestock census (2008, pg 135), % of indigeneous and crossbred cows are:
#'   Isingiro: 94.1% Indigeneous (of which 96.6% Ankole and 3.1% zebu), 5.9% crossbred
#'   Kiruhura: 80.5% indigeneous (of which 94% Ankole and 6% zebu), 19.5% crossbred 
#' It is assumed that all indigenous cows are Ankole cows. 

body_weight_standard_cow <- (c(fun_body_weight_standard_cow(
                                 fraction_cow        = 0.495,
                                 fraction_heifer     = 0.293,
                                 fraction_calve      = 0.178,
                                 fraction_bull       = 0.021,
                                 fraction_steer      = 0.013,
                                 fraction_ankole     = 0.941,
                                 fraction_crossbreed = 0.059),
                  fun_body_weight_standard_cow(
                                 fraction_cow        = 0.495,
                                 fraction_heifer     = 0.293,
                                 fraction_calve      = 0.178,                   
                                 fraction_bull       = 0.021,
                                 fraction_steer      = 0.013,
                                 fraction_ankole     = 0.805,
                                 fraction_crossbreed = 0.195) ))

body_weight_standard_cow  <- unlist(body_weight_standard_cow)
district_stock_dens       <- cbind(district_stock_dens, body_weight_standard_cow)

#convert standard cow to TLU and calculate stocking density in TLU/ha ! at district level!
district_stock_dens$tlu_standard_cow <- (district_stock_dens$body_weight_standard_cow ^0.75) / 
                                         (250^0.75)

district_stock_dens$tlu_total  <- district_stock_dens$total_cattle * district_stock_dens$tlu_standard_cow
district_stock_dens$sd_tlu_ha  <- district_stock_dens$tlu_total / district_stock_dens$area_ha

#' attach stocking density (TLU/ha) for each district to dataframe rangeland_area to calculate 
#' the total TLU at sub_county level

stocking_dens  <- subset(district_stock_dens, select = c("district", "sd_tlu_ha"))
rangeland_area <- merge(stocking_dens, rangeland_area, by = c("district"))
remove(stocking_dens)
rangeland_area$total_tlu <- rangeland_area$area_ha * rangeland_area$sd_tlu_ha

#' how much manure can be collected from the rangeland area?
#'  > calculate metabolic body weight: 1 TLU is 250 kg, metabolic BW is 250^0.75

#'  1 TLU produces
manure_production_TLU <- (250^0.75)*12.58

#' manure production of all TLU in the catchment area  
rangeland_area$total_manure_excreted <- rangeland_area$total_tlu * manure_production_TLU 

#' not all excreted manure is available for collection, cows stay in the kraal for 13 hours a day
#' collectable manure, and during storage, on average 47% of manure mass is lost
rangeland_area$manure_collectable <- rangeland_area$total_manure_excreted * (13/24) * 0.53

#' total catchment area (ha) for each sub-county
#' 
total_catchment_area <- aggregate(rangeland_area$area_ha,
                                  by = list(catchment = rangeland_area$study_region),
                                  FUN = sum)
names(total_catchment_area)[2] <- "total_catchment_area_ha"

#' total TLU present in cathcment area
total_TLU_per_catchment <- aggregate(rangeland_area$total_tlu,
                                     by = list(catchment = rangeland_area$study_region),
                                     FUN = sum)
names(total_TLU_per_catchment)[2] <- "total_TLU_catchment"

#' total collectable manure for each sub-county
total_manure_per_catchment <- aggregate(rangeland_area$manure_collectable,
                                        by = list(catchment = rangeland_area$study_region),
                                        FUN = sum)
names(total_manure_per_catchment)[2] <- "total_kg_collectable_DM_manure"


total_TLU_per_catchment <- merge(total_TLU_per_catchment, total_catchment_area, 
                                    by = c("catchment"))
total_manure_per_catchment <- merge(total_TLU_per_catchment, total_manure_per_catchment, 
                                    by = c("catchment"))
remove(total_catchment_area, total_TLU_per_catchment)

#' convert hectares to square km, and kg manure to ton manure 
total_manure_per_catchment$total_catchment_area_km2 <- 
                       total_manure_per_catchment$total_catchment_area_ha * 0.01


total_manure_per_catchment$total_tons_collectable_DM_manure <- 
             total_manure_per_catchment$total_kg_collectable_DM_manure / 1000

#' manure available/ha banana, if spread evenly
#' get banana area and remove unneeded scripts and functions
banana_area_ha             <- c(birere_banana_area, rugaaga_banana_area)
total_manure_per_catchment <- cbind(total_manure_per_catchment, banana_area_ha)

total_manure_per_catchment$banana_area_km2 <- total_manure_per_catchment$banana_area_ha * 0.01

#' calculate the manure (kg DM/ha banana) avaialable, if all the collectable manure in the 
#' catchment area would be spread evenly over the banana area
total_manure_per_catchment$manure_available_kg_DM_ha <- total_manure_per_catchment$total_kg_collectable_DM_manure /
                                                        total_manure_per_catchment$banana_area_ha

