#'   Herd composition and manure production
#'
#' WHAT DOES IT DO?
#' 
#' This module calculates the bodyweight, TLU of a 'standard cow' in Isingiro and Kiruhura district. 
#' The UBOS livestock census presents stocking density in nr of cows/district.
#'___________________________________________________________________________________________________________ 
#' 
#'__________________________________________________________________________________________________________
#'
#'
#'
#'  INPUT PARAMETERS 
#' 
#' Input parameters and the definitions are:		
#'   >  fraction_cow           - fraction of cows in the herd
#'   >  fraction_heifer        - fraction of heifers in the herd
#'   >  fraction_calve         - fraction of calves in the herd  
#'   >  fraction_bull          - fraction of bulls in the herd
#'   >  fraction_steer         - fraction of steers in the herd  
#'                                    
#'   >  fraction_ankole           - The fraction of ankole cows in the herd
#'   >  fraction_crossbreed       - The fraction of crossbreed cows in the herd. 
#'                                  Note that fraction_ankole + fraction_crossbreed should be 100 
#'                                  
#'__________________________________________________________________________________________________________  

#'  OUTPUT VARIABLES
#'  
#'  body_weight_standard_cow  <- body weight (kg) of a standard cow
#'  
#'___________________________________________________________________________________________________________
#'                               
#'  FUNCTIONS                                                          

fun_body_weight_standard_cow <- function(fraction_cow,
                                  fraction_heifer,
                                  fraction_calve,
                                  fraction_bull,
                                  fraction_steer,
                                  fraction_ankole,
                                  fraction_crossbreed){
                                
  
  
  # 1.1 prepare vectors for dataframe: df_manure
  
  animal_category  <- rep(c("cow", "heifer", "calf", "bull", "steer"), times = 2) 
  breed            <- rep(c("Ankole", "crossbreed"), each = 5)
  herd_composition <- rep(c(fraction_cow, fraction_heifer, fraction_calve, fraction_bull, fraction_steer),
                          times = 2)  
  
  
  #' 1.2  Insert weights of typical livestock; use this to calculate body weight of different
  #'      sex X age X breed categories 
  
  #' Body weight (kg) of typical livestock in SSA (kg) for cows, heifers, calves, bulls 
  #' (Njuki et al., 2011). Steers were assumed to have a similar body weight as bulls.
  #'
  weight_typical_livestock <- rep(c(250, 180, 75, 320, 320), times = 2)   
  
  #' Average weight of an Ankole cow, as measured by Byenka, 2004 is 306 kg. 
  #' Average weight of an Ankole X Friesian crossbreeds, as reported by Mulindwa et al., 2008 is 460 kg.
  #'  
  #' Body weight of heifers, calves, bulls and steers for Ankole and crossbreeds cattle is calculated
  #' according to weight_typical_livestock, taking 306 and 460 kg/ mature cow as a reference 
  #' for Ankole and crossbreed cattle, respectively.
  #' 
  conversion_factor_ankole      <- 306 / 250 
  conversion_factor_crossbreed  <- 460 / 250 
  
  df_manure <- data.frame(animal_category, breed, herd_composition, weight_typical_livestock)
  remove(animal_category, herd_composition, breed, weight_typical_livestock)
  
  df_manure$body_weight <- ifelse(df_manure$breed == "Ankole", 
                                  df_manure$weight_typical_livestock * conversion_factor_ankole,
                                  ifelse(df_manure$breed == "crossbreed", 
                                         df_manure$weight_typical_livestock * conversion_factor_crossbreed,
                                         NA))
  
  
  # 2 Calculate the manure production of a 'standard cow'
  #'_____________________________________________________
  
  # 2.1 calculate the fraction of each category in the standard herd
  
  # (i.e. crossbreed cows, crossbreed heifers,  ankole steers etc),
  # given the herd compostion and the fraction of both breeds in the herd    
  
  df_manure$fraction_per_category <- ifelse ( df_manure$breed == "Ankole", 
                                              df_manure$herd_composition * fraction_ankole,
                                              ifelse ( df_manure$breed == "crossbreed",
                                                       df_manure$herd_composition * fraction_crossbreed,
                                                       NA))
  
  # 2.2 Calculate body weight and TLU  and manure production of a 'standard' cow 
  #     (kg DM manure/ animal category) 
  
  #df_manure$manure_production <- df_manure$fraction_per_category * 
  # df_manure$manure_production_per_category
  
  body_weight_standard_cow  <- df_manure$body_weight * df_manure$fraction_per_category
  body_weight_standard_cow  <- sum(body_weight_standard_cow) 
  
  TLU_standard_cow <- df_manure$TLU * df_manure$fraction_per_category
  TLU_standard_cow <- sum( TLU_standard_cow)
  
  
  # 3 return outcome variables
  
  return(list( body_weight_standard_cow = body_weight_standard_cow))
}

