% scriptfile TwoBlocksThreeFaults
%
% Computes depletion-induced fault stresses in a two-block reservoir with one central and two
% bounding faults. Generates Figure 9 in Cornelissen and Jansen: "On the Derivation of
% Closed-Form Expressions for Displacements, Strains and Stresses Inside Poroelastic 
% Reservoirs", submitted to Journal of Geophysical Research - Solid Earth, in 2023.
%
% Jan Dirk Jansen, j.d.jansen@tudelft.nl
%


clear all
close all

% Initialize:
o = zeros(6,1);
p = zeros(6,1);
q = zeros(6,1);
r = zeros(6,1);
s = zeros(6,1);
theta_deg = zeros(6,1);
theta = zeros(6,1);

% Define element geometry:
h = 250; % height (m)
t = 150; % fault throw (m)
i = [2;5]; % rectangle indices (-)
j = [1;3;4;6]; % triangle indices (-)
theta_deg(j) = [110;80;80;100]; % dip angle (deg)
%theta_deg(j) = [90.01;89.99;89.99;90.01]; % dip angle (deg) %%% test case; near-vertical faults
%theta_deg(j) = [100;80;80;100]; % dip angle (deg) %%% test case; symmetry around y axis
theta(j) = theta_deg(j)*pi/180; % dip angle (rad)
tt = tan(theta);
w(i) = [500;750]; % width, rectangles (m)
% w(i) = [750;750]; % width, rectangles (m) %%% test case; symmetry around y axis
w(j) = abs(h./tt(j)); % width, triangles (m)

% Compute corner point coordinates rectangles (m):
a = (h-t)/2;
b = (h+t)/2;
Dx = a/tan(theta(3)); % x value cornerpoint o (top right corner) element 3 (m)  
p(i) = [-w(2)-w(3)+Dx;w(4)-Dx]; % left
q(i) = [-w(3)+Dx;w(4)+w(5)-Dx]; % right
r(i) = [-b;-a]; % bottom
s(i) = [a;b]; % top

% Compute corner point coordinates and intersections triangles (m):
o(j) = [-w(1)-w(2)-w(3)+Dx; Dx; -Dx; w(4)+w(5)+w(6)-Dx]; % x values corner points opposite ...
    % right-angle corner points
p(j) = [-w(2)-w(3)+Dx;-w(3)+Dx;w(4)-Dx;w(4)+w(5)-Dx]; % x values right-angle corner points
r(j) = [ a; a;-a;-a]; % y values right-angle corner points 
s(j) = [-b;-b; b; b]; % y values corner points above/below right-angle corner points

% Define physical properties:
D_0 = 3500;
g = 9.81;
G = 6500e6;
K_0 = 0.40;
p_0_0 = 35e6;
alpha = 0.9;
beta = 0.9;
mu = 0.50;
nu = 0.20;
p_inc = -30e6;
rho_f = 1020;
rho_s = 2650;
phi = 0.20;

% Define plot parameters:
N = 5000; % number of grid points along y axis

% Compute auxiliary variables:
C = ((1-2*nu)*alpha*p_inc)/(2*pi*(1-nu));

% Compute initial pressures and stresses:
help = linspace(-300,300,N+2)';
y = help(2:end-1); % excludes end points
p_0 = p_0_0 - rho_f*g*(y);
sigma_yy_0 = ( (1-phi)*rho_s + phi*rho_f)*g*(y-D_0); 
sigma_xx_0 = K_0*(sigma_yy_0 + alpha*p_0) - alpha*p_0; 

% Compute incremental stresses:
G_xx = zeros(N,6);
G_xy = zeros(N,6);
G_yy = zeros(N,6);
sigma_xx = zeros(N,3);
sigma_xy = zeros(N,3);
sigma_yy = zeros(N,3);
Sigma_xx = zeros(N,3);
Sigma_xy = zeros(N,3);
Sigma_yy = zeros(N,3);
Sigma_norm = zeros(N,3);
Sigma_norm_eff = zeros(N,3);
Sigma_shear = zeros(N,3);
Sigma_slip = zeros(N,3);
small = 1.e-6; % to stay just to the right of the fault
ct = zeros(1,3);
st = zeros(1,3);
x = zeros(N,3);
DxL = -a/tt(1); % correction factor for element w(1)  
DxR = -a/tt(6); % correction factor for element w(6)  
x(:,1) = -w(1)-w(2)-w(3)+Dx+DxL + y/tt(1) + small; % (m)
x(:,2) = y/tt(4) + small; % (m) 
x(:,3) = w(4)+w(5)+w(6)-Dx-DxR + y/tt(6) - small; % (m) 
for k = [1,2,3]
    for i=[2,5]
        [G_xx(:,i),G_xy(:,i),G_yy(:,i)] = G_rec(x(:,k),y,p(i),q(i),r(i),s(i));
    end
    pos = [4,0,3,1,0,2]; % positions of triangles
    for j=[1,3,4,6]
        [G_xx(:,j),G_xy(:,j),G_yy(:,j)] = ...
            G_tri(x(:,k),y,o(j),p(j),r(j),s(j),theta(j),pos(j));
    end
    switch k
        case 1
            sigma_xx(:,k) = C.* ( sum(G_xx,2) - 2*pi .* (-b < y) .* (y < a) ); % sums rows
            sigma_yy(:,k) = C.* ( sum(G_yy,2) - 2*pi .* (-b < y) .* (y < a) );
        case {2,3}
            sigma_xx(:,k) = C.* ( sum(G_xx,2) - 2*pi .* (-a < y) .* (y < b) );
            sigma_yy(:,k) = C.* ( sum(G_yy,2) - 2*pi .* (-a < y) .* (y < b) );
    end
    sigma_xy(:,k) = C.* sum(G_xy,2);

    % Compute combined stresses:
    Sigma_xx(:,k) = sigma_xx_0 + sigma_xx(:,k);
    Sigma_xy(:,k) = 0 + sigma_xy(:,k); % principal initial stresses oriented along x and y axes
    Sigma_yy(:,k) = sigma_yy_0 + sigma_yy(:,k);

    % Rotate stresses:
    switch k
        case 1
        st(k) = sin(theta(1));
        ct(k) = cos(theta(1));
        case 2
        st(k) = sin(theta(3));
        ct(k) = cos(theta(3));
        case 3
        st(k) = sin(theta(6));
        ct(k) = cos(theta(6));
    end
    Sigma_norm(:,k) =  Sigma_xx(:,k)*(st(k)^2) + Sigma_yy(:,k)*(ct(k)^2) - ...
        2*Sigma_xy(:,k)*st(k)*ct(k);
    Sigma_shear(:,k) = (Sigma_xx(:,k)-Sigma_yy(:,k))*(st(k)*ct(k)) + ...
        Sigma_xy(:,k)*(st(k)^2-ct(k)^2);

    % Compute slip stress:
    switch k
        case 1
            Sigma_norm_eff(:,k) = Sigma_norm(:,k) + ...
                beta * (p_0 + p_inc .* (-b < y) .* (y < a) ); % l.h.s. fault
        case 2
            Sigma_norm_eff(:,k) = Sigma_norm(:,k) + ...
                beta * (p_0 + p_inc .* (-b < y) .* (y < b) ); % central fault
        case 3
            Sigma_norm_eff(:,k) = Sigma_norm(:,k) + ...
                beta * (p_0 + p_inc .* (-a < y) .* (y < b) ); % r.h.s. fault
    end
    Sigma_slip(:,k) = -mu * Sigma_norm_eff(:,k);
end

%----------------------------------------------------------------------------------------------

% Create figures:
black  = [0/255 0/255 0/255];
blue   = [0/255 128/255 255/255];
coal   = [128/255 128/255 128/255];
gray   = [235/255 235/255 235/255];
green  = [50/255 168/255 82/255];
orange = [255/255 185/255 0/255];
purple = [204/255 0/255 204/255];
red    = [255/255 0/255 0/255];

% Depict elements:
x_min = -700;
x_max = 900;
y_min = -800;
y_max = 800;
patch([p(2);q(2);q(2);p(2)],[r(2);r(2);s(2);s(2)],red)
hold on
patch([p(5);q(5);q(5);p(5)],[r(5);r(5);s(5);s(5)],red)
patch([o(1);p(1);p(1)],[r(1);r(1);s(1)],blue)
patch([o(3);p(3);p(3)],[r(3);r(3);s(3)],blue)
patch([o(4);p(4);p(4)],[r(4);r(4);s(4)],blue)
patch([o(6);p(6);p(6)],[r(6);r(6);s(6)],blue)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
hold off
box on
xlabel('$x$ (m)','Interpreter','latex','FontSize',11)
ylabel('$y$ (m)','Interpreter','latex','FontSize',11)
axis([x_min x_max y_min y_max])
axis square

%----------------------------------------------------------------------------------------------

% Plot data:
figure
subplot(1,3,1)
x_min = -100;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-b -b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[a a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
plot(Sigma_xx(:,1)/1e6,y,'-','LineWidth',1.5,'Color',blue) 
plot(Sigma_xy(:,1)/1e6,y,'-','LineWidth',1.5,'Color',green)
plot(Sigma_yy(:,1)/1e6,y,'-','LineWidth',1.5,'Color',orange) 
hold off
ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
axis([x_min x_max y_min y_max])
box on

subplot(1,3,2)
x_min = -100;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-b -b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[-a -a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[a a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[b b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
plot(Sigma_xx(:,2)/1e6,y,'-','LineWidth',1.5,'Color',blue) 
plot(Sigma_xy(:,2)/1e6,y,'-','LineWidth',1.5,'Color',green)
plot(Sigma_yy(:,2)/1e6,y,'-','LineWidth',1.5,'Color',orange)
hold off
axis([x_min x_max y_min y_max])
box on

subplot(1,3,3)
x_min = -100;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-a -a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[b b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
xx = plot(Sigma_xx(:,3)/1e6,y,'-','LineWidth',1.5,'Color',blue); 
xy = plot(Sigma_xy(:,3)/1e6,y,'-','LineWidth',1.5,'Color',green);
yy = plot(Sigma_yy(:,3)/1e6,y,'-','LineWidth',1.5,'Color',orange);
hold off
legend([xx,xy,yy],{'$\Sigma_{xx},$','$\Sigma_{xy},$','$\Sigma_{yy}\;{\rm (MPa)}$'},...
    'Interpreter','latex','FontSize',11,'NumColumns',3,'Position',[0.51 0.034 0.01 0.01],...
    'Box','off')
axis([x_min x_max y_min y_max])
box on

%----------------------------------------------------------------------------------------------

figure
subplot(1,3,1)
x_min = -30;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-b -b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[a a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
plot(Sigma_slip(:,1)/1e6,y,'-','LineWidth',1.5,'Color',black) % right slip threshold
plot(-Sigma_slip(:,1)/1e6,y,'--','LineWidth',1.5,'Color',black) % left slip threshold
plot(Sigma_shear(:,1)/1e6,y,'-','LineWidth',1.5,'Color',red) % shear stress
line([14.5153 30],[-200.06 -200.06],'LineStyle','-','Color',red,'LineWidth',1.5)
hold off
ylabel('$\it y \rm(m)$','Interpreter','latex','FontSize',11)
axis([x_min x_max y_min y_max])
box on

subplot(1,3,2)
x_min = -30;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-b -b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[-a -a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[a a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[b b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
plot(Sigma_slip(:,2)/1e6,y,'-','LineWidth',1.5,'Color',black) % right slip threshold
plot(-Sigma_slip(:,2)/1e6,y,'--','LineWidth',1.5,'Color',black) % left slip threshold
plot(Sigma_shear(:,2)/1e6,y,'-','LineWidth',1.5,'Color',red) % shear stress
line([-30 -19.3322],[200.06 200.06],'LineStyle','-','Color',red,'LineWidth',1.5)
line([-30 -18.506],[-200.06 -200.06],'LineStyle','-','Color',red,'LineWidth',1.5)
hold off
axis([x_min x_max y_min y_max])
box on

subplot(1,3,3)
x_min = -30;
x_max = 30;
y_min = -300;
y_max = 300;
hold on
line([x_min x_max],[-a -a],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[b b],'LineStyle',':','Color',black,'LineWidth',1)
line([x_min x_max],[0 0],'LineStyle','-','Color',black,'LineWidth',0.5)
line([0 0],[y_min y_max],'LineStyle','-','Color',black,'LineWidth',0.5)
tr = plot(Sigma_slip(:,3)/1e6,y,'-','LineWidth',1.5,'Color',black); % right slip threshold
tl = plot(-Sigma_slip(:,3)/1e6,y,'--','LineWidth',1.5,'Color',black); % left slip threshold
ss = plot(Sigma_shear(:,3)/1e6,y,'-','LineWidth',1.5,'Color',red); % shear stress
line([21.642 30],[200.06 200.06],'LineStyle','-','Color',red,'LineWidth',1.5)
hold off
legend([ss,tl,tr],{'$\Sigma_{\parallel},$','$-\Sigma_{sl},$',...
    '$\Sigma_{sl}{\;\rm(MPa)}$'},'Interpreter','latex','FontSize',11,...
    'NumColumns',3,'Position',[0.51 0.034 0.01 0.01],...
    'Box','off')
axis([x_min x_max y_min y_max])
box on

%==============================================================================================

function [G_xx,G_xy,G_yy] = G_rec(x,y,p,q,r,s)
G_xx = atan((y-s)./(x-q)) - atan((y-r)./(x-q)) - atan((y-s)./(x-p)) + atan((y-r)./(x-p)); 
G_yy = atan((x-q)./(y-s)) - atan((x-q)./(y-r)) - atan((x-p)./(y-s)) + atan((x-p)./(y-r)); 
G_xy = 0.5*log( ( ( (x-q).^2+(y-s).^2 ).*( (x-p).^2+(y-r).^2 ) )./...
                ( ( (x-q).^2+(y-r).^2 ).*( (x-p).^2+(y-s).^2 ) ) );
end

%----------------------------------------------------------------------------------------------

function [G_xx,G_xy,G_yy] = G_tri(x,y,o,p,r,s,theta,pos)
switch pos % position of right angle
    case 2 % right angle @ bottom left
        x = -x;
        o = -o;
        p = -p;
        theta = pi-theta;
    case 3 % right angle @ top left
        x = -x;
        y = -y;
        o = -o;
        p = -p;
        r = -r;
        s = -s;       
    case 4 % right angle @ top right
        y = -y;
        r = -r;
        s = -s;       
        theta = pi-theta;
end
y_0 = (p*r-o*s)/(p-o); % y intersections with y axis (m) 
y = y-y_0; % shifts vertical coordinate
r = r-y_0;
s = s-y_0;
st = sin(theta);
ct = cos(theta);
tt = tan(theta);
ctt = cot(theta);
G_xx = atan((y-s)./(x-p)) - atan((y-r)./(x-p)) + ...
       (st*ct/2)*log( ( (x-o).^2+(y-r).^2 ) ./ ( (x-p).^2+(y-s).^2 ) ) - ...
       st^2*( atan( ((x-p)*ctt+y-s)./(x-y*ctt) ) - atan( ((x-o)*ctt+y-r)./(x-y*ctt) ) ); 
G_yy = atan((x-o)./(y-r)) - atan((x-p)./(y-r)) - ...
       (st*ct/2)*log( ( (x-o).^2+(y-r).^2 ) ./ ( (x-p).^2+(y-s).^2 ) ) + ... 
       ct^2*( atan( (x-p+(y-s)*tt)./(y-x*tt) ) - atan( (x-o+(y-r)*tt)./(y-x*tt) ) ); 
G_xy = 0.5     *log( ( (x-p).^2+(y-s).^2 ) ./( (x-p).^2+(y-r).^2 ) ) + ...
       (st^2/2)*log( ( (x-o).^2+(y-r).^2 ) ./( (x-p).^2+(y-s).^2 ) ) + ...
       st*ct*( atan( ((x-p)*ctt+y-s)./(x-y*ctt) ) - atan( ((x-o)*ctt+y-r)./(x-y*ctt) ) );
if pos == 2 || pos == 4  % right angle @ bottom left or top right
    G_xy = - G_xy;
end
end
