from config import *
from FitnessEvaluator import *

import logging
logger = logging.getLogger(__name__)

import uuid

class Candidate:
    def __init__(self, actions=None):
        # unique identifier
        self.unique_id = uuid.uuid4()
        # fitness
        self.previous_fitness = None
        self.fitness = None
        # actions
        self.previous_actions = None
        if actions is not None:
            self.actions = actions
        else:
            # If no actions are provided, randomly choose a number of actions between minimum and maximum_length, ensuring the presence of mandatory actions
            self.actions = random.sample(ACTIONS, k=random.randint(MINIMUM_LENGTH, MAXIMUM_LENGTH - len(ACTIONS_MANDATORY))) + ACTIONS_MANDATORY
            random.shuffle(self.actions)

    def add_action(self, index, action):
        """ Adds the specified action to the candidate's actions at the specified index. """
        if index <= len(self.actions):
            self.actions.insert(index, action)
        else:
            logger.warning(f"Attempted to add action at index {index}, which is out of bounds for {self.actions}.")

    def remove_action(self, index):
        """ Removes an action from the candidate's actions at the specified index. """
        if 0 <= index < len(self.actions):
            self.actions.pop(index)
        else:
            logger.warning(f"Attempted to remove action at index {index}, which is out of range for current actions.")

    def permute_actions(self, index1, index2):
        """ Switches two actions in the candidate's actions at the specified indices. """
        if 0 <= index1 < len(self.actions) and 0 <= index2 < len(self.actions):
            self.actions[index1], self.actions[index2] = self.actions[index2], self.actions[index1]
        else:
            logger.warning(f"Attempted to permute actions at indices {index1} and {index2}, which are out of range for current actions.")

    def apply_mutation_to_candidate(self, mutation_type, action, index, second_index=None):
        # update the previous actions before applying the mutation directly within self.actions
        self.previous_actions = self.actions.copy()

        # apply the mutation based on the mutation type
        if mutation_type == 1:
            # Mutation type 1 requires an action and uses index
            if action is not None:
                self.add_action(index, action)
            else:
                logger.error("Mutation type 1 requires an action to add.")
        elif mutation_type == 2:
            # Mutation type 2 only uses index
            self.remove_action(index)
        elif mutation_type == 3:
            # Mutation type 3 uses index and second_index
            if second_index is not None:
                self.permute_actions(index, second_index)
            else:
                logger.error("Mutation type 3 requires two indices to permute actions.")
        else:
            logger.error(f"Invalid mutation type {mutation_type}.")
    
    def update_fitness(self, new_fitness):
        """ Updates the candidate's fitness with the new value. """
        self.previous_fitness = self.fitness
        self.fitness = new_fitness    
