import importlib
import random
import sys
from typing import List
import numpy as np
import subprocess
import time
import os
import platform
# os.chdir(os.path.dirname(os.path.realpath(__file__)))

###############################################################
# Helper Functions for Setup
###############################################################

def import_parameters(model_name):
    # Base directory where the model subfolder is located
    base_dir = os.path.dirname(__file__)
    overall_folder = os.path.dirname(base_dir)
    model_dir = os.path.join(overall_folder, 'model')
    
    # Ensure the model directory is in sys.path
    if model_dir not in sys.path:
        sys.path.append(model_dir)

    # Create the module name in a format suitable for importlib
    module_name = f"{model_name}_parameters"

    # Full path to the parameter file to check existence
    parameter_file_path = os.path.join(model_dir, f"{module_name}.py")

    # Check if the file exists
    if not os.path.exists(parameter_file_path):
        print(f"File {parameter_file_path} does not exist.")
        user_input = input("Do you want to create the file? (y/n): ")
        if user_input.lower() == 'y':
            print("Creating...")
            try:
                # Use subprocess to call the Python script
                script_path = os.path.join(base_dir, "create_parameters_file.py")
                process  = subprocess.run(['python', script_path, model_name], check=True)

                # Wait until the file is created, with a timeout
                timeout = 30  # Timeout in seconds
                start_time = time.time()
                while not os.path.exists(parameter_file_path):
                    if time.time() - start_time > timeout:
                        print(f"Timeout: File {parameter_file_path} was not created within {timeout} seconds.")
                        process.terminate()
                        return None
                    time.sleep(0.5)

                print(f"File {parameter_file_path} created successfully. Loading parameters...")
            except subprocess.CalledProcessError as e:
                print(f"An error occurred while running the script: {e}")
                exit()
            except Exception as e:
                print(f"An unexpected error occurred: {e}")
                exit()
        else:
            print("Aborting.")
            exit()

    # Import the module dynamically using its module name
    try:
        module = importlib.import_module(module_name)
        # Access the parameters dictionary
        parameters = module.parameters_dict
        return parameters
    except ModuleNotFoundError as e:
        print(f"Failed to import the module: {e}")
        exit()
    except AttributeError as e:
        print(f"Module does not contain 'parameters_dict': {e}")
        exit()

def select_run_parameters(parameters, amount, strategy) -> List:
    '''
    This function selects amount-many parameter settings from the parameter grid from the parameters dictionary
    '''
    parameters_list = []
    for i in range(amount):
        temp_dict = {}
        # go through the keys of the dictionary "parameters"
        for key, value in parameters.items():
            selected_value = None
            # select a value from the list "value" based on the strategy
            if strategy == "random":
                # if value is float, select value itself
                if isinstance(value, float):
                    selected_value = value
                # if value is a list or ndarray, select a random value from the list
                elif isinstance(value, list) or isinstance(value, np.ndarray):
                    selected_value = random.choice(value)
                else:
                    print(f"Value {value} of type {type(value)} not implemented. Exiting...")
                    exit()
            else:
                print(f"Strategy {strategy} not implemented. Exiting...")
                exit()
            # create a new dictionary with the same keys and random values
            temp_dict.update({key: selected_value})
        # append the dictionary to the list
        parameters_list.append(temp_dict)
    return parameters_list

def get_netlogo_home():
    main_py_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
    parent = os.path.join(main_py_dir, "NetLogo-6.1.1")
    grandparent = os.path.abspath(os.path.join(main_py_dir, "..", "NetLogo-6.1.1"))
#    print(f"main_py_dir: {main_py_dir}")
#    print(f"parent: {parent}")
#    print(f"grandparent: {grandparent}")
    if os.path.exists(parent):
#        print(f"Found NetLogo-6.1.1 at {parent}")
        return parent
    elif os.path.exists(grandparent):
#        print(f"Found NetLogo-6.1.1 at {grandparent}")
        return grandparent
    else:
#        print("NetLogo-6.1.1 directory not found.")
        return None
    
def read_target_data(data_name: str):
    # navigate to the data folder
    data_folder = os.path.join(os.path.dirname(__file__), 'data')