import re
import sys
import os

# Change working directory to the 'model' subfolder
current_script_directory = os.path.dirname(os.path.abspath(__file__))
project_root_directory = os.path.dirname(current_script_directory)
model_subfolder_path = os.path.join(project_root_directory, 'model')
os.chdir(model_subfolder_path)

#####################################################################
# Get user argument for file name
# if there is a file name, use it; otherwise, prompt user for file name
file_path = None
model_name = None

while file_path is None:
    if len(sys.argv) > 1 and os.path.exists(f'{sys.argv[1]}.nlogo'):
        file_path = os.path.join(model_subfolder_path, f'{sys.argv[1]}.nlogo')
    else:
        model_name = input("No model given or incomprehensible. Enter the name of the NetLogo model file in the model folder (without '.nlogo'): ")
        print(f'Searching for {model_name}... in {os.getcwd()}')
        if os.path.exists(f'{model_name}.nlogo'):
            file_path = os.path.join(model_subfolder_path, f'{model_name}.nlogo')
        else:
            print("File does not exist. Please try again.")
print(f"Valid file path: {file_path}")

#####################################################################
# Read the model content
model_path = os.path.splitext(os.path.basename(file_path))[0]
with open(file_path, 'r') as file:
    model_content = file.read()

# Parse parameters from the NetLogo model content
def parse_netlogo_parameters(model_content):
    variable_definitions = re.split(r"\n\n+", model_content)
    parameter_space = {}
    
    def parse_and_use_linspace(definition):
        lines = definition.strip().split("\n")
        if "SLIDER" in lines[0]:
            start = float(lines[7].strip())
            stop = float(lines[8].strip())
            step = float(lines[10].strip())
            num_points = int((stop - start) / step + 1)  # calculate number of points for linspace
            linspace_expression = f"np.linspace({start}, {stop}, {num_points})"
            return linspace_expression
        return None
    
    for definition in variable_definitions:
        if "SLIDER" in definition:
            lines = definition.strip().split("\n")
            slider_name = lines[6].strip()
            slider_range = parse_and_use_linspace(definition)
            if slider_range:
                parameter_space[slider_name] = slider_range
        elif "SWITCH" in definition:
            lines = definition.strip().split("\n")
            switch_name = lines[5].strip()
            parameter_space[switch_name] = "[False, True]"
    
    return parameter_space

parameters = parse_netlogo_parameters(model_content)

# Generate Python code from the dictionary
parameters_py = "\n".join([f"\"{key}\" : {value}," for key, value in parameters.items()])

# Save to a Python file
file_name = f"{model_path}_parameters.py"
with open(file_name, 'w') as file:
    file.write("import numpy as np\n")
    file.write("parameters_dict = {\n")
    file.write(parameters_py)
    file.write("\n}")
print(f"Parameters extracted and saved to {file_name}")
