from typing import List

import os
os.chdir(os.path.dirname(os.path.realpath(__file__)))
# os.environ["NUMEXPR_MAX_THREADS"] = "12" # Set the environment variable to use all threads

import sys
sys.path.append(os.path.join(os.path.dirname(__file__), 'src'))

import logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
logging.captureWarnings(True) # Enable capturing of warnings through the logging system
logger = logging.getLogger(__name__)
logger.info("Setting up...")

from src.config import *

# Model Setup
if platform.system() == 'Linux':
    try:
        if NETLOGO_HOME is None:
            raise FileNotFoundError("NetLogo-6.1.1 directory not found.")
        logger.info(f"Detected Linux system and found NetLogo installation at {NETLOGO_HOME}...")
    except FileNotFoundError as e:
        logger.error(e)
        logger.error("Expected NetLogo folder 'NetLogo-6.1.1' in (grand)parent directory above, but it was not found. Exiting...")
        sys.exit(1)
else:
    logger.info("Detected non-Linux system, using default values for netlogo_home...")

logger.info("Model: {}".format(MODEL_NAME))
# ALL_PARAMETERS_DICT_LIST = import_parameters(MODEL_NAME)
# RUN_PARAMETERS_DICT_LIST = select_run_parameters(ALL_PARAMETERS_DICT_LIST, AMOUNT_OF_PARAMETERS, SELECTION_OF_PARAMETERS)
logger.info(f"Selected: {AMOUNT_OF_PARAMETERS} from parameter grid via strategy: {SELECTION_OF_PARAMETERS}")

# Optimisation Setup
if MINIMISATION:
    logger.info("Minimisation problem.")
else:
    logger.info("Maximisation problem.")

logger.info("Features used for training: {}".format(TRAINING_COLUMNS))

logger.info("Mandatory actions at beginning: {}".format(ACTIONS_MANDATORY_BEGIN))
logger.info("Mandatory actions at end: {}".format(ACTIONS_MANDATORY_END))
logger.info("Mandatory actions in candidate: {}".format(ACTIONS_MANDATORY))

# Population Setup
from src.PopulationManager import Population
from src.ParentSelection import *
import random
# The selection strategies need to be adapted to handle multiple fitness values and fitness as list.
SELECTION_STRATEGY = select_all # random.choice([roulette_wheel_selection, tournament_selection, select_better_half_eliminate_worse_half, select_all])

from src.ReproductionStrategy import *
# The reproduction strategies need to be adapted to the previous_fitness because that causes issues in the history 
# batch when subtracting the previous_fitness from the fitness
REPRODUCTION_STRATEGY = no_crossover # random.choice([random_crossover, no_crossover])

from src.RandomMutator import RandomMutator
from src.ForestMutator import ForestMutator
from src.BayesianMutator import BayesianMutator
MUTATOR = RandomMutator() #random.choice([ForestMutator(), BayesianMutator(), RandomMutator()])

logger.info("Setup complete.")

# check if folder "history" exists, if not create it
if RECORDING or MUTATOR._learning:
    os.makedirs('history', exist_ok=True)

def main():
    # Code to initialise the population
    start = datetime.now()
    population = Population(MUTATOR, SELECTION_STRATEGY, REPRODUCTION_STRATEGY, VISUALISATION)
    time = (datetime.now() - start).total_seconds()
    logger.info(f"Took {time:.2f}s to initialise {POPULATION_SIZE} candidates. For {GENERATIONS} generations, expect at least {GENERATIONS * time / 3600:.2f} hours.")

    # Code to run the evolution process
    logger.info("Evolving population...")
    population.evolve_population()

    # Save the history of the evolution

    logger.info("Done!")

if __name__ == "__main__":
    main()