import requests
import json
import os
from pathlib import Path
import base64
import functools
import time
import fig.auth
import re
import datetime

''' Goodies to simplify life with the Figshare API '''

HOST = 'https://data.4tu.nl'
INSTITUTION = 898
INST_NAME = '4tu'
INDEX_DIR = 'C:/DATA/4TURD/python-progs/figshare/index'
group_4TU = 28585
group_other = 28598
groups_Wageningen = (28595,)
groups_Delft = (28586, 28628)
groups_Eindhoven = (28589, 28631)
groups_Twente = (28592, 28634)
groups_Erasmus = (34904,)
groups_Leiden = (34895,)
groups_Maastricht = (34919,)
groups_Nijmegen = (34913,)
groups_UvA = (34901, 35313)
groups_Groningen = (34916,)
groups_Utrecht = (34898,)
groups_IHE = (34910,)
groups_Deltares = (34907,)
groups_NIOZ = (34892,)
groups_4TUmembers = groups_Wageningen + groups_Delft + groups_Eindhoven + groups_Twente
groups_anon = (group_4TU, group_other)
groups_other = (groups_Erasmus + groups_Leiden + groups_Maastricht
                + groups_Nijmegen + groups_UvA + groups_Groningen + groups_Utrecht
                + groups_IHE + groups_Deltares + groups_NIOZ)
groups_non4TU = groups_anon + groups_other
base = 'https://api.figshare.com/v2'
headers = {'Authorization': 'token '+fig.auth.token, 'Content-Type': 'application/json'}
inst_param = {'institution': INSTITUTION}
stats_auth = '4tu_1851:6VK8Tnk8n5'
stats_auth64 = base64.b64encode(stats_auth.encode('ascii')).decode('ascii')
stats_headers = {'Authorization': 'Basic '+stats_auth64, 'Content-Type': 'application/json'}
stats_base = f'https://stats.figshare.com/{INST_NAME}'
PAGE_SIZE = 100
aus_codes = {
    13431:'01', 13432:'0101', 13433:'0102', 13434:'0103', 13435:'0104',
    13436:'0105', 13437:'0199', 13603:'02', 13604:'0201', 13605:'0202',
    13606:'0203', 13607:'0204', 13608:'0205', 13609:'0206', 13610:'0299',
    13594:'03', 13595:'0301', 13596:'0302', 13597:'0303', 13598:'0304',
    13599:'0305', 13600:'0306', 13601:'0307', 13602:'0399', 13551:'04',
    13552:'0401', 13553:'0402', 13554:'0403', 13555:'0404', 13556:'0405',
    13557:'0406', 13558:'0499', 13410:'05', 13411:'0501', 13412:'0502',
    13413:'0503', 13414:'0599', 13578:'06', 13579:'0601', 13580:'0602',
    13581:'0603', 13582:'0604', 13583:'0605', 13584:'0606', 13585:'0607',
    13586:'0608', 13587:'0699', 13376:'07', 13377:'0701', 13378:'0702',
    13379:'0703', 13380:'0704', 13381:'0705', 13382:'0706', 13383:'0707',
    13384:'0799', 13611:'08', 13612:'0801', 13613:'0802', 13614:'0803',
    13615:'0804', 13616:'0805', 13617:'0806', 13618:'0807', 13619:'0899',
    13630:'09', 13631:'0901', 13632:'0902', 13633:'0903', 13634:'0904',
    13635:'0905', 13636:'0906', 13637:'0907', 13638:'0908', 13639:'0909',
    13640:'0910', 13641:'0911', 13642:'0912', 13643:'0913', 13644:'0914',
    13645:'0915', 13646:'0999', 13652:'10', 13653:'1001', 13654:'1002',
    13655:'1003', 13656:'1004', 13657:'1005', 13658:'1006', 13659:'1007',
    13660:'1099', 13474:'11', 13475:'1101', 13476:'1102', 13477:'1103',
    13478:'1104', 13479:'1105', 13480:'1106', 13481:'1107', 13482:'1108',
    13483:'1109', 13484:'1110', 13485:'1111', 13486:'1112', 13487:'1113',
    13488:'1114', 13489:'1115', 13490:'1116', 13491:'1117', 13492:'1199',
    13362:'12', 13363:'1201', 13364:'1202', 13365:'1203', 13366:'1204',
    13367:'1205', 13368:'1299', 13647:'13', 13648:'1301', 13649:'1302',
    13650:'1303', 13651:'1399', 13566:'14', 13567:'1401', 13568:'1402',
    13569:'1403', 13570:'1499', 13500:'15', 13501:'1501', 13502:'1502',
    13503:'1503', 13504:'1504', 13505:'1505', 13506:'1506', 13507:'1507',
    13508:'1599', 13464:'16', 13465:'1601', 13466:'1602', 13467:'1603',
    13468:'1604', 13469:'1605', 13470:'1606', 13471:'1607', 13472:'1608',
    13473:'1699', 13453:'17', 13454:'1701', 13455:'1702', 13456:'1799',
    13427:'18', 13428:'1801', 13430:'1899', 13559:'19', 13560:'1901',
    13561:'1902', 13562:'1903', 13563:'1904', 13564:'1905', 13565:'1999',
    13517:'20', 13518:'2001', 13519:'2002', 13520:'2003', 13521:'2004',
    13522:'2005', 13523:'2099', 13448:'21', 13449:'2101', 13450:'2102',
    13451:'2103', 13452:'2199', 13588:'22', 13589:'2201', 13590:'2202',
    13591:'2203', 13592:'2204', 13593:'2299', 13360:'81', 13361:'8101',
    13401:'82', 13402:'8201', 13403:'8202', 13404:'8203', 13405:'8204',
    13406:'8205', 13407:'8206', 13408:'8298', 13409:'8299', 13440:'83',
    13441:'8301', 13442:'8302', 13443:'8303', 13444:'8304', 13445:'8305',
    13446:'8398', 13447:'8399', 13421:'84', 13422:'8401', 13423:'8402',
    13424:'8403', 13425:'8498', 13426:'8499', 13620:'85', 13621:'8501',
    13622:'8502', 13623:'8503', 13624:'8504', 13625:'8505', 13626:'8506',
    13627:'8507', 13628:'8598', 13629:'8599', 13524:'86', 13525:'8601',
    13526:'8602', 13527:'8603', 13528:'8604', 13529:'8605', 13530:'8606',
    13531:'8607', 13532:'8608', 13533:'8609', 13534:'8610', 13535:'8611',
    13536:'8612', 13537:'8613', 13538:'8614', 13539:'8615', 13540:'8616',
    13541:'8617', 13542:'8698', 13543:'8699', 13509:'87', 13510:'8701',
    13511:'8702', 13512:'8703', 13513:'8704', 13514:'8705', 13515:'8798',
    13516:'8799', 13661:'88', 13662:'8801', 13663:'8802', 13664:'8803',
    13665:'8898', 13666:'8899', 13544:'89', 13545:'8901', 13546:'8902',
    13547:'8903', 13548:'8904', 13549:'8998', 13550:'8999', 13457:'90',
    13458:'9001', 13459:'9002', 13460:'9003', 13461:'9004', 13462:'9098',
    13463:'9099', 13667:'91', 13668:'9101', 13669:'9102', 13670:'9103',
    13671:'9104', 13672:'9105', 13673:'9199', 13415:'92', 13416:'9201',
    13417:'9202', 13418:'9204', 13419:'9205', 13420:'9299', 13571:'93',
    13572:'9301', 13573:'9302', 13574:'9303', 13575:'9304', 13576:'9305',
    13577:'9399', 13369:'94', 13370:'9401', 13371:'9402', 13372:'9403',
    13373:'9404', 13374:'9405', 13375:'9499', 13493:'95', 13494:'9501',
    13495:'9502', 13496:'9503', 13497:'9504', 13498:'9505', 13499:'9599',
    13385:'96', 13386:'9601', 13387:'9602', 13388:'9603', 13389:'9604',
    13390:'9605', 13391:'9606', 13392:'9607', 13393:'9608', 13394:'9609',
    13395:'9610', 13396:'9611', 13397:'9612', 13398:'9613', 13399:'9614',
    13400:'9699', 13438:'97', 13439:'9701'
}

def UNITS(binary = False):
    uDec = {'T':1000000000000, 'G':1000000000, 'M':1000000, 'K':1000, '': 1}
    uBin = {'T':1099511627776, 'G':1073741824, 'M':1048576, 'K':1024, '': 1}
    return uBin if binary else uDec

def unitsToSize(size, char, binary = False):
    units = UNITS(binary)
    if (char in units):
        size *= units[char]
    return int(size)

def sizeToUnits(size, binary = False):
    units = UNITS(binary)
    for (char, unitSize) in units.items():
        fsize = size/unitSize
        if fsize > 1:
            break
    return (fsize, char)

def sizeToString(size, sigDigits = 3, binary = False):
    size, char = sizeToUnits(size, binary = binary)
    n = max(sigDigits - len(str(int(size))),0) if char else 0
    i = 'i' if binary and char else ''
    return f'{size:.{n}f} {char}{i}B'

def sizeToStrings(size, sigDigits = 3):
    strings = [sizeToString(size, sigDigits, b) for b in (0,1)]
    return f'{strings[0]} ({strings[1]})'

def sget (action, params=None):
    ''' get stats with json response '''
    retry = True
    while retry:
        try:
            resp = requests.get(stats_base+action, headers=stats_headers, params=params)
            retry = False
        except:
            print('Error, retry')
    return jreturn(resp)

def askIf(label, default=False):
    dic = {'n':False, 'y':True}
    inp = input(f"{label} [{('y/N', 'Y/n')[default]}] ").lower()
    return dic[inp] if inp in dic else default

def timer(func):
    """Print the runtime of the decorated function"""
    @functools.wraps(func)
    def wrapper_timer(*args, **kwargs):
        start_time = time.perf_counter()
        value = func(*args, **kwargs)
        end_time = time.perf_counter()
        run_time = end_time - start_time
        print(f"Finished {func.__name__!r} in {run_time:.4f} secs")
        return value
    return wrapper_timer

def isPerson(name):
    ''' Check if a name in Contributors is personal '''
    C = ['Ab', 'Ad', 'Adr', 'Af', 'Ald', 'Alf', 'Ali', 'Ams', 'An', 'Anim', 'Ari', 'Atm', 'Ba', 'Bei', 'Ben', 'Bio', 'Birr', 'Blu', 'Blue-',
         'Brem', 'Bren', 'Bu', 'Bud', 'Bun', 'C', 'CA', 'Ca', 'Car', 'Cas', 'Ce', 'Ces', 'Ch', 'Che', 'Chengd', 'Cho', 'Chu', 'Cle', 'Cli',
         'Coe', 'Col', 'Cr', 'Cu', 'Cá', 'DE', 'Da', 'Dal', 'Dan', 'De Vries', 'De W', 'Del', 'Dell', 'Delt', 'Der', 'Des', 'Dig', 'Dip',
         'Dir', 'Div', 'Do', 'Don', 'Dou', 'Dut', 'D’', 'E', 'Em', 'Env', 'Ep', 'Eq', 'Erh', 'Eu', 'F', 'FA', 'Fern', 'Fil', 'Fis', 'Fl',
         'Font', 'Fr', 'Fri', 'Fried', 'G', 'GE', 'Ga', 'Gaz', 'Ge', 'Gen', 'Ger', 'Gh', 'Gi', 'Grad', 'Gro', 'Gron', 'Grov', 'Gu', 'Guim',
         'H', 'Ha', 'Hanz', 'Har', 'Harv', 'Has', 'Heb', 'Hec', 'Her', 'Herm', 'Hoges', 'Hoi', 'Hon', 'Hoo', 'Hub', 'Hum', 'Hun', 'Hö', 'I',
         'Ia', 'If', 'Il', 'Im', 'J', 'Jaco', 'Jacobs,', 'Ji', 'Jin', 'Jinl', 'Jo', 'Johanne', 'Jon', 'K', 'Ka', 'Kav', 'Ke', 'Key', 'Ki',
         'Kin', 'King,', 'Kipp', 'Kl', 'Kon', 'Koo', 'Kor', 'Kort', 'Ky', 'L', 'Labo', 'Lam', 'Leed', 'Lem', 'Liv', 'Lo', 'Lom', 'Lou',
         'Ludw', 'Ludwig,', 'MA', 'Ma', 'Mac', 'Mad', 'Mas', 'Masi', 'Mass', 'Mat', 'Max', 'Mc', 'Me', 'Mee', 'Met', 'Mez', 'Mi', 'Mid',
         'Min', 'Mo', 'Mon', 'Monf', 'Mont', 'Moo', 'Mos', 'Moss', 'N', 'Nee', 'Net', 'Ng', 'Nik', 'Nin', 'Ning', 'Oa', 'Op', 'Orr', 'PA',
         'Pa', 'Pe', 'Pet', 'Philips', 'Pi', 'Pon', 'Pop', 'Pot', 'Pou', 'Pu', 'Pé', 'Q', 'Qi', 'Qu', 'Qui', 'RN', 'Ra', 'Rab', 'Ram', 'Res',
         'Rh', 'Rij', 'Rin', 'Rott', 'Rust', 'Rz', 'S', 'Sa', 'Sac', 'Sap', 'Sav', 'Schoo', 'Schö', 'Se', 'Sed', 'Sh', 'Shi', 'Shih', 'Shu',
         'Si', 'Sick', 'Simu', 'Sir', 'Sm', 'Smi', 'Sof', 'Son', 'Sou', 'Sp', 'Spac', 'Spe', 'St', 'Sta', 'Stan', 'Ste', 'Sti', 'Stiv', 'Sun',
         'Sun,', 'Sw', 'Sz', 'T', 'Ta', 'Tal', 'Tap', 'Te', 'Teck', 'Th', 'Ther', 'Thir', 'Ti', 'Tianj', 'Tim', 'Toms', 'Ton', 'Tong', 'Tonn',
         'Tur', 'Tut', 'U', 'Uc', 'Un', 'Ur', 'Ut', 'Va', 'Van H', 'Van K', 'Van O', 'Van R', 'Verbu', 'Verh', 'Vien', 'Vis', 'Vol', 'Voo',
         'Vrij', 'Vrijl', 'Vy', 'W', 'Wag', 'Wal', 'Wars', 'We', 'Wes', 'Westerw', 'Wet', 'Wi', 'Wil', 'Win', 'Witteveen +', 'Wo', 'Wor', 'X',
         'Yı', 'Z', 'Zhe', 'Zi', 'Zie', 'Zij', 'da', 'de', 'f', 'v']
    return bool(((name in C) + sorted(C + [name]).index(name))%2)

def printDict(adict, indent='  ', sep='  ', keyLen=None):
    n = keyLen if keyLen else max([len(k) for k in adict.keys()])
    for key, val in adict.items():
        print(f'{indent}{key:{n}}{sep}{val}')

def jprint (jobj, sort_keys=False, indent=4):
    ''' pretty print of json object with some nice defaults '''
    print (json.dumps(jobj, sort_keys=sort_keys, indent=indent))

def jdump(fname, obj, **kwargs):
    with open(fname, 'w', encoding='utf-8') as f:
        json.dump(obj, f, **kwargs)

def jload(fname):
    with open(fname, 'r', encoding='utf-8') as f:
        return json.load(f)

def jreturn(resp):
    ''' return resp as json object '''
    if resp.ok:
        return resp.json()
    else:
        print (resp.status_code, resp.reason, resp.content)
        return []

def jget (action, params=None):
    ''' simple get with json response '''
    retry = True
    while retry:
        try:
            resp = requests.get(base+action, headers=headers, params=params)
            retry = False
        except:
            print('Error, retry')
    return jreturn(resp)

def jpost (action, data):
    ''' simple post with dict data and json response '''
    retry = True
    while retry:
        try:
            resp = requests.post(base+action, headers=headers, data=json.dumps(data))
            retry = False
        except:
            print('Error, retry')
    return jreturn(resp)

def jput (action, data):
    ''' simple put with dict data and json response '''
    retry = True
    while retry:
        try:
            resp = requests.put(base+action, headers=headers, data=json.dumps(data))
            retry = False
        except:
            print('Error, retry')
    return resp

def jgetAll(action, params=None):
    ''' does the paging for you with GET and fixes self-impersonate '''
    return jdoAll('get', action, params)

def jpostAll(action, data):
    ''' does the paging for you with POST and fixes self-impersonate (untested) '''
    return jdoAll('post', action, data)

def jdoAll(method, action, params=None):
    ''' used for jgetAll en jpostAll '''
    par = {} if params == None else dict(params)
    par['page'] = 1
    if 'page_size' not in par:
        par['page_size'] = PAGE_SIZE
    if 'institution' not in par:
        par['institution'] = INSTITUTION #workaround for latest bug
    if 'impersonate' in par:
        if par['impersonate'] == fig.auth.me:
            par.pop('impersonate')
            print("it's me!", end=' ')
    page_size = par['page_size']
    last_page_size = page_size
    records = []
    do = jpost if method == 'post' else jget
    while last_page_size == page_size:
        new = do(action, par)
        records.extend(new)
        last_page_size = len(new)
        par['page'] += 1
        print('+',end='')
    print(len(records))
    return records

def getItem(pid, account=None, Type='article', version=None):
    ''' returns public or private version depending on presence of account '''
    allowed_types = ('article','collection')
    if not(Type in allowed_types):
        print(f'Wrong type {Type}. Allowed: {allowed_types}')
        return {}
    elif not(account):
        action = f'/{Type}s/{pid}'
        if version:
            action += f'/versions/{version}'
        return jget(action)
    else:
        params = {} if account == fig.auth.me else {'impersonate':account}
        return jget(f'/account/{Type}s/{pid}', params=params)

def getPrivateItem(pid=None, doi=None, Type='article'):
    ''' like getItem with built-in lookup of account '''
    jresp = {}
    try:
        if pid:
            jresp = getItem(pid, pidIndex()[pid], Type=Type)
        elif doi:
            jresp = getItem(*doiIndex()[doi], Type=Type)
        else:
            print('pid or doi required')
    except KeyError:
        print(f'Key error in getPrivateItem({pid},{doi},{Type})')
    return jresp

def makePublicIndex(dir = INDEX_DIR):
    Types = ['article', 'collection']
    for Type in Types:
        itemVersions = []
        data = []
        items = jgetAll(f'/{Type}s')
        n = len(items)
        print(f'{Type}s {n}')
        for i, item in enumerate(items):
            print(Type, i+1,'/',n)
            pid = item['id']
            versionDescrs = jget(f'/{Type}s/{pid}/versions')
            versions = [int(v['url'].split('/')[-1]) for v in versionDescrs]
            artVs = [(v, getSmoothItem(pid, Type, v)) for v in versions]
            data.append((pid, artVs))
        jdump(f'{dir}/{Type}s.json', data)
        print(f'{Type}s ready')
    
def makeIndex(dir=INDEX_DIR, full=False):
    '''
    Make json files to be used in scripts (ca 15 mins, or 2 hours for full):
    - list of metadata of accounts
    - list of metadata of accounts + articles/collections
        if full, complete article/collection metadata
    - dict of pid -> account
    - dict of doi -> pid, account
    '''
    Path(dir).mkdir(parents=True, exist_ok=True)
    print('accounts:', end=' ')
    All = jgetAll('/account/institution/accounts')
    jdump(dir+'/accounts.json', All)
    pid_index = {}
    doi_index = {}
    ALL = []
    for i, acc in enumerate(All):
        id = acc['id']
        imp = {'impersonate': id}
        acc_f = jget('/account', {} if id == fig.auth.me else imp)
        print(f'{i+1}: account {id}')
        for what in ('article','collection'):
            whats = what + 's'
            print(f'  {whats:<12}:', end=' ')
            items = jgetAll(f'/account/{whats}', imp)
            for item in items:
                pid = item['id']
                doi = item['doi']
                pid_index[pid] = id
                if doi:
                    doi_index[doi] = (pid, id)
            if full:
                items_full = []
                doPr = len(items) >= PAGE_SIZE
                if doPr:
                    print(16*' ', end='^')
                for i, item in enumerate(items):
                    if (i+1) % PAGE_SIZE == 0:
                        print('^', end='')
                    pid = item['id']
                    items_full.append(smooth(compactItem(getItem(pid, id, Type=what))))
                if doPr:
                    print()
                acc_f[whats] = items_full
            else:
                acc_f[whats] = items
        ALL.append(acc_f)
    jdump(dir+'/accounts_items.json', ALL)
    if full:
        jdump(dir+'/accounts_items_full.json', ALL)
    jdump(dir+'/pid_index.json',dict(sorted(pid_index.items())))
    jdump(dir+'/doi_index.json',dict(sorted(doi_index.items())))

def index(Type, dir):
    ''' used for pidIndex() and doiIndex()'''
    fname = f'{dir}/{Type}_index.json'
    if not(os.path.isfile(fname)):
        print('No index found. Make index...')
        makeIndex(dir)
        print('Index ready!')
    return jload(fname)

def doiIndex(dir=INDEX_DIR):
    ''' dict of doi -> pid,account, created with makeIndex '''
    return index('doi', dir)

def pidIndex(dir=INDEX_DIR):
    ''' dict of pid -> account, created with makeIndex '''
    ind = index('pid', dir)
    return {int(i):ind[i] for i in ind}

def decustomize(aList):
    ''' returns custom metadata formatted as standard metadata '''
    return {x['name']:x['value'] for x in aList}

def compact(obj):
    ''' eliminate from construct of dicts and lists all values x for which bool(x)==False '''
    cobj = obj
    if isinstance(obj, dict):
        co = {x:compact(obj[x]) for x in obj if obj[x]}
        cobj = {x:co[x] for x in co if co[x]}
    elif isinstance(obj, list):
        co = [compact(x) for x in obj if x]
        cobj = [x for x in co if x]
    if cobj:
        return cobj

def decustomizeItem(item):
    Item = item.copy()
    if 'custom_fields' in Item:
        Item['custom_fields'] = decustomize(Item['custom_fields'])
    return Item

def compactItem(item):
    return compact(decustomizeItem(item))
    
def getCompactItem(pid, Type='article', version=None):
    return compact(decustomizeItem(getItem(pid, Type=Type, version=version)))

def sanitize(adict):
    new_dict = adict.copy()
    key = 'impersonate'
    if key in new_dict and new_dict[key] in (None, fig.auth.me):
        del new_dict[key]
    return new_dict

def getCollections(since = None):
    params = {'page_size':1000}
    groups = jget('/account/institution/groups', params)
    collections = []
    if since:
        params['published_since'] = since
    for group in groups:
        id = group['id']
        name = group['name']
        params['group'] = id
        group_colls = jget('/collections', params)
        print(f'{id:<8}{name:<46}{len(group_colls):>8}')
        collections += group_colls
    print(f'{len(collections):>62}')
    return collections

def getOrgs(compactItem):
    custom = compactItem['custom_fields']
    sep = r'\s*[;\n]\s*'
    orgs = []
    if 'Organizations' in custom:
        orgs = [c for c in re.split(sep, custom['Organizations']) if c !='']
    elif 'Contributors' in custom:
        orgs = [c for c in re.split(sep, custom['Contributors']) if not isPerson(c)]
    return orgs

def dictNameOrcid(text):
    ''' transforms contributor string to Fig API creators style dict '''
    ret = {'name': text}
    if text.endswith(']'):
        parts = text[:-1].split(' [orcid:', 1)
        if parts[1:]:
            ret = {'name': parts[0], 'orcid_id': parts[1]}
    return ret

def stringNameOrcid(dic):
    ''' does the reverse of dictNameOrcid() '''
    ret = dic['name']
    if 'orcid_id' in dic:
        ret += f" [orcid:{dic['orcid_id']}]"
    return ret

def smooth(compactItem, splitContrib = True):
    ''' sanitizes contributors and orgainizations '''
    item = compactItem.copy()
    custom = item.pop('custom_fields', {})
    pers = []
    orgs = []
    orgs_raw = custom.pop('Organizations', None)
    cont_raw = custom.pop('Contributors', None)
    metadata_version = 1
    if orgs_raw:
        orgs = [x for x in re.split(r'\s*[;\n]\s*', orgs_raw) if x != '']
    if cont_raw:
        metadata_version = 0
        contribs = cont_raw.split(';\n')
        pers = [c for c in contribs if isPerson(c)]
        orgs = [c for c in contribs if not c in pers]
    custom['metadata_version'] = metadata_version
    if orgs:
        custom['Organizations'] = orgs
    if pers:
        custom['Contributors'] = [dictNameOrcid(p) for p in pers] if splitContrib else pers
    item['custom_fields'] = custom #EDIT
    return item

def getSmoothItem(pid, Type='article', version=None, splitContrib = True):
    return smooth(getCompactItem(pid, Type=Type, version=version), splitContrib=splitContrib)

def addStats(item, Type='article', start_date=None, end_date=None, counters=('views', 'downloads', 'shares'), granularity='month'):
    '''
        item may be any dict containing key "id"
        stats are added in place, no return value
        dates are isoform strings
    '''
    pid = item['id']
    if Type == 'auto':
        Type = 'collection' if pid < 10000000 else 'article'
    params = {}
    if start_date:
        params['start_date'] = start_date
    if end_date:
        params['end_date'] = end_date
    counts = {counter: 0 for counter in counters}
    timeline = {}
    for counter in counters:
        resp = sget(f'/timeline/{granularity}/{counter}/{Type}/{pid}', params = params)
        if resp:
            tl = resp['timeline']
            if tl:
                timeline[counter] = dict(sorted(tl.items()))
    item['usage'] = timeline

def statsList(items, start_date=None, end_date=None, counters=('views', 'downloads', 'shares'), granularity='month'):
    '''
        Returns list of records with added usage statistics per record and a summary with totals
    '''
    records = items.copy()
    totals = {}
    n = len(records)
    for i, record in enumerate(records):
        print(f'adding stats for record {i+1} / {n}')
        Type = ('article', 'collection')['articles_count' in record]
        addStats(record, Type=Type, start_date=start_date, end_date=end_date, counters=counters, granularity=granularity)
    for counter in counters:
        tot = {}
        for record in records:
            usage = record['usage']
            if counter in usage:
                counts = usage[counter]
                for Date, count in counts.items():
                    if Date in tot:
                        tot[Date] += count
                    else:
                        tot[Date] = count
        totals[counter] = dict(sorted(tot.items()))
    usage = {'date_start': start_date,
             'date_end': end_date,
             'date_scan': datetime.date.today().isoformat(),
             'totals': totals}
    result = {'usage_summary': usage, 'records': records}
    return result

def padDates(dates=[], start=None, end=None):
    ''' Pad list of dates in Y, M or D granularity '''
    dlen = len(dates[0] if dates else start)
    ret = dates
    try:
        if not start:
            start = min(dates)
        if not end:
            end = max(dates)
        if dlen == 4:
            ret = [f'{i}' for i in range(int(start), int(end) + 1)]
        elif dlen == 7:
            y, m = [int(x) for x in start.split('-')]
            nStart = 12*y + m
            y, m = [int(x) for x in end.split('-')]
            nEnd = 12*y + m
            ret = [f'{i//12}-{i%12 + 1:02d}' for i in range(nStart - 1, nEnd)]
        elif dlen == 10:
            dStart = datetime.date.fromisoformat(start)
            dEnd = datetime.date.fromisoformat(end)
            ret = [(dStart + datetime.timedelta(days=i)).isoformat() for i in range((dEnd - dStart).days + 1)]
        else:
            print(f'ERROR padding dates: wrong date length {dlen}')
    except:
        print('ERROR padding dates')
    return ret

def padDatesDict(datesDict={}, start=None, end=None, default=0):
    ''' Pad dict with keys as in padDates(), new keys get with default value '''
    return {**{x: default for x in padDates(list(datesDict.keys()), start, end)}, **datesDict}

#TEST
if __name__ == '__main__':
    pid = 17185607
    acc = pidIndex()[pid]
    art = getItem(pid, acc)
    print(acc, pid)
    jprint(compactItem(art))
