#ifndef DATACOLLECTOR_HH
#define DATACOLLECTOR_HH

#include <iostream>
#include <fstream>
#include <gecode/int.hh>
#include <string>
#include <gecode/brancher/ml-tree.cpp>
#include <gecode/brancher/ml-node.cpp>
#pragma push_macro("slots")
#undef slots
#include "Python.h"
#pragma pop_macro("slots")

using namespace Gecode;
using namespace std;

// Singleton data collector to collect data from search nodes in order to compute node scores based on
// domain size, variable choice, and value assigned to the variable.
class DataCollector {

    public:
        // Retrieves the Singleton instance
        static DataCollector* getInstance();
        // Store information in the Machine learning node
        void storeMLNode(vector<int> assigned, int domain_size, long degree, int var_order, int pos, int val, 
                         int val_pos, int min_val_dom, int max_val_dom, int regret_min, int regret_max);
        // Use constructed trees to compute deep impact (naive implementation for now). Depth denotes how deep the impact is computed in the tree
        void computeDeepImpact(int depth);
        void computeDeepSmallest(int depth);
        void computeDeepMaxRegret(int depth);
        void computeDeepAntiFF(int depth);
        // Send all data to a python script to use Machine Learning
        int runML();
        // Tests calling python scripts with a simple sum function
        int testPythonSum(int a, int b);
        // Predict a score based on ml model
        vector<double> predictML(vector<int> dom_sizes, vector<long> degrees, vector<int> var_orders, vector<int> var_posses, vector<int> values, 
                                 vector<int> val_posses, vector<int> min_vals, vector<int> max_vals, vector<int> regret_min_vals, vector<int> regret_max_vals);
        // Finalizen python calls
        int finalizeML();
        vector<int> getRootDomain();
        void removeChecked(int checked);
        void setRootDomain(ViewArray<Int::IntView> y);
        // Experimental data
        void init();
        // Get number of predictions done given an ML model
        int getNumberOfPredictions();
        // Get number of predictions done given an ML model
        int getNumberOfGetNodeCalls();
        void incrementTiebreaks();
        int getNumberOfTiebreaks();
        void incrementNodesSearched();
        int getNodesSearched();
        
        // Var ordering 
        void addVar(int var);
        vector<int> varsSelected();
        void addVarOrdering(vector<int> var_order);
        vector<vector<int>> getVarOrderings();
        void resetVarOrderings();
        vector<int> getRootVars();
        void removeCheckedVars(int checked);
        void setRootVars(int size);

        // Heuristic function
        string getFunction();
        void setFunction(string f);

        // Get pointers to the trees that were generated
        vector<MLTree*> getTrees();

    private:
        // Private constructor to prevent creating instances
        DataCollector();
        // Heuristic function
        string func;
        // Checks whether the asset id is already in the data collection and adds it if it is not
        int computeAssetId(int asset_id);
        // A python object which stores an ML model learned with the runSVR function
        PyObject *p_ml_model;
        PyObject *p_ml_scaler;
        // The Singleton instance
        static DataCollector* instance;
        // Each asset represents it's own collection of data (and is typically a different search tree)
        vector<int> assets;

        // Data storage, a vector for each asset/search tree
        vector<MLTree*> trees;  

        // Probing
        vector<int> root_values;

        // Restart-based heuristic search
        vector<int> root_vars;

        // Var ordering hashmap
        vector<vector<int>> var_orders;
        vector<int> var_order_indices;

        // Experimental data
        int number_of_predictions;
        int number_of_get_node_calls;
        int tiebreaks;
        int nodes_searched;
};

#endif