import cv2
import numpy as np
import utils


class Kamel_Zhao:

    def __init__(self, image):
        """
        Initialization of the Kamel_Zhao class, that thresholds using the method described by Kamel and Zhao (1993).
        :param image: the image to be thresholded.
        """
        self.image = image.astype(np.int64)
        self.image = np.abs(image - 255)
        self.image = image / 255

    def binarize(self, SW=5, T=-0.03):
        """
        Implementation of thresholding as explained byKamel and Zhao (1993).
        :param SW: the stroke width that determines the size of the window.
        :param T: the threshold that determines if a pixel is considered black or white according to this algorithm.
        :return: the binarized image. Note that the size of the returned image is different from the input,
        since the binarized image is bordered with (2 * SW + 1) black pixels. These are removed in the returned image.
        """

        bin_image = np.zeros(self.image.shape)

        for x in range((2 * SW + 1), self.image.shape[0] - (2 * SW + 1)):
            for y in range((2 * SW + 1), self.image.shape[1] - (2 * SW + 1)):
                # Calculates the boolean value for each P surrounding the central pixel
                L_P0 = self._calculate_LP(x, y, x, y - SW, SW, T)
                L_P1 = self._calculate_LP(x, y, x - SW, y - SW, SW, T)
                L_P2 = self._calculate_LP(x, y, x - SW, y, SW, T)
                L_P3 = self._calculate_LP(x, y, x - SW, y + SW, SW, T)
                L_P4 = self._calculate_LP(x, y, x, y + SW, SW, T)
                L_P5 = self._calculate_LP(x, y, x + SW, y + SW, SW, T)
                L_P6 = self._calculate_LP(x, y, x + SW, y, SW, T)
                L_P7 = self._calculate_LP(x, y, x + SW, y - SW, SW, T)

                # Applies the logic calculation to determine the pixel value
                if ((L_P0 and L_P4) or (L_P2 and L_P6)) and ((L_P1 and L_P5) or (L_P3 and L_P7)):
                    bin_image[x, y] = 255

        # Kamel crops, so it has to be padded with zeros to fit the ground truth
        crop_edges = bin_image[(2 * SW + 1): bin_image.shape[0] - (2 * SW + 1),
                         (2 * SW + 1): bin_image.shape[1] - (2 * SW + 1)]
        return np.pad(crop_edges, (2 * SW + 1), utils.pad_with, padder=255)

    def _calculate_LP(self, x, y, p_x, p_y, SW, T):
        """
        Calculates the average for a P value that surrounds (x,y) and determines whether the P is above
        or below the threshold.
        :param x: the x coordinate of the considered pixel.
        :param y: the y coordinate of the considered pixel.
        :param p_x: the x coordinate of the pixel P.
        :param p_y: the y coordinate of the pixel P.
        :param SW: the stroke width that determines the size of the window.
        :param T: the threshold that determines if a pixel is considered black or white according to this algorithm.
        """

        avg = 0
        for i in range(-SW, SW + 1):
            for j in range(-SW, SW + 1):
                avg += self.image[p_x - i, p_y - j]
        avg = avg / ((2 * SW + 1) ** 2)

        return (avg - self.image[x, y]) > T
