import numpy as np


class Kavallieratou:

    def __init__(self, image):
        """
        Initialization of the Kavallieratou class, that thresholds using the method described by Kavallieratou (2005).
        :param image: the image to be thresholded.
        """
        self.image = image

    def binarize(self, max_iterations=999999):
        """
        Implementation of thresholding as explained in Kavallieratou (2005).
        :param max_iterations: the maximum number of iterations that the algorithm should perform. By default it is set
        very high, so essentially no limit.
        :return: the binarized image.
        """
        # This implementation assumes that black pixels are the minority and that the range of the image is [0, 1].
        image = np.abs(self.image - 255)
        image = image / 255
        # Calculates T_(i-1) and T_i, the previous and current image averages.
        T_previous = 0
        T = np.average(image)

        # Goes through each iteration, calculates the average of the image and subtracts it from the image. The image is
        # then stretched such that the histogram fits the whole range. Iterations are repeated until the change is
        # negligible or the maximum number of iterations is reached.
        iterations = 0
        while np.abs(T - T_previous) > 0.1 and iterations < max_iterations:
            image_s = 1 - T + image
            image = np.clip((1 - ((1 - image_s) / (1 - np.min(image_s)))), 0, 1)
            T_previous = T
            T = np.average(image)
            iterations += 1

        # Everywhere that isn't a 1 gets turned into a zero.
        image = np.where(image != 1, 0, 1)
        # Recasts to the rance [0, 255]
        image = image * 255

        return image, iterations
