import os
import sklearn.metrics as Metrics
import numpy as np
import cv2
import csv
import utils
from hybrid_methods.Rao_et_al import Rao_et_al
from local_methods.Bolan_Su import Bolan_Su
from local_methods.Kamel_Zhao import Kamel_Zhao
from global_methods.Mello_Costa import Mello_Costa
from global_methods.Otsu import Otsu
from local_methods.Niblack import Niblack
from local_methods.Lantink import Lantink
from local_methods.Gatos import Gatos
from local_methods.SoftwareProject import SoftwareProject

"""
This file contains code for all of the evaluation metrics. When run as main, this file goes through all
of the relevant algorithms and all of the images at a certain path, applies all algorithms to each image,
and applies the evaluation metrics to them. 
"""

def F_1(result, ground_truth):
    return Metrics.f1_score(ground_truth.flatten(), result.flatten(), pos_label=255)


def PSNR(result, ground_truth):
    MSE = Metrics.mean_squared_error(ground_truth, result)
    return 10 * np.log10((255 ** 2) / MSE)


def NRM(result, ground_truth):
    result = result.astype(np.int64)
    ground_truth = ground_truth.astype(np.int64)

    true_pos = np.sum(np.logical_and(result, ground_truth))
    true_neg = np.sum(np.logical_and(np.logical_not(result), np.logical_not(ground_truth)))
    false_pos = np.sum(np.logical_and(result, np.logical_not(ground_truth)))
    false_neg = np.sum(np.logical_and(np.logical_not(result), ground_truth))

    NR_FN = false_neg / (false_neg + true_pos)
    NR_FP = false_pos / (false_pos + true_neg)

    print("True pos")
    print(true_pos)
    print("True neg")
    print(true_neg)
    print("False pos")
    print(false_pos)
    print("False neg")
    print(false_neg)

    print("NR_FN")
    print(NR_FN)
    print("NR_FP")
    print(NR_FP)

    return (NR_FN + NR_FP) / 2


def MPM(result, ground_truth):
    border = cv2.Canny(ground_truth,10,200)
    gt_pixels = np.nonzero(border)

    D = 0
    for x in range(result.shape[0]):
        for y in range(result.shape[1]):
            distances = np.sqrt((gt_pixels[0] - x) ** 2 + (gt_pixels[1] - y) ** 2)
            D += np.min(distances)

    false_pos = np.where(np.logical_and(result, np.logical_not(ground_truth)), 255, 0)
    MP_FP = 0
    for (x, y) in zip(np.nonzero(false_pos)[0], np.nonzero(false_pos)[1]):
        distances = np.sqrt((gt_pixels[0] - x) ** 2 + (gt_pixels[1] - y) ** 2)
        MP_FP += np.min(distances)
    MP_FP = MP_FP / D

    false_neg = np.where(np.logical_and(np.logical_not(result), ground_truth), 255, 0)
    MP_FN = 0
    for (x, y) in zip(np.nonzero(false_neg)[0], np.nonzero(false_neg)[1]):
        distances = np.sqrt((gt_pixels[0] - x) ** 2 + (gt_pixels[1] - y) ** 2)
        MP_FN += np.min(distances)
    MP_FN = MP_FN / D

    return (MP_FN + MP_FP) / 2


if __name__ == "__main__":
    data_path = "../dataset/data/synthetic_test/noised"
    ground_path = "../dataset/data/synthetic_test/ground_truth"

    data_psnr = [['Image', 'Gatos', 'Kamel', 'Lantink', 'Mello', 'Niblack', 'Otsu', 'Rao', 'Software Project', 'Su']]
    data_nrm = [['Image', 'Gatos', 'Kamel', 'Lantink', 'Mello', 'Niblack', 'Otsu', 'Rao', 'Software Project', 'Su']]
    data_f_1 = [['Image', 'Gatos', 'Kamel', 'Lantink', 'Mello', 'Niblack', 'Otsu', 'Rao', 'Software Project', 'Su']]
    data_mpm = [['Image', 'Gatos', 'Kamel', 'Lantink', 'Mello', 'Niblack', 'Otsu', 'Rao', 'Software Project', 'Su']]

    index = 1
    for file in os.listdir(data_path):
        if file[-4:] != ".png" and file[-4:] != ".jpg":
            continue

        print(index)
        index += 1
        image = cv2.imread(data_path + "/" + file, cv2.IMREAD_GRAYSCALE)
        ground = cv2.imread(ground_path + "/" + file, cv2.IMREAD_GRAYSCALE)
        ground = utils.fix_reading(ground)

        # These assume that 255 is positive and 0 is negative, which is the opposite to the binarized images
        # Generates results for all 9 algorithms
        gatos = Gatos(image).binarize().astype(np.int64)
        gatos = np.abs(gatos - 255)

        kamel = Kamel_Zhao(image).binarize().astype(np.int64)
        kamel = np.abs(kamel - 255)

        lantink = Lantink(image).binarize().astype(np.int64)
        lantink = np.abs(lantink - 255)

        mello = Mello_Costa(image).binarize().astype(np.int64)
        mello = np.abs(mello - 255)

        niblack = Niblack(image).binarize().astype(np.int64)
        niblack = np.abs(niblack - 255)

        otsu = Otsu(image).binarize().astype(np.int64)
        otsu = np.abs(otsu - 255)

        rao = Rao_et_al(image).binarize().astype(np.int64)
        rao = np.abs(rao - 255)

        sp = SoftwareProject(image).binarize().astype(np.int64)
        sp = np.abs(sp - 255)

        su = Bolan_Su(image).binarize().astype(np.int64)
        su = np.abs(su - 255)

        # Compare using F1
        image_f_1_results = [file]
        image_f_1_results.append(F_1(gatos, ground))
        image_f_1_results.append(F_1(kamel, ground))
        image_f_1_results.append(F_1(lantink, ground))
        image_f_1_results.append(F_1(mello, ground))
        image_f_1_results.append(F_1(niblack, ground))
        image_f_1_results.append(F_1(otsu, ground))
        image_f_1_results.append(F_1(rao, ground))
        image_f_1_results.append(F_1(sp, ground))
        image_f_1_results.append(F_1(su, ground))
        data_f_1.append(image_f_1_results)

        # Compare using PSNR
        image_psnr_results = [file]
        image_psnr_results.append(PSNR(gatos, ground))
        image_psnr_results.append(PSNR(kamel, ground))
        image_psnr_results.append(PSNR(lantink, ground))
        image_psnr_results.append(PSNR(mello, ground))
        image_psnr_results.append(PSNR(niblack, ground))
        image_psnr_results.append(PSNR(otsu, ground))
        image_psnr_results.append(PSNR(rao, ground))
        image_psnr_results.append(PSNR(sp, ground))
        image_psnr_results.append(PSNR(su, ground))
        data_psnr.append(image_psnr_results)

        # Compare using NRM
        image_nrm_results = [file]
        image_nrm_results.append(NRM(gatos, ground))
        image_nrm_results.append(NRM(kamel, ground))
        image_nrm_results.append(NRM(lantink, ground))
        image_nrm_results.append(NRM(mello, ground))
        image_nrm_results.append(NRM(niblack, ground))
        image_nrm_results.append(NRM(otsu, ground))
        image_nrm_results.append(NRM(rao, ground))
        image_nrm_results.append(NRM(sp, ground))
        image_nrm_results.append(NRM(su, ground))
        data_nrm.append(image_nrm_results)

        # Compare using MPM
        image_mpm_results = [file]
        image_mpm_results.append(MPM(gatos, ground))
        image_mpm_results.append(MPM(kamel, ground))
        image_mpm_results.append(MPM(lantink, ground))
        image_mpm_results.append(MPM(mello, ground))
        image_mpm_results.append(MPM(niblack, ground))
        image_mpm_results.append(MPM(otsu, ground))
        image_mpm_results.append(MPM(rao, ground))
        image_mpm_results.append(MPM(sp, ground))
        image_mpm_results.append(MPM(su, ground))
        data_mpm.append(image_mpm_results)

    csv_path_f_1 = "synthetic_eval_test_f_1.csv"
    csv_path_psnr = "synthetic_eval_test_psnr.csv"
    csv_path_nrm = "synthetic_eval_test_nrm.csv"
    csv_path_mpm = "synthetic_eval_test_mpm.csv"

    # Output results to .csv files
    with open(csv_path_f_1, mode='w', newline='') as file:
        writer = csv.writer(file)
        writer.writerows(data_f_1)

    with open(csv_path_psnr, mode='w', newline='') as file:
        writer = csv.writer(file)
        writer.writerows(data_psnr)

    with open(csv_path_nrm, mode='w', newline='') as file:
        writer = csv.writer(file)
        writer.writerows(data_nrm)

    with open(csv_path_mpm, mode='w', newline='') as file:
        writer = csv.writer(file)
        writer.writerows(data_mpm)





