import os
import cv2
import random
import numpy as np

"""
This file contains code to noise ground truth synthetic images.
The code in this file is a heavily modified version of the code from image_noising.py found at:
https://github.com/EvgheniiBeriozchin/watermark-detection
"""


def change_intensities(image):
    # Randomly select background intensity from a range
    background_intensity = random.randint(50, 200)
    # Randomly select foreground intensity based on the background intensity. This ensures low contrast
    # Note that although this range looks like a lot, the horizontal lines obscure the watermark significantly, lowering
    # the contrast.
    foreground_intensity = np.clip(random.randint(background_intensity + 40, background_intensity + 60), 0, 255)
    # Replace the binarized image intensities with those randomly selected
    return np.where(image == 255, foreground_intensity, background_intensity), \
           foreground_intensity, background_intensity


def apply_noise(image: np.ndarray, min_limit=25, max_limit=150):
    # Applies noise to the image with random hyper-parameters
    noise = 5 + random.randint(min_limit, max_limit) * (np.random.randn(image.shape[0], image.shape[1]) - 0.5)
    noisy_image = np.clip(image + noise, 0, 255)

    return noisy_image.astype(np.uint8)


def apply_blur(image: np.ndarray, apply_deviation=False):
    # Applies blur to the image with random hyper-parameters
    blur_value = int(random.randrange(5, 20, 2))
    if not apply_deviation:
        blurry_image = cv2.GaussianBlur(image, (blur_value, blur_value), cv2.BORDER_DEFAULT)
    else:
        sigmaX = random.randint(0, 3)
        sigmaY = random.randint(0, 3)
        blurry_image = cv2.GaussianBlur(image, (blur_value, blur_value), sigmaX=sigmaX, sigmaY=sigmaY)

    return blurry_image


def apply_vertical_lines(image: np.ndarray, color):
    # Randomly select the number of lines to add.
    num_lines = random.randint(0, 2)
    if num_lines == 0:
        return image

    # Space out the lines to add linearly across the image width. Two extra lines are added here because linspace
    # always has lines at the extreme ends of the image, so adding two makes it easier for spacing. This means that,
    # depending on the spacing, a max of 3 vertical lines can appear in the image.
    spaced_lines = np.linspace(0, image.shape[1], num=num_lines + 2)
    # Determines the spacing in between the lines, and shifts the lines by a random amount up to the spacing size. This
    # ensures that the lines don't have the same central spacing each time, but that the distance between the lines
    # is always equidistant (since this is the case for the watermarks)
    spacing = int(spaced_lines[1] - spaced_lines[0])
    shifted_lines = spaced_lines + random.randint(0, spacing)

    line_thickness = random.randint(3, 5)

    # Generates the lines according to a randomly selected thickness, using the positions found in shifted_lines
    lines_image = np.zeros(image.shape, np.uint8)
    for line in shifted_lines:
        # This thickness randomness is added so that there is variability between lines as well
        actual_line_thickness = random.randint(-1, 1) + line_thickness
        lines_image = cv2.line(lines_image, (int(line), 0), (int(line), image.shape[0]),
                               color, actual_line_thickness)

    # Applies noise to the lines
    if color[0] > 50:
        noisy_lines_image = apply_noise(lines_image, min_limit=50, max_limit=100)
    else:
        noisy_lines_image = apply_noise(lines_image, max_limit=70)

    # Adds the lines to the image
    return np.clip(image + noisy_lines_image, 0, 255)

def apply_horizontal_lines(image: np.ndarray, color):
    # Generates horizontal lines with the same general procedure as described in the apply_vertical_lines method.

    # Generates the number of lines based on the height of the image. This generates much more horizontal lines than
    # vertical lines.
    num_lines = random.randint(int(image.shape[0] / 20), int(image.shape[0] / 12))

    spaced_lines = np.linspace(0, image.shape[0], num=num_lines)
    spacing = int(spaced_lines[1] - spaced_lines[0])
    shifted_lines = spaced_lines + random.randint(0, spacing)

    line_thickness = random.randint(1, 2)

    lines_image = np.zeros(image.shape, np.uint8)
    for line in shifted_lines:
        actual_line_thickness = line_thickness
        lines_image = cv2.line(lines_image, (0, int(line)), (image.shape[1], int(line)),
                               color, actual_line_thickness)

    if color[0] > 50:
        noisy_lines_image = apply_noise(lines_image, min_limit=10, max_limit=30)
    else:
        noisy_lines_image = apply_noise(lines_image, min_limit=0, max_limit=20)

    return np.clip(image + noisy_lines_image, 0, 255)


def make_lines_grey(image: np.ndarray):
    # Makes lines more similar to chain and laid line by turning them gray
    threshold_value = random.randint(80, 180)
    threshold = np.full((image.shape[0], image.shape[1]), 170, dtype=int)
    return np.clip(image + threshold, 0, 255), threshold_value


def create_realistic_drawing(image: np.ndarray):
    # Dilate the drawing image so that the lines become a little thicker, closer to the real images
    image = image.astype(np.uint8)
    # processed_image = cv2.dilate(image, cv2.getStructuringElement(cv2.MORPH_CROSS, (3,3)))
    # First changes the foreground and background intensities of the binarized image.
    processed_image, foreground, background = change_intensities(image)

    # Determines the color of the vertical and horizontal lines using the foreground color. Horizontal lines should be
    # somewhat further from the foreground than the vertical lines. Note the background is subtracted to make image
    # addition work.
    vert_color = random.randint(foreground - 20, foreground - 5) - background
    # horiz_color = random.randint(foreground - 20, foreground - 5) - background
    # Vertical and horizontal lines are added.
    processed_image = apply_vertical_lines(processed_image, (vert_color, vert_color, vert_color))
    processed_image = apply_horizontal_lines(processed_image, (vert_color, vert_color, vert_color))
    # processed_image, threshold = make_lines_grey(processed_image)
    # Noise is added to the image, and it is blurred.
    threshold = random.randint(30, 100)
    processed_image = apply_noise(processed_image, max_limit=max(int(threshold / 2), 50))
    processed_image = apply_blur(processed_image)
    processed_image = apply_blur(processed_image, True)

    return processed_image


if __name__ == '__main__':
    in_path = "data/synthetic_test/ground_truth"
    out_path = "data/synthetic_test/noised"

    # Goes through each image in the ground truth folder, noises it, and moves it to the `noised` folder.
    for file in os.listdir(in_path):
        if file[-4:] != ".png" and file[-4:] != ".jpg":
            continue

        image = cv2.imread(in_path + "/" + file, cv2.IMREAD_GRAYSCALE)
        realistic_image = create_realistic_drawing(image)
        cv2.imwrite(out_path + "/" + file, realistic_image)

