import os
from global_methods.Otsu import Otsu
import cv2
import numpy as np
import shutil

"""
Creating ground truths using the images from the drawings dataset. This involves cropping the image,
converting it into the correct binarized form using Otsu thresholding, and moving the image into the correct
folder.
"""

def crop(image):
    """
    Crops a binarized image 20 pixels away from the bounding box
    :param image: image to crop
    :return: the cropped image
    """
    contours, _ = cv2.findContours(image, cv2.RETR_LIST, cv2.CHAIN_APPROX_SIMPLE)[-2:]

    x_min = image.shape[0]
    x_max = 0
    y_min = image.shape[1]
    y_max = 0
    for contour in contours:
        x, y, w, h = cv2.boundingRect(contour)
        if x < x_min:
            x_min = x
        if x + w > x_max:
            x_max = x + w
        if y < y_min:
            y_min = y
        if y + h > y_max:
            y_max = y + h

    return image[y_min - 20: y_max + 20, x_min - 20: x_max + 20]

def move_image(files_list, from_path, to_path):
    """
    Moves images specified in the files_list from one place to another
    :param files_list: the txt file that contains the files to move
    :param from_path: the path that the files are that need to be moved
    :param to_path: the path for the files to move to
    """
    with open(files_list) as file:
        lines = file.read().splitlines()
        for line in lines:
            # This split line works specifically for the synthetic images which are stored as image_type/image_name.png
            # Since we only need the image name, the image_type directory goes away
            file_name = line.split("/")[1]
            shutil.copy(from_path+"/"+line, to_path + "/" + file_name)

if __name__ == "__main__":
    files_txt_path = "synthetic_test_samples.txt"
    start_path = "synthetic"
    data_path = "data/synthetic_validation/raw"
    write_path = "data/synthetic_validation/ground_truth"

    # First moves the images to the correct folder
    # move_image(files_txt_path, start_path, data_path)

    # For all images in the folder
    for file in os.listdir(data_path):
        if file[-4:] == ".png" or file[-4:] == ".jpg":
            # Read the image and resize it to a height of 512
            image = cv2.imread(data_path + "/" + file, cv2.IMREAD_GRAYSCALE).astype(np.int64)
            h, w = image.shape[:2]
            aspect = w / h
            image = cv2.resize(image.astype(np.uint8), (512, int(512 * aspect)), cv2.INTER_LINEAR)

            # The images are binarized using otsu. This works because the synthetic images have a very clear distinction
            # between foreground and background.
            image = Otsu(image).binarize()
            # The images are cropped 10 pixels around their bounding box
            cropped_image = crop(image)
            # The cropped binary images (that are now the ground truth) are saved to the ground truth file as jpg images
            cv2.imwrite(write_path + "/" + file[:-4] + ".jpg", cropped_image, [int(cv2.IMWRITE_JPEG_QUALITY), 100])

