% This file applies the Theil-Sen estimator on the time series within TS_cell

% time_stamps_input contains the time stamps of the given time series
    % If it is a list, these time stamps are used for all the given time series
    % If it is a cell, individual time stamps are used for the given time series

% partitions_input gives the limits of the partitions of the given time series
% The specific purpose of this input is to include break points, or determine rates of change for smaller time spans

    % Note: the inputs are expected to be indices, not time stamp values
    % Note: the inputs [1, length(TS)] are assumed and do not have to be included
    
    % If it is a list, these partitions are used for all the given time series
    % If it is a cell, individual partitions are used for the given time series
    
% This file is suited for parallel computing
function [Theil_Sen_slopes_cell, Theil_Sen_levels_cell] = Theil_Sen_Estimator_P(TS_cell, time_stamps_input, partitions_input, rows_data, columns_data, number_files)
    %%% Theil-Sen estimator %%%
       % Number of combinations becomes enormous if length is greater than ~150
        time_stamp_limit = 150;
    
        % These cells contain the final results for each partition within the time series
        Theil_Sen_slopes_cell = cell(rows_data, columns_data);
        Theil_Sen_levels_cell = cell(rows_data, columns_data);
        
        % Parallel computing loop, native resolution
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = rows_data * columns_data;
        afterEach(DQ, @ProgressUpdate);
        
        parfor n = 1:rows_data * columns_data
            if rows_data * columns_data > 1     % If only a single entry is calculated, a progress bar is not necessary
                % Update the progress bar
                send(DQ, n);
            end
            
            % Data belonging to this pixel
            TS = TS_cell{n};    % Time series data
            
            if iscell(time_stamps_input)    % Individual inputs are given for each time series
                time_stamps = time_stamps_input{n};
            else
                time_stamps = time_stamps_input;
            end
            
            % The given partitions
            if iscell(partitions_input)     % Individual inputs are given for each time series
                partitions = partitions_input{n};
            else
                partitions = partitions_input;
            end
            
            % The start and end of the time series are included
            partition_limits = [0, partitions, number_files];
            partition_limits = sort(partition_limits);
            number_partitions = length(partition_limits) - 1;
                                    
            % If the time series is constant, there is no point in continuing
            if max(TS) - min(TS) < 1e-6   % Due to slight rounding, a margin is used
                Theil_Sen_slopes = zeros(1, number_partitions);
                Theil_Sen_levels = mean(TS) * ones(1, number_partitions);
                
                Theil_Sen_slopes_cell{n} = Theil_Sen_slopes;
                Theil_Sen_levels_cell{n} = Theil_Sen_levels;
                
                continue
            end

            %--% The Theil-Sen estimator is ran for each partition %--%
            Theil_Sen_slopes = zeros(1, number_partitions);
            Theil_Sen_levels = zeros(1, number_partitions);
            
            for p = 1 : number_partitions
                % Indices of this partition
                i_start = partition_limits(p) + 1;
                i_end = partition_limits(p + 1);
                
                indices_partition = i_start : i_end;
                
                % The temporal resolution is reduced, if need be
                number_time_stamps_partition = length(indices_partition);
                
                if number_time_stamps_partition < 3     % The Theil-Sen estimator cannot be used and 0 values are appended
                    Theil_Sen_slopes(p) = NaN;
                    Theil_Sen_levels(p) = NaN;
                
                    continue
                end
                
                division_factor = ceil(number_time_stamps_partition / time_stamp_limit);
                indices_partition = indices_partition(1 : division_factor : end);
                
                number_time_stamps_partition = length(indices_partition);

                % This pixel's data for this partition, coarsened if need be
                TS_partition = TS(indices_partition);
                time_stamps_partition = time_stamps(indices_partition);

                % Determine the slopes between all pairs of data within the partition
                possible_combinations = factorial(number_time_stamps_partition) / (2 * factorial(number_time_stamps_partition - 2));

                % Due to Matlab rounding, possible combinations might not be an integer value
                possible_combinations = floor(possible_combinations);

                slope_list = zeros(1, possible_combinations);

                v = 0;
                for i = 1 : number_time_stamps_partition
                    for j = i + 1 : number_time_stamps_partition
                        v = v + 1;

                        slope = (TS_partition(j) - TS_partition(i)) / (time_stamps_partition(j) - time_stamps_partition(i));

                        slope_list(v) = slope;
                    end
                end

                % Determine the final slope and level
                TS_slope = median(slope_list);

                TS_level = median(TS_partition) - TS_slope * median(time_stamps_partition);  

                % Append the results for this partition
                Theil_Sen_slopes(p) = TS_slope;
                Theil_Sen_levels(p) = TS_level;
            end
            
            % Append the results for this pixel
            Theil_Sen_slopes_cell{n} = Theil_Sen_slopes;
            Theil_Sen_levels_cell{n} = Theil_Sen_levels;
        end

    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Theil-Sen estimator progress: %g%% \n', round(progress));
        end
    end
end