% This function applies the Mann-Kendall test on the given time series
% If the MK test statistic value is negative, it indicates a negative trend and vice versa
% Only the absolute magnitude is indicative of its significance

% partitions_input gives the limits of the partitions of the given time series
% The specific purpose of this input is to include break points, or determine the statistical significance for smaller time spans

    % Note: the inputs are expected to be indices, not time stamp values
    % Note: the inputs [1, length(TS)] are assumed and do not have to be included
    
    % If it is a list, these partitions are used for all the given time series
    % If it is a cell, individual partitions are used for the given time series

% This file is suited for parallel computing
function [Mann_Kendall_cell, p_values_cell] = Mann_Kendall_Test_P(TS_cell, partitions_input, rows_data, columns_data)

    %%% Determine the MK test statistic values and their p-values %%%
        Mann_Kendall_cell = cell(rows_data, columns_data);
        p_values_cell = cell(rows_data, columns_data);
        
        % Parallel loop, native resolution
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = rows_data * columns_data;
        afterEach(DQ, @ProgressUpdate);
        
        parfor i = 1 : rows_data * columns_data              
            if rows_data * columns_data > 1     % If only a single entry is calculated, a progress bar is not necessary
                % Update the progress bar
                send(DQ, i);
            end
            
            % This pixel's time series data
            TS = TS_cell{i};

            % If the time series is constant, there is no point in continuing and the default values are used
            if max(TS) - min(TS) < 1e-6   % Due to slight rounding, a margin is used
                Mann_Kendall_cell{i} = 0;
                p_values_cell{i} = 0.5;
                
                continue
            end
            
            % The given partitions
            if iscell(partitions_input)     % Individual inputs are given for each time series
                partitions = partitions_input{i};
            else
                partitions = partitions_input;
            end
            
            % The start and end of the time series are included
            partition_limits = [0, partitions, length(TS)];
            partition_limits = unique(partition_limits);    % In case 0 or number_files were given as inputs
            partition_limits = sort(partition_limits);
            number_partitions = length(partition_limits) - 1;
            
            % The Mann-Kendall values and the associated p-values are computed for each partition
            Mann_Kendall_list = zeros(1, number_partitions);
            p_values_list = zeros(1, number_partitions);
            
            for p = 1:number_partitions
                % Indices of this partition
                i_start = partition_limits(p) + 1;
                i_end = partition_limits(p + 1);
                
                indices_partition = i_start : i_end;
                
                n = length(indices_partition);  % Number of samples within this partition
                
                % This partition's time series data
                TS_p = TS(indices_partition);

                % Determine the sign of the change between each successive pair of data points
                S = 0;
                for a = 1:n - 1
                    for b = a + 1:n
                        % Small variations due to Matlab rounding should be ignored
                        if abs(TS_p(b) - TS_p(a)) > 1e-6
                            S = S + sign(TS_p(b) - TS_p(a));
                        end
                    end
                end

                s = n * (n - 1) * (2*n + 5)/18;

                % Determine the MK statistic values
                if S > 0
                    Z_MK = (S - 1)/sqrt(s);
                elseif S < 0
                    Z_MK = (S + 1)/sqrt(s);
                else
                    Z_MK = 0;
                end

                % Determine the associated p-value
                p_value = 1 - normcdf(abs(Z_MK));
                
                Mann_Kendall_list(p) = Z_MK;
                p_values_list(p) = p_value;
            end
            
            % Append the values of this pixel
            Mann_Kendall_cell{i} = Mann_Kendall_list;
            p_values_cell{i} = p_values_list;
        end         
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Mann-Kendall test progress: %g%% \n', round(progress));
        end
    end
end