% This function determines the frequency response of the given time series signal
% Plot = [On / Off] determines whether or not 12 subplots are created to show samples of the frequency response of the given data

function [fft_cell, magn_cell] = Fast_Fourier_Transform(TS_cell, number_files, rows_data, columns_data, Plot)
    %%% Inputs %%%
        Fs = (1.0/10.0);                % Sampling frequency, one every ten days by default

    %%% Discrete Fourier Transform %%%
        fft_cell = cell(rows_data, columns_data);
        magn_cell = cell(rows_data, columns_data);

        for i = 1:columns_data
            for j = 1:rows_data
                TS = TS_cell{j, i};                  

                list_mean = mean(TS);
                data_m_list = TS - list_mean;

                nfft2 = 2.^nextpow2(number_files);
                fft_l = fft(data_m_list, nfft2);
                fft_l = fft_l(1:nfft2/2);       % Symmetrical, so divide it in half

                xfft = Fs.*(0:nfft2/2 - 1)/nfft2;
                mfft = abs(fft_l/max(fft_l));

                fft_cell{j, i} = xfft;
                magn_cell{j, i} = mfft;
            end
        end

    %%% Plot %%%
        if strcmp(Plot, 'On')
            % Frequency magnitudes
            figure(1111)
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1.0 1.0])
            set(gcf, 'color', [1, 1, 1])
            sgtitle('Frequency magnitudes (mean was subtracted) of randomly selected pixels')

            number_plots = 12;      % Plots are randomly selected in the data

            c = 0;
            while c < number_plots
                c = c + 1;

                index = round(rand(1) * rows_data * columns_data);

                subplot(4, 3, c) 
                hold on
                % The frequency response
                plot(fft_cell{index}, magn_cell{index}, 'DisplayName', 'Frequency response', 'LineWidth', 2)

                % Time markers
                plot([1/360, 1/360], [0, 1], 'DisplayName', '360 days');
                plot([1/180, 1/180], [0, 1], 'DisplayName', '180 days');
                plot([1/90, 1/90], [0, 1], 'DisplayName', '90 days');
                hold off

                axis([0 Fs/2.0 0 1.0])

                xlabel('Frequency [1/day]')
                ylabel('Norm. magnitude')
                legend('show');
            end    

            % Save the figure
            export_fig('FFT_time_series.png');
            close(1111)
        end
end