% This file determines the textural features based on the given feature data

% All outputs are normalised, with 0 and 1 being the lowest and highest amount found within the domain

% Suitable for parallel computing
function [feature_data_years_cell, number_features, Texture_feature_names] = Textural_Features(feature_data_IMG_cell, number_years, rows_data, columns_data)
    
    %%% Inputs %%%
        window_width    = 15;   % The single-sided window width over which the textural features are computed in both x and y
        number_levels   = 10;   % The number of levels between which is distinguished to create the GLCM
          
    %%% Feature names %%%
        Texture_feature_names = {'Ang. Second Moment', 'Contrast', 'Correlation', 'Homogeneity'};
        number_features = length(Texture_feature_names);        
       
    %%% The textural features for each year %%%
        feature_data_years_cell = cell(1, number_years);
    
        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = number_years;
        afterEach(DQ, @ProgressUpdate);
        
        parfor y = 1 : number_years
            % This section's data
            feature_data_matrix = feature_data_IMG_cell{y};
            
            data_LB = min(min(feature_data_matrix));
            data_UB = max(max(feature_data_matrix));
            
            ASM_matrix = zeros(rows_data, columns_data);        
            Contrast_matrix = zeros(rows_data, columns_data);
            Correlation_matrix = zeros(rows_data, columns_data);
            Homogeneity_matrix = zeros(rows_data, columns_data);
            
            for r = 1 : rows_data
                % The limits of the assessed area
                r_N = max(1, r - window_width);
                r_S = min(rows_data, r + window_width);

                for c = 1 : columns_data
                    c_W = max(1, c - window_width);
                    c_E = min(columns_data, c + window_width);

                    % This area's data
                    data_area = feature_data_matrix(r_N : r_S, c_W : c_E);

                    % The gray-level co-occurence matrix
                    GLCM = graycomatrix(data_area, 'GrayLimits', [data_LB, data_UB], 'NumLevels', number_levels);       

                    % The textural properties
                    GLCM_properties = graycoprops(GLCM, {'Correlation', 'Contrast', 'Energy', 'Homogeneity'});
                    contrast = GLCM_properties.Contrast;
                    correlation = GLCM_properties.Correlation;
                    ASM = GLCM_properties.Energy;
                    homogeneity = GLCM_properties.Homogeneity;

                    Contrast_matrix(r, c) = contrast;
                    Correlation_matrix(r, c) = correlation;
                    ASM_matrix(r, c) = ASM;
                    Homogeneity_matrix(r, c) = homogeneity;
                end
            end
            
            % If the number of unique digits > number of levels (i.e. for water bodies), NaN values can appear
            % These values are substituted
            Correlation_matrix(isnan(Correlation_matrix)) = 1;  % A constant area is assumed to be fully correlated
            
            % Append the data
            feature_data = {ASM_matrix, Contrast_matrix, Correlation_matrix, Homogeneity_matrix};
            feature_data_years_cell{y} = feature_data;
            
            % Progress update
            send(DQ, y);
        end
        
        % The feature data is normalised
        for y = 1 : number_years
            for f = 1 : number_features
                % This year's and metric's feature data
                feature_data = feature_data_years_cell{y}{f};
                
                % The bounds of this feature data
                LB = min(min(feature_data));
                UB = max(max(feature_data));
                
                % Normalisation
                feature_data = (feature_data - LB)/(UB - LB);
                
                % The feature data is substituted
                feature_data_years_cell{y}{f} = feature_data;
            end
        end

    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        progress = tick / N * 100;
			
        fprintf('   Determining textural metrics progress: %g%% \n', round(progress));

        % Print the current time
        time = datestr(now, 'HH:MM:SS');
        fprintf('       t = %s \n', time);
    end
end