% This script retrieves the national indices belonging to each subdomain
% Note that rows_native and columns_native are expected to be in the native resolution!

% This file is suited for parallel computing
function [subdomain_indices_cell, subdomain_code_cell, number_subdomains] = Subdomain_Indices_Reader_P(Resolution_downscaling, coarsening_factor, rows_native, columns_native)
    % List of files containing the indices of the subdomains
    subdomain_indices_files = dir('*IND*.mat');
    
    number_subdomains = length(subdomain_indices_files);
    
    % Retrieve the individual lists of national indices, as well as the country codes
    subdomain_indices_cell = cell(1, number_subdomains);
    subdomain_code_cell = cell(1, number_subdomains);
    
    % Size of the data, if coarsened
    if strcmp(Resolution_downscaling, 'On')
        rows_reduced = ceil(rows_native / coarsening_factor);
        columns_reduced = ceil(columns_native / coarsening_factor);
    else
        rows_reduced = rows_native;
        columns_reduced = columns_native;
    end
    
    % Parallel computing loop
    DQ = parallel.pool.DataQueue;
    tick = 0;
    N = number_subdomains;
    afterEach(DQ, @ProgressUpdate);
    
    for n = 1:number_subdomains
        subdomain_indices_file_name = subdomain_indices_files(n).name;
        
        % The subdomain code is the last part of the file
        subdomain_parts = strsplit(subdomain_indices_file_name, '_');
        subdomain_code = subdomain_parts{end};
        subdomain_code = erase(subdomain_code, '.mat');
        subdomain_code_cell{n} = subdomain_code;
        
        subdomain_indices_file = load(subdomain_indices_file_name);
        subdomain_indices = subdomain_indices_file.national_indices;
        
        % Free memory
        clear subdomain_indices_file
        
        if strcmp(Resolution_downscaling, 'On')
            [R, C] = ind2sub([rows_native, columns_native], subdomain_indices);
            
            % Free memory
            clear subdomain_indices
                
            % Need to floor them
            R_coarsened = floor((R - 1)/coarsening_factor + 1);
            
            % Free memory
            clear R
            
            C_coarsened = floor((C - 1)/coarsening_factor + 1);
            
            % Free memory
            clear C
            
            subdomain_indices = sub2ind([rows_reduced, columns_reduced], R_coarsened, C_coarsened);
            subdomain_indices = unique(subdomain_indices);    % Multiple identical indices are possible
        end
        
        % The coarser resolution indices are appended
        subdomain_indices_cell{n} = subdomain_indices;
    end
    
    % The codes of the subdomains are saved for reference
    save('Subdomain_Codes.mat', 'subdomain_code_cell');
    
    function ProgressUpdate(~)
        tick = tick + 1;

        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;

        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Progress in retrieving subdomain indices: %g%% \n', round(progress));
            % Print the current time
            time = datestr(now, 'HH:MM:SS');
            fprintf('       t = %s \n', time);
        end
    end
end