% This script determines the size of the data in the directory as well as R, if a geotiff fiile is read 

% Data_type     = [NDVI, LST, ...], the type of data to be read
% Data_format   = [geotiff, mat], the format the data is in 

function [rows_data, columns_data, number_files, R] = Size_Reader(Data_type, Data_format)
    %%% Geotiff files %%%
    if strcmp(Data_format, 'geotiff')
        % The size of *.tif(f) files is assessed
        file_identifier = sprintf('%s*.tif*', Data_type);
        file_list = dir(file_identifier);

        % When parts of the data are pre-processed at a time, they are given coordinates xXXyYY
        % These aren't desired to be read again
        file_list = file_list(~contains({file_list.name}, 'x'));

        number_files = length(file_list);

        file = file_list(1).name;
        info = geotiffinfo(file);

        rows_data = info.Height;
        columns_data = info.Width;

        [~, R] = geotiffread(file);
        
    %%% Mat files %%%
    elseif strcmp(Data_format, 'mat')
        % The size of an UB file is assessed if it is available, otherwise any available .mat file
        file_identifier = sprintf('%s*.mat', Data_type);
        
        file_list = dir(file_identifier);
        
        if isempty(file_list)
            file_identifier = sprintf('%s*.mat', Data_type);
            
            file_list = dir(file_identifier);
        end
        
        % When parts of the data are pre-processed at a time, they are given coordinates xXXyYY
        % These aren't desired to be read again
        file_list = file_list(~contains({file_list.name}, 'x'));
        
        % The size of the file is read
        file = file_list(1).name;
        parameter_list = whos('-file', file);
        
        rows_data = parameter_list.size(1);
        columns_data = parameter_list.size(2);
        
        % The remaining outputs
        number_files = 1;   % Only one file per type is read
        R = [];             % R is used to create new geotiff files, and not available for mat files
    end
end



