% This script reads the pixel reliability data of the 250m MODIS NDVI data

% It is suitable for parallel computing
function [PR_IMG_cell, rows_data, columns_data, number_files] = MODIS_Pixel_Reliability_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, col_W, col_E)
    if strcmp(Resolution_downscaling, 'On')
        fprintf('The resolution will be reduced by factor %i. \n', coarsening_factor)
    elseif strcmp(Resolution_downscaling, 'Off')
        disp('The original resolution will be kept.')
    else
       	disp('Error: no resolution downscaling choice was made, the original resolution has been kept.')
    end

    %%% RETRIEVE DATA %%%
        file_list = dir('MOD*pixel_reliability*.tif');    
        
        % When parts of the data are pre-processed at a time, they are given coordinates xXXyYY
        % These aren't desired to be read again
        number_files = length(file_list);
        
        for k = 1:number_files
            file = file_list(k).name;
            
            % All the MODIS files contain x and y as well, but the chunk files do not contain aid0001
            if ~contains(file, 'aid0001')
                file_list(k) = [];
            end
        end
        
        number_files = length(file_list);

        PR_IMG_cell = cell(1, number_files);                         % Cell array containing the NDVI data sets for each time stamp

        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = number_files;
        afterEach(DQ, @ProgressUpdate);
        
        parfor k = 1:number_files
            file = file_list(k).name;
            
            PR_data = imread(file);             % retrieve the data, which is of type int16
            
            % A subsection of the data is selected, if desired
            [rows_data, columns_data] = size(PR_data);
            
            rows_removed_N = 1 : row_N - 1;
            rows_removed_S = row_S + 1 : rows_data;
            rows_removed = [rows_removed_N, rows_removed_S];
            PR_data(rows_removed, :) = [];
            
            cols_removed_W = 1 : col_W - 1;
            cols_removed_E = col_E + 1 : columns_data;
            cols_removed = [cols_removed_W, cols_removed_E];
            PR_data(:, cols_removed) = [];
            
            % The resolution is coarsened, if desired
            if strcmp(Resolution_downscaling, 'On')
                PR_data = PR_data(1:coarsening_factor:end, 1:coarsening_factor:end);
            end

            % The files are stored in int8 format, so need to be converted to doubles
            PR_data = double(PR_data);
            
            PR_IMG_cell{k} = PR_data;
            
            % Progress update
            send(DQ, k);
        end
        
    %%% Size %%%
        [rows_data, columns_data] = size(PR_IMG_cell{1});
        
    %%% Printed statements regarding the read data %%%
        fprintf('The number of read files is %g. \n', number_files);
        fprintf('Each file is %g x %g pixels. \n', rows_data, columns_data);
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Pixel reliability data reading progress: %g%% \n', round(progress));
        end
    end
end