% This script reads the 250m MODIS NDVI data

% It is suitable for parallel computing
function [NDVI_IMG_cell, rows_data, columns_data, number_files, file_date_list, number_years, year_list] = MODIS_NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, col_W, col_E)
    if strcmp(Resolution_downscaling, 'On')
        fprintf('The resolution will be reduced by factor %i. \n', coarsening_factor)
    elseif strcmp(Resolution_downscaling, 'Off')
        disp('The original resolution will be kept.')
    else
       	disp('Error: no resolution downscaling choice was made, the original resolution has been kept.')
    end

    %%% RETRIEVE DATA %%%
        file_list = dir('MOD*.tif*');    
        
        % When parts of the data are pre-processed at a time, they are given coordinates xXXyYY
        % These aren't desired to be read again
        file_list = file_list(~contains({file_list.name}, 'x'));
        
        number_files = length(file_list);

        NDVI_IMG_cell = cell(1, number_files);                          % Cell array containing the NDVI data sets for each time stamp
        file_date_list = cell(1, number_files);                         % The data of each file
        
        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = number_files;
        afterEach(DQ, @ProgressUpdate);
        
        parfor k = 1:number_files
            file = file_list(k).name;
            
            data = geotiffread(file);             % retrieve the data, which is of type int16
            
            % A subsection of the data is selected, if desired
            [rows_data, columns_data] = size(data);
            
            rows_removed_N = 1 : row_N - 1;
            rows_removed_S = row_S + 1 : rows_data;
            rows_removed = [rows_removed_N, rows_removed_S];
            data(rows_removed, :) = [];
            
            cols_removed_W = 1 : col_W - 1;
            cols_removed_E = col_E + 1 : columns_data;
            cols_removed = [cols_removed_W, cols_removed_E];
            data(:, cols_removed) = [];
            
            % The resolution is coarsened, if desired
            if strcmp(Resolution_downscaling, 'On')
                data = data(1:coarsening_factor:end, 1:coarsening_factor:end);
            end

            % The files are stored in int16 format, so need to be converted to doubles and to the NDVI values
            data = double(data);
            NDVI_data = data * 1e-4;
            
            NDVI_IMG_cell{k} = NDVI_data;
            
            % Date of each file
            file = erase(file, 'MOD13Q1');
            digit_ind = isstrprop(file, 'digit');
            date = file(digit_ind);
            
            year = date(1:4);
            month = date(5:6);
            day = date(7:8);
            
            file_date = [day, '.', month, '.', year];

            file_date_list{k} = file_date;
            
            % Progress update
            send(DQ, k);
        end
        
    %%% Size %%%
        [rows_data, columns_data] = size(NDVI_IMG_cell{1});
        
    %%% Years the data covers %%%
        date_start = strsplit(file_date_list{1}, '.');
        year_start = str2double(date_start{3});

        date_end = strsplit(file_date_list{number_files}, '.');
        year_end = str2double(date_end{3});

        year_list = year_start : year_end;
        number_years = length(year_list);
        
    %%% Printed statements regarding the read data %%%
        fprintf('The number of read files is %g. \n', number_files);
        fprintf('Each file is %g x %g pixels. \n', rows_data, columns_data);
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every 10 percent is printed
        percentage_increment = 10;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = round(tick / N * 100);
            
            fprintf('   NDVI data reading progress: %g%% \n', round(progress));
        end
    end
end