% This file reads the DTC generated classification files
% It can also retrieve the class names from Grouped_Classes*.xls
% If this file is not within the directory, the script will still function, but class_names will be empty

% Smoothed          = [Yes / No],        if yes the smoothed classification results are read, if no then the original results are read
%                                        Smoothed = SLC_YYYY.tif, original = LC_YYYY.tif
% Color_Choice      = [Corine / Colorbrewer], determines whether the prettier but less distinguishable Corine colours are used or the easy to distinguish Colorbrewer palette

function [class_data_cell, number_classes, class_names, class_values, RGB_codes] = DTC_Class_Data(Resolution_downscaling, coarsening_factor, Smoothed, Color_Choice, row_N, row_S, col_W, col_E)
    %%% Retrieve classification data files %%%
        if strcmp(Smoothed, 'Yes')
            class_list = dir('SLC_*.tif');
        elseif strcmp(Smoothed, 'No')
            class_list = dir('LC_*.tif');
        else
            disp('Error: the choice for smoothed or regular land cover was not understood.');
        end
            
    %%% Retrieve the land cover data %%%
        number_years = length(class_list);

        class_data_cell = cell(1, number_years);

        class_values = [];

        for y = 1 : number_years
            class_name = class_list(y).name;

            class_data = geotiffread(class_name);              % retrieve raw data in uint16
            class_data = double(class_data);
            
            % A subsection of the data is selected, if desired
            [rows_data, columns_data] = size(class_data);

            rows_removed_N = 1 : row_N - 1;
            rows_removed_S = row_S + 1 : rows_data;
            rows_removed = [rows_removed_N, rows_removed_S];
            class_data(rows_removed, :) = [];

            cols_removed_W = 1 : col_W - 1;
            cols_removed_E = col_E + 1 : columns_data;
            cols_removed = [cols_removed_W, cols_removed_E];
            class_data(:, cols_removed) = [];

            % Reduce the resolution of the class data by the same coarsening factor as the NDVI data
            if strcmp(Resolution_downscaling, 'On')
                class_data = class_data(1:coarsening_factor:end, 1:coarsening_factor:end);
            end

            % With MODIS classification, a few indices showed 0 
            class_data(class_data == 0) = 511;
            
            class_data_cell{y} = class_data;

            % Class values are appended if have not yet been found
            c_v = unique(class_data);

            for v = 1:length(c_v)
                if ~ismember(c_v(v), class_values)
                    class_values = [class_values, c_v(v)];
                end
            end
            
            % Progress
            if strcmp(Smoothed, 'Yes')
                fprintf('The smoothed land cover of %g/%g of the years has been retrieved \n', y, number_years);
            elseif strcmp(Smoothed, 'No')
                fprintf('The land cover of %g/%g of the years has been retrieved \n', y, number_years);
            end
        end
    
        % Class values are sorted to make sure the order is ascending
        class_values = sort(class_values);
        
        % The missing class is removed, if it exists
        class_values = setdiff(class_values, 999);
        
        % The number of classes
        number_classes = length(class_values);
    
    %%% Retrieve the names of the classes %%%
        grouped_classes_list = dir('Grouped_Classes*.xls');
        
        if ~isempty(grouped_classes_list)
            grouped_classes_file = grouped_classes_list(1).name;

            % Read the file
            grouped_classes_data = readtable(grouped_classes_file);

            % Retrieve the grouped classes and class values
            [number_classes_CLC, ~] = size(grouped_classes_data);

            grouped_class_values = cell(1, number_classes_CLC);
            class_names = cell(1, number_classes_CLC);
            class_values = zeros(1, number_classes_CLC);

            for c = 1:number_classes_CLC
                % The name of this group of classes
                group_name = grouped_classes_data{c, 1};
                class_names{c} = group_name{1};
                
                % The values of the classes within this group
                group_class_values = grouped_classes_data{c, 2 : end};
                group_class_values(isnan(group_class_values)) = [];

                grouped_class_values{c} = group_class_values;

                % The first value, used to denote this group in the future
                class_values(c) = group_class_values(1);
            end
            
            % The missing class is removed
            missing_ind = class_values == 999;
            class_values(missing_ind) = [];
            class_names(missing_ind) = [];

            number_classes = length(class_values);    
        else
            class_names = {};
        end
        
    %%% The classes are aggregated %%%
        if ~isempty(grouped_classes_list)
            for c = 1:number_classes
                class_group = grouped_class_values{c};
                class_value = class_values(c);

                number_classes_group = length(class_group);

				if number_classes_group > 1
					for n = 1:number_classes_group
						class = class_group(n);

						for y = 1:number_years
							class_data = class_data_cell{y};

							% The first class value is given
							class_indices = class_data == class;
							class_data(class_indices) = class_value;

							class_data_cell{y} = class_data;
						end
					end
				end
                
                % Progress
                if strcmp(Smoothed, 'No')
                    fprintf('The land cover data has been aggregated for %g%% of the classes \n', c / number_classes * 100);
                elseif strcmp(Smoothed, 'Yes')
                    fprintf('The smoothed land cover data has been aggregated for %g%% of the classes \n', c / number_classes * 100);
                end
            end
        end
        
    %%% The colour map %%%
        % Total set of class values
        class_values_urban = [111, 112, 121, 122, 123, 124, 131, 132, 133, 141, 142];
        class_values_agricultural = [211, 212, 213, 221, 222, 223, 231, 241, 242, 243, 244];
        class_values_forest = [311, 312, 313, 321, 322, 323, 324, 331, 332, 333, 334, 335];
        class_values_wetlands = [411, 412, 421, 422, 423];
        class_values_waterbodies = [511, 512, 521, 522, 523];
        class_values_nodata = 999;

        class_values_total = [class_values_urban, class_values_agricultural, class_values_forest, class_values_wetlands, class_values_waterbodies, class_values_nodata];
    
        % Corine
        if strcmp(Color_Choice, 'Corine')
            % RGB codes are the same as the Corine data base uses
            RGB_codes_total = [230, 0, 77; 255, 0, 0; 204, 77, 242; 204, 0, 0; 230, 204, 204; 230, 204, 230; 166, 0, 204; 166, 77, 0; 255, 77, 255; 255, 166, 255; 255, 230, 255;  ...
                               255, 255, 168; 255, 255, 0; 230, 230, 0; 230, 128, 0; 242, 166, 77; 230, 166, 0; 230, 230, 77; 255, 230, 166; 255, 230, 77; 230, 204, 77; 242, 204, 166;
                               128, 255, 0; 0, 166, 0; 77, 255, 0; 204, 242, 77; 166, 255, 128; 166, 230, 77; 166, 242, 0; 230, 230, 230; 204, 204, 204; 204, 255, 204; 0, 0, 0; 166, 230, 204;
                               166, 166, 255; 77, 77, 255; 204, 204, 255; 230, 230, 255; 166, 166, 230;
                               0, 204, 242; 128, 242, 230; 0, 255, 166; 166, 255, 230; 230, 242, 255;
                               0, 0, 0] / 255; 

           RGB_codes = zeros(number_classes, 3);

           for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
           end
           
       % Custom
        elseif strcmp(Color_Choice, 'Custom')
            RGB_codes_total =   [230, 176, 170; 217, 136, 128; 205, 97, 85; 192, 57, 43; 169, 50, 38; 146, 43, 33;  ...         % Reds: Urban fabric, commerical, industrial
                                 171, 178, 185; 86, 101, 115; 28, 40, 51; ...                                                   % Blacks: Mining, dumping, constructions
                                 212, 239, 223; 125, 206, 160; ...                                                              % Light greens: artificial vegetation
                                 253, 235, 208; 252, 243, 207; 247, 220, 111; 241, 196, 15; 183, 149, 11; 154, 125, 10; ...     % Yellows: agriculture
                                 248, 196, 113; 243, 156, 18; 230, 126, 34; 235, 152, 78; 175, 96, 26; ...                      % Oranges: pastures and complex agriculture
                                 34, 153, 84; 20, 90, 50; ...                                                                   % Dark greens: forests
                                 167, 217, 206; 115, 198, 182; 22, 160, 133; 19, 141, 117; 14, 102, 85; ...                     % Pastel greens: shrubs & herbaceous vegetaton
                                 98, 101, 103; 144, 148, 151; 189, 195, 199; 215, 219, 221; 242, 243, 244; ...                  % Greys: non-vegetated
                                 215, 189, 226; 195, 155, 211; 175, 122, 197; 136, 78, 160; 99, 57, 116; ...                    % Purples: wetlands
                                 52, 152, 219; 46, 134, 193; 40, 116, 166; 33, 97, 140; 27, 79, 114; ...                        % Blues: water bodies
                                 255, 255, 255] / 255;                                                                          % White: missing
                             
            RGB_codes = zeros(number_classes, 3);
                             
            for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
            end

        % Colorbrewer
        else
            RGB_codes = cbrewer('qual', 'Set1', number_classes);
        end
end