% The goal of this file is to read the Corine Land Cover files and group them based on the Grouped_Classes*.xls file

% Class_names includes all class values for that category, class_values is more succinct and gives only the first in the group
% Note that CLC_year should be a value denoting the year from which you want to retrieve the CLC data
% Plot              = [On / Off], determines whether a plot of the ground truth data is created and saved
% Color_Choice      = [Corine / Colorbrewer], determines whether the prettier but less distinguishable Corine colours are used or the easy to distinguish Colorbrewer palette

function [class_data, number_classes, class_names, class_values, RGB_codes] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, Plot, Color_Choice, row_N, row_S, col_W, col_E)
    %%% Class information from Corine %%%
        % The following classes are distinguished between by the Corine data base
        % Classes are 1st level distinguished in Xxx, 2nd level xXx, 3rd level xxX
        % i.e., Class 1.1.1 = 111, Class 2.3.2 = 232

        class_names_urban = ["Cont. urban", "Discont. urban", "Ind./comm. units", "Road/rail networks", "Ports", "Airports", "Mining", "Dump sites", "Construction", "Green urban", "Leisure"];
        class_values_urban = [111, 112, 121, 122, 123, 124, 131, 132, 133, 141, 142];
        class_names_agricultural = ["Non-irrigated", "Permanently irrigated", "Rice fields", "Vineyards", "Fruit trees", "Olive groves", "Pastures", "Annual crops", "Complex cultivation", "Agr. w/ natural veg.", "Forestry"];
        class_values_agricultural = [211, 212, 213, 221, 222, 223, 231, 241, 242, 243, 244];
        class_names_forest = ["Broad-leaved forest", "Coniferous forest", "Mixed forest", "Grasslands", "Moors", "Scrub", "Shrub", "Sand", "Rocks", "Sparsely vegetated", "Burnt areas", "Permanent snow"];
        class_values_forest = [311, 312, 313, 321, 322, 323, 324, 331, 332, 333, 334, 335];
        class_names_wetlands = ["Marshes", "Bogs", "Salt marshes", "Salines", "Intertidal flats"];
        class_values_wetlands = [411, 412, 421, 422, 423];
        class_names_waterbodies = ["Water courses", "Water bodies", "Lagoons", "Estuaries", "Sea/ocean"];
        class_values_waterbodies = [511, 512, 521, 522, 523];
        class_names_nodata = "No data";
        class_values_nodata = 999;

        class_names_total = [class_names_urban, class_names_agricultural, class_names_forest, class_names_wetlands, class_names_waterbodies, class_names_nodata];
        class_values_total = [class_values_urban, class_values_agricultural, class_values_forest, class_values_wetlands, class_values_waterbodies, class_values_nodata];
                                   
    %%% Retrieve the grouped classes %%%
        grouped_classes_list = dir('Grouped_Classes*.xls');
        grouped_classes_file = grouped_classes_list(1).name;
        
        % Read the file
        grouped_classes_data = readtable(grouped_classes_file);

        % Retrieve the grouped classes
        [number_classes, ~] = size(grouped_classes_data);

        grouped_class_values = cell(1, number_classes);
        class_names = cell(1, number_classes);
        class_values = zeros(1, number_classes);

        for c = 1:number_classes
            % The name of this group of classes
            group_name = grouped_classes_data{c, 1};
            class_names{c} = group_name{1};

            % The values of the classes within this group
            group_class_values = grouped_classes_data{c, 2 : end};
            group_class_values(isnan(group_class_values)) = [];

            grouped_class_values{c} = group_class_values;

            % The first value, used to denote this group in the future
            class_values(c) = group_class_values(1);
        end
        
    %%% Retrieve classification data %%%
        CLC_file_name = sprintf('CLC_%s*.tif', num2str(CLC_year));
        CLC_file_list = dir(CLC_file_name);

        if isempty(CLC_file_list)           % Check if the file was found
            fprintf('Error: The %s file was not found \n', CLC_file_name);
        end   

        CLC_file = CLC_file_list(1).name;
        class_data_full = geotiffread(CLC_file);             % Retrieve raw data in uint16        
        class_data_full = double(class_data_full);           % Convert to double
        
        % A subsection of the data is selected, if desired
        [rows_data, columns_data] = size(class_data_full);

        class_data = class_data_full;
        
        rows_removed_N = 1 : row_N - 1;
        rows_removed_S = row_S + 1 : rows_data;
        rows_removed = [rows_removed_N, rows_removed_S];
        class_data(rows_removed, :) = [];

        cols_removed_W = 1 : col_W - 1;
        cols_removed_E = col_E + 1 : columns_data;
        cols_removed = [cols_removed_W, cols_removed_E];
        class_data(:, cols_removed) = [];
                    
        % Reduce the resolution of the class data by the same coarsening factor as the NDVI data
        if strcmp(Resolution_downscaling, 'On')
            class_data = class_data(1:coarsening_factor:end, 1:coarsening_factor:end);
        end

        % The missing values, are given the 999 value
        % Note that this will not be a class, but the missing values will be ignored in classification
        class_data(class_data <= 0) = 999;
        class_data_full(class_data_full <= 0) = 999;
    
    %%% Group the classes together %%%
        for c = 1 : number_classes
            class_group = grouped_class_values{c};
            class_value = class_values(c);
                        
            number_classes_group = length(class_group);
            
            % The Corine data is grouped together
			if number_classes_group > 1
				for n = 1 : number_classes_group
					class = class_group(n);
					class_indices = class_data == class;
					
					class_data(class_indices) = class_value;
					
					if strcmp(Plot, 'On')
						class_indices = class_data_full == class;
						
						class_data_full(class_indices) = class_value;
					end
				end
			end
        end
            
    %%% The missing class is not included anymore %%%
        missing_ind = find(class_values == 999);
        
        if ~isempty(missing_ind)
            class_names(missing_ind) = [];
            class_values(missing_ind) = [];
            number_classes = number_classes - 1;
        end
        
    %%% Colour map choice %%%
        % Corine
        if strcmp(Color_Choice, 'Corine')
            % RGB codes are the same as the Corine data base uses
            RGB_codes_total = [230, 0, 77; 255, 0, 0; 204, 77, 242; 204, 0, 0; 230, 204, 204; 230, 204, 230; 166, 0, 204; 166, 77, 0; 255, 77, 255; 255, 166, 255; 255, 230, 255;  ...
                               255, 255, 168; 255, 255, 0; 230, 230, 0; 230, 128, 0; 242, 166, 77; 230, 166, 0; 230, 230, 77; 255, 230, 166; 255, 230, 77; 230, 204, 77; 242, 204, 166;
                               128, 255, 0; 0, 166, 0; 77, 255, 0; 204, 242, 77; 166, 255, 128; 166, 230, 77; 166, 242, 0; 230, 230, 230; 204, 204, 204; 204, 255, 204; 0, 0, 0; 166, 230, 204;
                               166, 166, 255; 77, 77, 255; 204, 204, 255; 230, 230, 255; 166, 166, 230;
                               0, 204, 242; 128, 242, 230; 0, 255, 166; 166, 255, 230; 230, 242, 255;
                               0, 0, 0] / 255; 

           RGB_codes = zeros(number_classes, 3);

           for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
           end
           
       % Custom
        elseif strcmp(Color_Choice, 'Custom')
            RGB_codes_total =   [230, 176, 170; 217, 136, 128; 205, 97, 85; 192, 57, 43; 169, 50, 38; 146, 43, 33;  ...         % Reds: Urban fabric, commerical, industrial
                                 171, 178, 185; 86, 101, 115; 28, 40, 51; ...                                                   % Blacks: Mining, dumping, constructions
                                 212, 239, 223; 125, 206, 160; ...                                                              % Light greens: artificial vegetation
                                 253, 235, 208; 252, 243, 207; 247, 220, 111; 241, 196, 15; 183, 149, 11; 154, 125, 10; ...     % Yellows: agriculture
                                 248, 196, 113; 243, 156, 18; 230, 126, 34; 235, 152, 78; 175, 96, 26; ...                      % Oranges: pastures and complex agriculture
                                 34, 153, 84; 20, 90, 50; ...                                                                   % Dark greens: forests
                                 167, 217, 206; 115, 198, 182; 22, 160, 133; 19, 141, 117; 14, 102, 85; ...                     % Pastel greens: shrubs & herbaceous vegetaton
                                 98, 101, 103; 144, 148, 151; 189, 195, 199; 215, 219, 221; 242, 243, 244; ...                  % Greys: non-vegetated
                                 215, 189, 226; 195, 155, 211; 175, 122, 197; 136, 78, 160; 99, 57, 116; ...                    % Purples: wetlands
                                 52, 152, 219; 46, 134, 193; 40, 116, 166; 33, 97, 140; 27, 79, 114; ...                        % Blues: water bodies
                                 255, 255, 255] / 255;                                                                          % White: missing
                             
            RGB_codes = zeros(number_classes, 3);
                             
            for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
            end

        % Colorbrewer
        else
            RGB_codes = cbrewer('qual', 'Set1', number_classes);
        end
                
    %%% Plot %%%
        if strcmp(Plot, 'On')        
            % The colours are specified
            class_image = cat(3, class_data_full, class_data_full, class_data_full);

            % RGB colour are given to the image
            for m = 1:3
                M = class_image(:, :, m);
                
                for c = 1:number_classes
                    class_value = class_values(c);
                    M(M == class_value) = RGB_codes(c, m);
                end
                
                % The missing class is shown in white
                M(M > 1) = 1;

                class_image(:, :, m) = M;
            end    

            figure(1000)        
            imshow(class_image);

            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 0.7 1]);
            set(gcf, 'color', [1, 1, 1]);

            if number_classes <= 10     % A legend is only included if it doesn't take up too much space
                for s = 1:number_classes
                    str = class_names{s};
                    annotation('textbox', [0.0 (0.91 - (0.5/number_classes)*s) 0.05 0.05], 'String', str, 'Fontweight', 'Bold', 'FitBoxToText', 'On', 'Color', RGB_codes(s, :, :), 'BackgroundColor', [0.5 0.5 0.5], 'FaceAlpha', 1.0, 'FontSize', 20);
                end
            end

            CLC_figure_name = ['CLC_', num2str(CLC_year), '.png'];

            try
                export_fig(CLC_figure_name);
            catch
                frame = getframe(1000);
                im = frame2im(frame);
                [imind, cm] = rgb2ind(im, 256);
                imwrite(imind, cm, CLC_figure_name);
            end

            close(1000)
        end
end