% This script reads the Copernicus land cover data

function [Copernicus_LC_data, number_classes, class_names, class_values, RGB_codes] = Copernicus_LC_Reader(Resolution_downscaling, coarsening_factor, Color_Choice, row_N, row_S, col_W, col_E)
    %%% Read the Copernicus data file %%%
        Copernicus_file_list = dir('Copernicus*.tif');
        LC_data = geotiffread(Copernicus_file_list(1).name);
        
        [rows_data, columns_data] = size(LC_data);

        rows_removed_N = 1 : row_N - 1;
        rows_removed_S = row_S + 1 : rows_data;
        rows_removed = [rows_removed_N, rows_removed_S];
        LC_data(rows_removed, :) = [];

        cols_removed_W = 1 : col_W - 1;
        cols_removed_E = col_E + 1 : columns_data;
        cols_removed = [cols_removed_W, cols_removed_E];
        LC_data(:, cols_removed) = [];
        
        % Coarsen, if need be
        if strcmp(Resolution_downscaling, 'On')
            LC_data = LC_data(1 : coarsening_factor : end, 1 : coarsening_factor : end);
        end
    
    %%% Change the classes s.t. they correspond to the Corine codes %%%
        Copernicus_codes = [0, 111, 112, 113, 114, 115, 116, 121, 122, 123, 124, 125, 126, 20, 30, 40, 50, 60, 70, 80, 90, 100, 200]; 
        Corine_codes = [999, 311 * ones(1, 12), 321, 321, 211, 111, 331, 335, 511, 411, 321, 511];        % Corresponding Corine codes

        Copernicus_LC_data = zeros(size(LC_data));

        for c = 1:length(Copernicus_codes)
            Copernicus_code = Copernicus_codes(c);
            Corine_code = Corine_codes(c);

            class_ind = LC_data == Copernicus_code;         % Indices of the Copernicus class
            Copernicus_LC_data(class_ind) = Corine_code;    % Append the Corine code to the new land cover data
        end

        class_values = unique(Copernicus_LC_data);
        number_classes = length(class_values);
    
    %%% Retrieve the names of the classes %%%
        grouped_classes_list = dir('Grouped_Classes*.xls');
        
        if ~isempty(grouped_classes_list)
            grouped_classes_file = grouped_classes_list(1).name;

            % Read the file
            grouped_classes_data = readtable(grouped_classes_file);

            % Retrieve the grouped classes and class values
            [number_classes_CLC, ~] = size(grouped_classes_data);

            grouped_class_values = cell(1, number_classes_CLC);
            class_names = cell(1, number_classes_CLC);
            class_values = zeros(1, number_classes_CLC);

            for c = 1:number_classes_CLC
                % The name of this group of classes
                group_name = grouped_classes_data{c, 1};
                class_names{c} = group_name{1};
                
                % The values of the classes within this group
                group_class_values = grouped_classes_data{c, 2 : end};
                group_class_values(isnan(group_class_values)) = [];

                grouped_class_values{c} = group_class_values;

                % The first value, used to denote this group in the future
                class_values(c) = group_class_values(1);
            end
            
            % The missing class is removed
            missing_ind = class_values == 999;
            class_values(missing_ind) = [];
            class_names(missing_ind) = [];

            number_classes = length(class_values);    
        else
            class_names = {};
        end
        
    %%% The colour map %%%
        % Total set of class values
        class_values_urban = [111, 112, 121, 122, 123, 124, 131, 132, 133, 141, 142];
        class_values_agricultural = [211, 212, 213, 221, 222, 223, 231, 241, 242, 243, 244];
        class_values_forest = [311, 312, 313, 321, 322, 323, 324, 331, 332, 333, 334, 335];
        class_values_wetlands = [411, 412, 421, 422, 423];
        class_values_waterbodies = [511, 512, 521, 522, 523];
        class_values_nodata = 999;

        class_values_total = [class_values_urban, class_values_agricultural, class_values_forest, class_values_wetlands, class_values_waterbodies, class_values_nodata];
    
        % Corine
        if strcmp(Color_Choice, 'Corine')
            % RGB codes are the same as the Corine data base uses
            RGB_codes_total = [230, 0, 77; 255, 0, 0; 204, 77, 242; 204, 0, 0; 230, 204, 204; 230, 204, 230; 166, 0, 204; 166, 77, 0; 255, 77, 255; 255, 166, 255; 255, 230, 255;  ...
                               255, 255, 168; 255, 255, 0; 230, 230, 0; 230, 128, 0; 242, 166, 77; 230, 166, 0; 230, 230, 77; 255, 230, 166; 255, 230, 77; 230, 204, 77; 242, 204, 166;
                               128, 255, 0; 0, 166, 0; 77, 255, 0; 204, 242, 77; 166, 255, 128; 166, 230, 77; 166, 242, 0; 230, 230, 230; 204, 204, 204; 204, 255, 204; 0, 0, 0; 166, 230, 204;
                               166, 166, 255; 77, 77, 255; 204, 204, 255; 230, 230, 255; 166, 166, 230;
                               0, 204, 242; 128, 242, 230; 0, 255, 166; 166, 255, 230; 230, 242, 255;
                               0, 0, 0] / 255; 

           RGB_codes = zeros(number_classes, 3);

           for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
           end
           
       % Custom
        elseif strcmp(Color_Choice, 'Custom')
            RGB_codes_total =   [230, 176, 170; 217, 136, 128; 205, 97, 85; 192, 57, 43; 169, 50, 38; 146, 43, 33;  ...         % Reds: Urban fabric, commerical, industrial
                                 171, 178, 185; 86, 101, 115; 28, 40, 51; ...                                                   % Blacks: Mining, dumping, constructions
                                 212, 239, 223; 125, 206, 160; ...                                                              % Light greens: artificial vegetation
                                 253, 235, 208; 252, 243, 207; 247, 220, 111; 241, 196, 15; 183, 149, 11; 154, 125, 10; ...     % Yellows: agriculture
                                 248, 196, 113; 243, 156, 18; 230, 126, 34; 235, 152, 78; 175, 96, 26; ...                      % Oranges: pastures and complex agriculture
                                 34, 153, 84; 20, 90, 50; ...                                                                   % Dark greens: forests
                                 167, 217, 206; 115, 198, 182; 22, 160, 133; 19, 141, 117; 14, 102, 85; ...                     % Pastel greens: shrubs & herbaceous vegetaton
                                 98, 101, 103; 144, 148, 151; 189, 195, 199; 215, 219, 221; 242, 243, 244; ...                  % Greys: non-vegetated
                                 215, 189, 226; 195, 155, 211; 175, 122, 197; 136, 78, 160; 99, 57, 116; ...                    % Purples: wetlands
                                 52, 152, 219; 46, 134, 193; 40, 116, 166; 33, 97, 140; 27, 79, 114; ...                        % Blues: water bodies
                                 255, 255, 255] / 255;                                                                          % White: missing
                             
            RGB_codes = zeros(number_classes, 3);
                             
            for c = 1:number_classes
               class_value = class_values(c);
               index = class_values_total == class_value;
               RGB_codes(c, :) = RGB_codes_total(index, :);
            end

        % Colorbrewer
        else
            RGB_codes = cbrewer('qual', 'Set1', number_classes);
        end
end