% This script retrieves the coordinates and projection information of the geotiffs of the specified data type

% The outputs are the horizontal coordinates (longitude / x) and the vertical coordinates (latitude / y)
% As well as the coordinate type, which is angular / cartesian
% Additionally the projection information is specified as well

% To reduce the memory cost, it is also possible for only a smaller section of the data to be read
% To do this, the limiting rows and columns must be given (North, South, West, East)
% Note that not all inputs need to be given, any empty entry is assumed to not be a limit

function [coordinates_h, coordinates_v, rows_data, columns_data, coordinate_type, projection_information] = Coordinate_Retriever(Data_type, Resolution_downscaling, coarsening_factor, row_N, row_S, col_W, col_E)
    %%% The file information %%%
        file_list = dir(sprintf('%s*.tiff', Data_type));
        file = file_list(1).name;
        
        try
            info = geotiffinfo(file);
        catch
            info = geotiffinfo_2(file);
        end
        
    %%% Coordinates %%%        
        [coordinates_h, coordinates_v] = pixcenters(info);
        
        rows_data = length(coordinates_v);
        columns_data = length(coordinates_h);
        
        rows_removed_N = 1 : row_N - 1;
        rows_removed_S = row_S + 1 : rows_data;
        rows_removed = [rows_removed_N, rows_removed_S];
        coordinates_v(rows_removed) = [];
        
        cols_removed_W = 1 : col_W - 1;
        cols_removed_E = col_E + 1 : columns_data;
        cols_removed = [cols_removed_W, cols_removed_E];
        coordinates_h(cols_removed) = [];
        
        % The resolution is reduced if need be
        if strcmp(Resolution_downscaling, 'On')
            coordinates_h = coordinates_h(1 : coarsening_factor : end);
            coordinates_v = coordinates_v(1 : coarsening_factor : end);
        end
        
        rows_data = length(coordinates_v);
        columns_data = length(coordinates_h);
        
    %%% Determine the coordinate type %%%
        coordinate_type_info = info.UOMLength;
        
        if isempty(coordinate_type_info)
            coordinate_type = 'Angular';
        else
            coordinate_type = 'Cartesian';
        end
        
    %%% The projection information %%%
        projection_information = info.ProjParm;     % False easting/northing, reference longitude/latitude
end