% This script coarsens the temporal resolution by computing the average value over the given window width

% This file is suitable for parallel computation
function TS_cell = Temporal_Data_Coarsener(TS_cell, rows_data, columns_data, window_width, number_files)
    %%% Coarsen the resolution temporally %%%
        % The number of windows within the time span
        number_windows = floor(number_files / window_width);    % Only full years are desired

        % Parallel loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = rows_data;
        afterEach(DQ, @ProgressUpdate);

        for r = 1:rows_data
            for c = 1:columns_data
                % The time series to be analysed
                TS = TS_cell{r, c};
                
                % The number of time stamps, if number_windows were an integer number
                number_time_stamps = number_windows * window_width;
                
                TS_c = TS(1 : number_time_stamps);
                
                % The mean is taken over the window widths
                TS_c = reshape(TS_c, [window_width, number_windows]);
                TS = nanmean(TS_c);
                
                % The new, coarsened time series is appended
                TS_cell{r, c} = TS;
            end
        end
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Temporal coarsening progress: %g%% \n', round(progress));
        end
    end
end