% This script creates a subset of the given data where only the subdomain has valid indices, all others are considered missing or sea
% Note that the full European domain files are expected as inputs, as is a single indices file (i.e. NAT_IND_FRA.mat)

% Subset_name   = The name of the subset indices to be read, i.e. FRA or IST
% Data_type     = Determines for which data the subset is created, see below for which data types are available
% Save_Files    = [Yes/No], determines whether the files are saved or only the output data is returned
% Plot          = [Yes/No], whether or not a plot is created showing the subset

% For quick reference if you want to save files of a subdomain
% Subdomain_Snipper('FRA', 'SLC', Subset_indices, 'Off', 1, 'Yes', 'Yes');

% The Subdomain_Data is a cell containing either the geotiff file entries in image format, or the .mat data 
% If the growing season .mat files are read, it is of the shape {GS_cell, t_GS_cell}
% If break point times .mat files are read, it is of the shape {BP_times_cell}
% For other data, it is of the shape {Trend_cell, Season_cell}

% The coordinates are only given if geotiff data is read

% This file is suited for parallel computing
function [Subdomain_Data, rows_subset, columns_subset, number_files, file_date_list, number_years, year_list, longitude_list_subset, latitude_list_subset] = Subdomain_Snipper(Subset_name, Data_type, subset_indices, Resolution_downscaling, coarsening_factor, Save_Files, Plot)

    %%% The invalid values %%%
        % These values are assigned to the indices which are inside the box surrounding the subset, but not part of the subset itself
        % The value corresponding to missing data is used, when possible

        % Copernicus vegetation data
        if strcmp(Data_type, 'NDVI') | strcmp(Data_type, 'LAI') | strcmp(Data_type, 'FAPAR') | strcmp(Data_type, 'FCOVER')
            invalid_value = 255;
            
        % MODIS NDVI data
        elseif strcmp(Data_type, 'MOD13Q1')
            invalid_value = -3000;

        % Metereological data
        elseif strcmp(Data_type, 'LST') | strcmp(Data_type, 'GPM')
            invalid_value = 0;
            
        % Land cover data
        elseif strcmp(Data_type, 'CLC') | strcmp(Data_type, 'SLC') | strcmp(Data_type, 'LC')
            invalid_value = 999;
            
        % Topographic data
        elseif strcmp(Data_type, 'EU_DEM_Slope') | strcmp(Data_type, 'EU_DEM_Elevation') | strcmp(Data_type, 'EU_DEM_Aspect')
            invalid_value = 0;

        % Pollinator data
        elseif strcmp(Data_type, 'Bumblebee')
            invalid_value = 0;
        end

    %%% Create subsets of the data %%%
        % Check whether the data is of .tif(f) or .mat format
        file_name = sprintf('%s*.tif*', Data_type);
        file_list = dir(file_name);

        if ~isempty(file_list)
            file_type = 'geotiff';
        else
            file_type = 'mat';        
        end

        %%% Create new geotiff files %%%
        if strcmp(file_type, 'geotiff')      
            % This file's projection information
            info = geotiffinfo(file_list(1).name);
            [longitude_list, latitude_list] = pixcenters(info);
            
            % Coarsen the resolution, if need be
            if strcmp(Resolution_downscaling, 'On')
                longitude_list = longitude_list(1 : coarsening_factor : end);
                latitude_list = latitude_list(1 : coarsening_factor : end);
            end
            
            % The data set's size
            rows_data = length(latitude_list);
            columns_data = length(longitude_list);
                                  
            % Determine the binding rows and columns of the subset
            [rows, columns] = ind2sub([rows_data, columns_data], subset_indices);

            row_N = min(rows);
            row_S = max(rows);
            col_W = min(columns);
            col_E = max(columns);

            rows_subset = row_S - row_N + 1;
            columns_subset = col_E - col_W + 1;
            
            % Modify R
            R_subset = info.SpatialRef;
            R_subset.RasterSize = [rows_subset, columns_subset];

            margin = longitude_list(2) - longitude_list(1);

            latitude_N = latitude_list(row_N) + 0.5 * margin;
            latitude_S = latitude_list(row_S) - 0.5 * margin;
            longitude_W = longitude_list(col_W) - 0.5 * margin;
            longitude_E = longitude_list(col_E) + 0.5 * margin;

            try     % Copernicus format
                R_subset.LatitudeLimits = [latitude_S, latitude_N];
                R_subset.LongitudeLimits = [longitude_W, longitude_E];
            catch   % MODIS format
                R_subset.XWorldLimits = [longitude_W, longitude_E];     % Note that these are actually x/y coordinates
                R_subset.YWorldLimits = [latitude_S, latitude_N];
            end
            
            % The coordinates of the subset
            longitude_list_subset = longitude_list(col_W : col_E);
            latitude_list_subset = latitude_list(row_N : row_S);
            
            % Create the subset geotiff files
            number_files = length(file_list);

            Subdomain_Data = cell(1, number_files);
            
            % Parallel loop
            DQ = parallel.pool.DataQueue;
            tick = 0;
            N = number_files;
            afterEach(DQ, @ProgressUpdate_tiff);
            
            parfor k = 1:number_files
                % Retrieve the original data
                file_name = file_list(k).name;
                
                file_name_short = erase(file_name, Data_type);    % In case the data type contains a digit
                digit_ind = isstrprop(file_name_short, 'digit');
                date = file_name_short(digit_ind);                % The data of this file, YYYYMMDD

                data = geotiffread(file_name);
                
                if strcmp(Resolution_downscaling, 'On')
                    data = data(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                end
                
                % Create a subset of the data
                data_subset = invalid_value * ones(rows_data, columns_data);
                data_subset(subset_indices) = data(subset_indices);
                data_subset = data_subset(row_N : row_S, col_W : col_E);

                if isa(data, 'uint8')                   % Convert the data back to 8-bit or 16-bit format
                    data_subset = uint8(data_subset);
                elseif isa(data, 'uint16')
                    data_subset = uint16(data_subset);
                end

                % Create the new file
                if strcmp(Save_Files, 'Yes')
                    subset_file_name = sprintf('%s_%s_%s', Data_type, date, Subset_name);

                    try     % Copernicus format
                        geotiffwrite(subset_file_name, data_subset, R_subset);
                    catch   % MODIS format
                        info = geotiffinfo(file_name);
                        Tag = info.GeoTIFFTags.GeoKeyDirectoryTag;

                        geotiffwrite(subset_file_name, data_subset, R_subset, 'GeoKeyDirectoryTag', Tag);
                    end

                    % If the original suffix is .tiff, it is added to the file
                    if strcmp(file_name(end - 4 : end), '.tiff')
                        movefile([subset_file_name, '.tif'], [subset_file_name, '.tiff']);
                    end
                    
                else    % If they are not saved, the real values are computed and saved
                    % Convert the data to the actual values
                    data_subset = double(data_subset);
                    
                    if strcmp(Data_type, 'NDVI')
                        data_subset = data_subset / 250 - 0.08;
                    elseif strcmp(Data_type, 'LAI')
                        data_subset = data_subset / 30;
                    elseif strcmp(Data_type, 'FCOVER') | strcmp(Data_type, 'FAPAR')
                        data_subset = data_subset / 250;
                    elseif strcmp(Data_type, 'LST')
                        data_subset = data_subset * 0.02 - 273.15;
                    elseif strcmp(Data_type, 'GPM')
                        data_subset = data_subset * 1000 / 255;
                    elseif strcmp(Data_type, 'EU_DEM_Aspect')
                        data_subset = data_subset * 360 / 255;
                    elseif strcmp(Data_type, 'EU_DEM_Slope')
                        data_subset = acos(data_subset / 250) * 180/pi;
                    end
                    
                    % Append the data
                    Subdomain_Data{k} = data_subset;
                end
                                                
                % Progress
                send(DQ, k);
            end
            
            % Plot
            if strcmp(Plot, 'Yes')
                % The newly created files
                subset_file_names = sprintf('%s_*_%s.tif*', Data_type, Subset_name);
                subset_file_list = dir(subset_file_names);
                
                % The first file's data
                subset_file = subset_file_list(1).name;
                data_subset = geotiffread(subset_file);
                
                figure(222)

                sgtitle(sprintf('%s data of the %s subset', Data_type, Subset_name))
                imshow(data_subset)
                caxis([min(min(data_subset)), max(max(data_subset))])
                colorbar
            end
        end

        %%% Create new .mat files %%%
        if strcmp(file_type, 'mat')            
            % Growing season metrics
            if strcmp(Data_type, 'SoS') | strcmp(Data_type, 'EoS') | strcmp(Data_type, 'LoS')
                % Start of season
                if strcmp(Data_type, 'SoS')
                    SoS_file = load('SoS_data.mat');
                    GS_cell = SoS_file.SoS_cell;
                    t_GS_cell = SoS_file.t_SoS_cell;
                    
                % End of season
                elseif strcmp(Data_type, 'EoS')
                    EoS_file = load('EoS_data.mat');
                    GS_cell = EoS_file.EoS_cell;
                    t_GS_cell = EoS_file.t_EoS_cell;

                % Length of season
                elseif strcmp(Data_type, 'LoS')
                    LoS_file = load('LoS_data.mat');
                    GS_cell = LoS_file.LoS_cell;
                    t_GS_cell = LoS_file.t_LoS_cell;
                end
                
                % Select the subdomain of the data
                [rows_data, columns_data] = size(GS_cell);

                [rows, columns] = ind2sub([rows_data, columns_data], subset_indices);

                row_N = min(rows);
                row_S = max(rows);
                col_W = min(columns);
                col_E = max(columns);
                
                GS_cell = GS_cell(row_N : row_S, col_W : col_E);
                t_GS_cell = t_GS_cell(row_N : row_S, col_W : col_E);
                
                % Fill the undesired indices with the invalid value
                rows_subset = row_S - row_N + 1;
                columns_subset = col_E - col_W + 1;

                rows_remapped = rows - row_N + 1;
                columns_remapped = columns - col_W + 1;

                remapped_subset_indices = sub2ind([rows_subset, columns_subset], rows_remapped, columns_remapped);
            
                GS_subset_cell = cell(rows_subset, columns_subset);
                GS_subset_cell(remapped_subset_indices) = GS_cell(remapped_subset_indices);
                GS_cell = GS_subset_cell;
                
                t_GS_subset_cell = cell(rows_subset, columns_subset);
                t_GS_subset_cell(remapped_subset_indices) = t_GS_cell(remapped_subset_indices);
                t_GS_cell = t_GS_subset_cell;
                
                % Coarsen the resolution if need be
                if strcmp(Resolution_downscaling, 'On')
                    GS_cell = GS_cell(1: coarsening_factor : end, 1 : coarsening_factor : end);
                    t_GS_cell = t_GS_cell(1: coarsening_factor : end, 1 : coarsening_factor : end);
                end

                % The data
                Subdomain_Data = {GS_cell, t_GS_cell};
                
                % Size of the data
                [rows_subset, columns_subset] = size(GS_cell);
                
                % Number of years for which growing season data is available varies
                % number_files is given value 1, s.t. the rest of the code functions, but it is important to note that t_GS_cell should be used instead
                number_files = 1;
                
                % Confirmation that the data has been read
                fprintf('The %s growing season metrics data has been retrieved \n', Data_type);
                
            % Break point times
            elseif strcmp(Data_type, 'BP_Trend') | strcmp(Data_type, 'BP_Season')

                % Load the file and save the BP times
                if strcmp(Data_type, 'BP_Trend')
                    BP_file = load('Trend_BP_times.mat');
                elseif strcmp(Data_type, 'BP_Season')
                    BP_file = load('Seasonal_BP_times.mat');
                end

                BP_times_cell = BP_file.BP_times_cell;
                
                % Select the subdomain of the data
                [rows_data, columns_data] = size(BP_times_cell);

                [rows, columns] = ind2sub([rows_data, columns_data], subset_indices);

                row_N = min(rows);
                row_S = max(rows);
                col_W = min(columns);
                col_E = max(columns);
                
                BP_times_cell = BP_times_cell(row_N : row_S, col_W : col_E);
                
                % Fill the undesired indices with the invalid value
                rows_subset = row_S - row_N + 1;
                columns_subset = col_E - col_W + 1;

                rows_remapped = rows - row_N + 1;
                columns_remapped = columns - col_W + 1;

                remapped_subset_indices = sub2ind([rows_subset, columns_subset], rows_remapped, columns_remapped);
            
                BP_times_subset_cell = cell(rows_subset, columns_subset);
                BP_times_subset_cell(remapped_subset_indices) = BP_times_cell(remapped_subset_indices);
                BP_times_cell = BP_times_subset_cell;
                
                % Coarsen the resolution if need be
                if strcmp(Resolution_downscaling, 'On')
                    BP_times_cell = BP_times_cell(1: coarsening_factor : end, 1 : coarsening_factor : end);
                end

                % The data
                Subdomain_Data = {BP_times_cell};
                
                % Size of the data
                [rows_subset, columns_subset] = size(BP_times_cell);
                
                % The break point times themselves are in time format
                % number_files is given value 1, s.t. the rest of the code functions
                number_files = 1;
                
                % Confirmation that the data has been read
                fprintf('The %s break point times have been retrieved \n', Data_type);
                
                % Save the new data
                if strcmp(Data_type, 'BP_Trend') & strcmp(Save_Files, 'Yes')
                    BP_file = sprintf('Trend_BP_times_%s.mat', Subset_name);
                    save(BP_file, 'BP_times_cell');
                    
                    fprintf('The trend BP times have been saved as %s \n', BP_file);
                elseif strcmp(Data_type, 'BP_Season') & strcmp(Save_Files, 'Yes')
                    BP_file = sprintf('Seasonal_BP_times_%s.mat', Subset_name);
                    save(BP_file, 'BP_times_cell');
                    
                    fprintf('The seasonal BP times have been saved as %s \n', BP_file);
                end    
            % Decomposed data
            else
                Subdomain_Data = cell(1, 2);

                %--% Trend component %--%
                % Determine the binding rows and columns of the subset
                Trend_LB_name = sprintf('%s_T_LB*.mat', Data_type);
                Trend_LB_file = dir(Trend_LB_name);
                Trend_LB_file = load(Trend_LB_file(1).name);

                T_LB_matrix = Trend_LB_file.T_LB_matrix;

                [rows_data, columns_data] = size(T_LB_matrix);

                [rows, columns] = ind2sub([rows_data, columns_data], subset_indices);

                row_N = min(rows);
                row_S = max(rows);
                col_W = min(columns);
                col_E = max(columns);

                rows_subset = row_S - row_N + 1;
                columns_subset = col_E - col_W + 1;

                rows_remapped = rows - row_N + 1;
                columns_remapped = columns - col_W + 1;

                remapped_subset_indices = sub2ind([rows_subset, columns_subset], rows_remapped, columns_remapped);

                % Retrieve the data
                Trend_TS_name = sprintf('%s_Trend*.mat', Data_type);
                Trend_TS_file = dir(Trend_TS_name);
                Trend_TS_file = load(Trend_TS_file(1).name);

                Trend_cell = Trend_TS_file.Trend_cell(row_N : row_S, col_W : col_E);

                Trend_UB_name = sprintf('%s_T_UB*.mat', Data_type);
                Trend_UB_file = dir(Trend_UB_name);
                Trend_UB_file = load(Trend_UB_file(1).name);

                T_UB_matrix = Trend_UB_file.T_UB_matrix(row_N : row_S, col_W : col_E);

                T_LB_matrix = T_LB_matrix(row_N : row_S, col_W : col_E);

                fprintf('The %s trend component data has been retrieved \n', Data_type);

                % A function that fills unneeded cells with lists containig the invalid value
                number_files = length(Trend_cell{1});
                invalid_list_fun = @(x) uint8(invalid_value * ones(1, number_files));  

                % Trend component subsets
                Trend_subset_cell = cellfun(invalid_list_fun, cell(rows_subset, columns_subset), 'UniformOutput', false);
                Trend_subset_cell(remapped_subset_indices) = Trend_cell(remapped_subset_indices);
                Trend_cell = Trend_subset_cell;

                T_UB_subset_matrix = invalid_value * ones(rows_subset, columns_subset);
                T_UB_subset_matrix(remapped_subset_indices) = T_UB_matrix(remapped_subset_indices);
                T_UB_matrix = T_UB_subset_matrix;

                T_LB_subset_matrix = invalid_value * ones(rows_subset, columns_subset);
                T_LB_subset_matrix(remapped_subset_indices) = T_LB_matrix(remapped_subset_indices);
                T_LB_matrix = T_LB_subset_matrix;

                if strcmp(Resolution_downscaling, 'On')
                    Trend_cell = Trend_cell(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                    T_UB_matrix = T_UB_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                    T_LB_matrix = T_LB_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                end

                [rows_subset, columns_subset] = size(Trend_cell);

                if strcmp(Save_Files, 'Yes')
                    trend_file_name = sprintf('%s_Trend_%s.mat', Data_type, Subset_name);
                    save(trend_file_name, 'Trend_cell', '-v7.3');

                    trend_UB_file_name = sprintf('%s_T_UB_%s.mat', Data_type, Subset_name);
                    save(trend_UB_file_name, 'T_UB_matrix');

                    trend_LB_file_name = sprintf('%s_T_LB_%s.mat', Data_type, Subset_name);
                    save(trend_LB_file_name, 'T_LB_matrix');

                    fprintf('The %s subset of the %s trend component data has been created and saved \n', Subset_name, Data_type);

                else    % If the files are not created, the actual trend component is computed
                    % Parallel computing loop
                    DQ = parallel.pool.DataQueue;
                    tick = 0;
                    N = rows_subset * columns_subset;
                    afterEach(DQ, @ProgressUpdate_mat_trend);

                    parfor i = 1:rows_subset * columns_subset
                        % Convert into the right format
                        T = Trend_cell{i};
                        T = double(T);

                        T_ub = T_UB_matrix(i);
                        T_lb = T_LB_matrix(i);

                        T = T*(T_ub - T_lb)/255 + T_lb;

                        % Append to the cell arrays
                        Trend_cell{i} = T;

                        send(DQ, i);
                    end

                    Subdomain_Data{1} = Trend_cell;
                end

                % Plot to show the subset
                if strcmp(Plot, 'Yes')
                    trend_matrix = cellfun(@mean, Trend_cell);

                    figure(1)
                    imshow(trend_matrix)
                    sgtitle(sprintf('%s mean trend data of the %s subset', Data_type, Subset_name))
                    caxis([min(min(trend_matrix)), max(max(trend_matrix))])
                    colorbar
                end

                % Free memory
                clear Trend_cell & Trend_subset_cell & trend_matrix & T_UB_matrix & T_UB_subset_matrix & T_LB_matrix & T_LB_subset_matrix
                clear Trend_TS_file & Trend_UB_file & Trend_LB_file

                %--% Seasonal component %--%
                % Retrieve the seasonal component data
                Season_TS_name = sprintf('%s_Season*.mat', Data_type);
                Season_TS_file = dir(Season_TS_name);
                Season_TS_file = load(Season_TS_file(1).name);

                Season_cell = Season_TS_file.Season_cell(row_N : row_S, col_W : col_E);

                Season_UB_name = sprintf('%s_S_UB*.mat', Data_type);
                Season_UB_file = dir(Season_UB_name);
                Season_UB_file = load(Season_UB_file(1).name);

                S_UB_matrix = Season_UB_file.S_UB_matrix(row_N : row_S, col_W : col_E);

                Season_LB_name = sprintf('%s_S_LB*.mat', Data_type);
                Season_LB_file = dir(Season_LB_name);
                Season_LB_file = load(Season_LB_file(1).name);

                S_LB_matrix = Season_LB_file.S_LB_matrix(row_N : row_S, col_W : col_E);

                fprintf('The %s seasonal component data has been retrieved \n', Data_type);

                % Create subsets of the seasonal component data
                rows_subset = row_S - row_N + 1;
                columns_subset = col_E - col_W + 1;

                Season_subset_cell = cellfun(invalid_list_fun, cell(rows_subset, columns_subset), 'UniformOutput', false);
                Season_subset_cell(remapped_subset_indices) = Season_cell(remapped_subset_indices);
                Season_cell = Season_subset_cell;

                S_UB_subset_matrix = invalid_value * ones(rows_subset, columns_subset);
                S_UB_subset_matrix(remapped_subset_indices) = S_UB_matrix(remapped_subset_indices);
                S_UB_matrix = S_UB_subset_matrix;

                S_LB_subset_matrix = invalid_value * ones(rows_subset, columns_subset);
                S_LB_subset_matrix(remapped_subset_indices) = S_LB_matrix(remapped_subset_indices);
                S_LB_matrix = S_LB_subset_matrix;

                if strcmp(Resolution_downscaling, 'On')
                    Season_cell = Season_cell(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                    S_UB_matrix = S_UB_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                    S_LB_matrix = S_LB_matrix(1 : coarsening_factor : end, 1 : coarsening_factor : end);
                end

                [rows_subset, columns_subset] = size(Season_cell);

                if strcmp(Save_Files, 'Yes')
                    season_file_name = sprintf('%s_Season_%s.mat', Data_type, Subset_name);
                    save(season_file_name, 'Season_cell', '-v7.3');

                    season_UB_file_name = sprintf('%s_S_UB_%s.mat', Data_type, Subset_name);
                    save(season_UB_file_name, 'S_UB_matrix');

                    season_LB_file_name = sprintf('%s_S_LB_%s.mat', Data_type, Subset_name);
                    save(season_LB_file_name, 'S_LB_matrix');

                    fprintf('Subset %s of the %s seasonal component data has been created and saved \n', Subset_name, Data_type);

                else    % If the files are not created, the actual seasonal component is computed
                    % Parallel computing loop
                    DQ = parallel.pool.DataQueue;
                    tick = 0;
                    N = rows_subset * columns_subset;
                    afterEach(DQ, @ProgressUpdate_mat_season);

                    parfor i = 1:rows_subset * columns_subset
                        % Convert into the right format
                        S = Season_cell{i};
                        S = double(S);

                        S_ub = S_UB_matrix(i);
                        S_lb = S_LB_matrix(i);

                        S = S*(S_ub - S_lb)/255 + S_lb;

                        % Append to the cell arrays
                        Season_cell{i} = S;

                        send(DQ, i);
                    end

                    Subdomain_Data{2} = Season_cell;
                end

                % Plot to show the subset
                if strcmp(Plot, 'Yes')
                    season_matrix = cellfun(@mean, Season_cell);

                    figure(2)
                    imshow(season_matrix)
                    sgtitle(sprintf('%s mean seasonal data of the %s subset', Data_type, Subset_name))
                    caxis([min(min(season_matrix)), max(max(season_matrix))])
                    colorbar
                end

            end

            % The coordinate lists are empty, as they cannot be retrieved from .mat files
            longitude_list_subset = [];
            latitude_list_subset = [];
        end
        
    %%% The file date list %%%
        % The first date is assumed to be the first date this data type was available    
        if strcmp(Data_type, 'NDVI') | strcmp(Data_type, 'LAI') | strcmp(Data_type, 'FAPAR') | strcmp(Data_type, 'FCOVER') 
            Temporal_resolution = 365.25/36;        % Days per year / time stamps per year
            date_start = '01.01.1999';   
        elseif strcmp(Data_type, 'MOD13Q1')
            Temporal_resolution = 365.25/23;        % Days per year / time stamps per year
            date_start = '01.01.2001';
        elseif strcmp(Data_type, 'LST')
            Temporal_resolution = 365.25/36;        % Days per year / time stamps per year
            date_start = '01.01.2003';        
        elseif strcmp(Data_type, 'GPM')
            Temporal_resolution = 365.25/36;        % Days per year / time stamps per year
            date_start = '01.06.2000';        
        elseif strcmp(Data_type, 'CLC')
            Temporal_resolution = 6*365.25;         % 6 years per data set
            date_start = '01.01.2000';
        elseif strcmp(Data_type, 'SLC') | strcmp(Data_type, 'LC')
            Temporal_resolution = 365.25;           % Annual
            date_start = '01.01.1999';
        elseif strcmp(Data_type, 'Bumblebee')
            Temporal_resolution = 0;                % Single data set
            date_start = '01.06.2002';              % Data set covers 1991 - 2012, this is the average date covered
        elseif strcmp(Data_type, 'EU_DEM_Slope') | strcmp(Data_type, 'EU_DEM_Elevation') | strcmp(Data_type, 'EU_DEM_Aspect')
            Temporal_resolution = 0;                % Single data set
            date_start = '01.01.2014';              % Based on multiple data sets, but it was published in 2014
        elseif strcmp(Data_type, 'SoS') | strcmp(Data_type, 'EoS') | strcmp(Data_type, 'LoS')
            Temporal_resolution = 0;                % t_GS_cell should be used instead
            date_start = '01.01.1999';              % The times are dekad timestmaps w.r.t. to this date
        elseif strcmp(Data_type, 'BP_Trend') | strcmp(Data_type, 'BP_Season')
            Temporal_resolution = 0;                % They are themselves in time format
            date_start = '01.01.1999';              % The times are dekad timestamps w.r.t. to this date
        end

        date_time_start = datetime(date_start, 'InputFormat', 'dd.MM.yyyy');  

        file_date_list = cell(1, number_files);

        for k = 1:number_files
            % Current date is computed by adding the days since the start
            date_time = date_time_start + days((k - 1) * Temporal_resolution);
            date = datestr(date_time, 'dd.mm.yyyy');

            % Append the dates
            file_date_list{k} = date;
        end
        
    %%% Years the data covers %%%
        date_start = strsplit(file_date_list{1}, '.');
        year_start = str2double(date_start{3});

        date_end = strsplit(file_date_list{number_files}, '.');
        year_end = str2double(date_end{3});

        year_list = year_start : year_end;
        number_years = length(year_list);
        
    % Progress function
    function ProgressUpdate_tiff(~)
        tick = tick + 1;
        
        % Ensures that at most every 10 percent is printed
        percentage_increment = 10;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = round(tick / N * 100);
            
            fprintf('   Progress in determining the %s data for subset %s: %g%% \n', Data_type, Subset_name, progress);
        end
    end        

    % Progress function
    function ProgressUpdate_mat_trend(~)
        tick = tick + 1;
        
        % Ensures that at most every 10 percent is printed
        percentage_increment = 10;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = round(tick / N * 100);
            
            fprintf('   Progress in computing the %s trend for subset %s: %g%% \n', Data_type, Subset_name, progress);
        end
    end    

    % Progress function
    function ProgressUpdate_mat_season(~)
        tick = tick + 1;
        
        % Ensures that at most every 10 percent is printed
        percentage_increment = 10;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = round(tick / N * 100);
            
            fprintf('   Progress in computing the %s season for subset %s: %g%% \n', Data_type, Subset_name, progress);
        end
    end  
end
    

        
        
        
        
        
        
        
        
        
        
        