% This script divides the input data into a possibly overlapping raster of pixels

function [Indices_sections, Number_sections] = Raster_Segmentation(Section_size, Section_overlap, rows_data, columns_data)
    
    %%% Indices of the sections offset in x %%%
        % Offset due to overlap, equal in x and y
        offset = ceil((100 - Section_overlap)/100 * Section_size);  

        % The sections that are offset in x, with distinct edges in y
        Number_sections_offset_x = ceil(columns_data / offset);
        Number_sections_height_y = ceil(rows_data / Section_size);

        Number_sections_x = Number_sections_offset_x * Number_sections_height_y;
        
        ind_sections_x = cell(1, Number_sections_x); 

        for s = 1:Number_sections_x
            [s_y, s_x] = ind2sub([Number_sections_height_y, Number_sections_offset_x], s);

            % The min statements ensure that when the data is not fully divisible by the number of chunks, the final sections have the remaining members
            col_West_x = (s_x - 1) * offset + 1;
            col_East_x = min(col_West_x + Section_size - 1, columns_data);
            row_North_x = (s_y - 1) * Section_size + 1;
            row_South_x = min(s_y * Section_size, rows_data);

            % The indices of the section
            rows_section = row_North_x : row_South_x;
            columns_section = col_West_x : col_East_x;

            [rows_matrix, columns_matrix] = ndgrid(rows_section, columns_section);

            indices_section = sub2ind([rows_data, columns_data], rows_matrix, columns_matrix);

            ind_sections_x{s} = indices_section(:)';     % The indices are appended horizontally
        end

    %%% Indices of the sections offset in y %%%
        % The sections that are offset in y, with distinct edges in x        
        if Section_overlap == 0             % Note that if the overlap is 0%, the sectionss in y and x are identical
            Number_sections_offset_y = 0;
            Number_sections_width_x = 0;
            Number_sections_y = 0;
        else
            Number_sections_offset_y = ceil(rows_data / offset);            
            Number_sections_width_x = ceil(columns_data / Section_size); 
            Number_sections_y = Number_sections_offset_y * Number_sections_width_x;
        end
        
        ind_sections_y = cell(1, Number_sections_y); 

        for s = 1:Number_sections_y
            [s_y, s_x] = ind2sub([Number_sections_offset_y, Number_sections_width_x], s);

            % The min statements ensure that when the data is not fully divisible by the number of chunks, the final lists have the remaining members
            col_West_y = (s_x - 1) * Section_size + 1;
            col_East_y = min(s_x * Section_size, columns_data);
            row_North_y = (s_y - 1) * offset + 1;
            row_South_y = min(row_North_y + Section_size - 1, rows_data);

            % The indices of the section
            rows_section = row_North_y : row_South_y;
            columns_section = col_West_y : col_East_y;

            [rows_matrix, columns_matrix] = ndgrid(rows_section, columns_section);

            indices_section = sub2ind([rows_data, columns_data], rows_matrix, columns_matrix);

            ind_sections_y{s} = indices_section(:)';     % The indices are appended horizontally
        end

    %%% Combine the results for x and y %%%
        Number_sections = Number_sections_x + Number_sections_y;
        Indices_sections = [ind_sections_x, ind_sections_y];

        % Free memory
        clear ind_sections_x & ind_sections_y
end