% The function of this goal is to pack/unpack the data,
% to change it from image data to time series data (unpacking)
% or vice versa (packing)

% PackUnpack = [Pack/Unpack], TS > IMG or IMG > TS

function data = Data_Packer(data, number_files, columns_data, rows_data, PackUnpack)      

    %%% Inputs %%%
        memory_limit = 2000 * 2000;     % A slower, but less memory intensive method of unpacking is also available 
                                        % If the number of pixels is above this limit, it is used instead

    %%% Unpack the original data into time series for each pixel %%%
    if strcmp(PackUnpack, 'Unpack')
        % Create a data matrix
        if rows_data * columns_data < memory_limit      % Faster, but memory intensive option
            % Create a variable to hold the input data
            input_data = zeros(rows_data, columns_data, number_files);

            for k = 1:number_files
                input_data(:, :, k) = data{k};
            end

            % Unpack the data
            data = mat2cell(input_data, ones(1, rows_data), ones(1, columns_data), number_files);
            
        else                                            % Slower method that does not require 'input_data' to be specified
            
            data = cell2mat(data);
            data = reshape(data, rows_data, columns_data, number_files);        % Reshape becomes slow if the operation exceeds the cache size
            
            % Unpack the data
            data = mat2cell(data, ones(1, rows_data), ones(1, columns_data), number_files);
        end
            
        % Set the desired shape
        data = cellfun(@squeeze, data, 'UniformOutput', false);
        data = cellfun(@transpose, data, 'UniformOutput', false);   
        
    %%% Pack the time series for each pixel into the original data format %%%
    elseif strcmp(PackUnpack, 'Pack')
        % Create a data matrix
        data = cat(1, data{:});
        data = reshape(data, [rows_data, columns_data, number_files]);
        
        data = mat2cell(data, rows_data, columns_data, ones(1, number_files));
        data = (squeeze(data))';
        
    %%% In case the pack/unpack string was not understood %%%
    else
        disp('Error: The packing/unpacking commend was not understood.')
    end
end