% This script defines superpixels based on the simple linear iterative clustering (SLIC) approach outlined in "SLIC Superpixels Compared to State-of-the-Art Superpixel Methods (2012)" [A331]
% As it is a modified, more complicated version, it is instead called CLIC where the C stands for complicated

% feature_data can be a cell of size {1, number_features}, where each cell entry is [rows_data, columns_data]
% Alternatively, it can also be of shape [rows_data, columns_data, number_features]

% This script is suitable for parallel computing
function [superpixel_indices_cell, number_superpixels, superpixel_feature_data_cell] = CLIC(feature_data, number_features, rows_data, columns_data, superpixel_size)
    %%% Inputs %%%
        gradient_window                 = 01;       % Single-sided window width of the area for which the lowest gradient is found to set the initial clusters, 1 is the default
        
        convergence_threshold           = 1e-2;     % The allowed maximum distance between the current superpixel vectors and the last  
        max_iterations                  = 0003;     % The maximum number of iterations, 10 is commonly used in case the convergence threshold is not met
        
        compactness                     = 2.00;     % The higher the compactness, the more strongly spatial distance is considered w.r.t. the distance in terms of features
                                                    % If it is 1, they are equally weighted
                                                                 
        Connectivity                    = 'Yes';    % [Yes, No], can be toggled to improve connectivity of the superpixels
                                                    % Please note however that if the input data is noisy, connectivity can not be guaranteed

        % Messages
        if strcmp(Connectivity, 'Yes')
            fprintf('CLIC is used to divide the data into superpixels with compactness %s, and taking connectivity into account \n', num2str(compactness));
        else
            fprintf('CLIC is used to divide the data into superpixels with compactness %s, and not taking connectivity into account \n', num2str(compactness));            
        end

    %%% Convert the feature data to matrix format and normalise it %%%
        % If the feature data is in the cell format, it is put into matrix format
        if iscell(feature_data)
            feature_data = cell2mat(feature_data);
            feature_data = reshape(feature_data, [rows_data, columns_data, number_features]);
        end

        % The feature data is converted to doubles, in case it is in integer format
        feature_data = double(feature_data);
        
        % The feature data is normalised
        lb_list = zeros(1, number_features);
        ub_list = zeros(1, number_features);
        
        for f = 1 : number_features
            feature = feature_data(:, :, f);
                        
            % The limits of this feature data
            lb = min(min(feature));
            lb_list(f) = lb;
            ub = max(max(feature));
            ub_list(f) = ub;
            
            feature_data(:, :, f) = (feature - lb)/(ub - lb);
        end

    %%% Initial superpixel centres %%%
        % The number of superpixels is roughly equivalent to the number of pixels divided by the superpixel size squared
        % This assumes that the superpixels are square in shape, but is a good first estimate
        number_superpixels_x = round(columns_data / superpixel_size);
        number_superpixels_y = round(rows_data / superpixel_size);
        number_superpixels = number_superpixels_x * number_superpixels_y;

        % Initial superpixel centres are placed at the lowest gradient near each initial superpixel centre estimate
        superpixel_centres = zeros(number_superpixels_x, number_superpixels_y, 2);
        superpixel_feature_data_cell = cell(number_superpixels_x, number_superpixels_y);

        for s_x = 1 : number_superpixels_x
            % Location of this superpixel, in x and y
            x = 0.5*superpixel_size + (s_x - 1)*superpixel_size;
            x = round(x);

            for s_y = 1 : number_superpixels_y
                y = 0.5*superpixel_size + (s_y - 1)*superpixel_size;
                y = round(y);
                
                % The feature data in the pixel's neighbourhood
                y_N = max(1, y - gradient_window);
                y_S = min(rows_data, y + gradient_window);
                x_W = max(1, x - gradient_window);
                x_E = min(columns_data, x + gradient_window);
                
                feature_data_neighbourhood = feature_data(y_N : y_S, x_W : x_E, :);
                
                % The gradient in this neighbourhood
                gradient_neighbourhood = gradient(feature_data_neighbourhood);
                
                % The minimum of the total gradient is used
                total_gradient_neighbourhood = sum(gradient_neighbourhood, 3);
                [lowest_gradient_y, lowest_gradient_x] = find(total_gradient_neighbourhood == min(min(total_gradient_neighbourhood)), 1);
                
                % The centre of this superpixel in x and y
                centre_x = x + lowest_gradient_x - (gradient_window + 1);
                centre_y = y + lowest_gradient_y - (gradient_window + 1);
                
                % The location of this superpixel is saved
                superpixel_centres(s_x, s_y, 1) = centre_x;
                superpixel_centres(s_x, s_y, 2) = centre_y;
                
                % The feature data of this superpixel
                feature_data_superpixel = feature_data(centre_y, centre_x, :);
                
                superpixel_feature_data_cell{s_x, s_y} = squeeze(feature_data_superpixel)';
            end
        end
    
    %%% Iterative superpixel centre loop %%%
        % The feature data matrix is reshaped
        feature_data = reshape(feature_data, [rows_data * columns_data, number_features]);
    
        convergence = false;
        iteration = 0;
    
        while convergence == false & iteration < max_iterations
            iteration = iteration + 1;
            
            %--% Determine the closest superpixel for each pixel %--%
            superpixel_matrix = zeros(rows_data, columns_data);
            
            % Parallel loop
            DQ = parallel.pool.DataQueue;
            tick = 0;
            N = rows_data * columns_data;
            afterEach(DQ, @ProgressUpdate_SP);
            
            parfor i = 1 : rows_data * columns_data
                % The feature data of this pixel
                feature_data_pixel = feature_data(i, :);
                
                % Its location
                [y, x] = ind2sub([rows_data, columns_data], i);

                % The distance of this pixel to the superpixels
                distance_superpixels_matrix = inf(number_superpixels_x, number_superpixels_y);

                for s_x = 1 : number_superpixels_x
                    for s_y = 1 : number_superpixels_y
                        % This superpixel's location
                        centre_x = superpixel_centres(s_x, s_y, 1);
                        centre_y = superpixel_centres(s_x, s_y, 2);  

                        % Superpixels that are outside twice the superpixel size are skipped
                        if abs(centre_x - x) > 2 * superpixel_size | abs(centre_y - y) > 2 * superpixel_size
                            continue
                        end

                        % This superpixel's feature data
                        superpixel_feature_data = superpixel_feature_data_cell{s_x, s_y};

                        % The distance to this superpixel
                        distance_location = sqrt(((centre_x - x)/superpixel_size)^2 + ((centre_y - y)/superpixel_size)^2);      % It is normalised w.r.t. the superpixel size    
                        distance_features = sqrt(sum((superpixel_feature_data - feature_data_pixel).^2));                       % The feature data is already normalised

                        difference_total = sqrt(distance_features^2 + distance_location^2 * compactness^2);

                        distance_superpixels_matrix(s_x, s_y) = difference_total;
                    end
                end
                    
                % The closest superpixel
                [nearest_superpixel_x, nearest_superpixel_y] = find(distance_superpixels_matrix == min(min(distance_superpixels_matrix)), 1);

                superpixel_matrix(i) = (nearest_superpixel_y - 1) * number_superpixels_x + nearest_superpixel_x;
            end  
            
            %--% The new superpixel centres are computed %--%
            new_superpixel_centres = zeros(number_superpixels_x, number_superpixels_y, 2);
            new_superpixel_feature_data_cell = cell(number_superpixels_x, number_superpixels_y);     

            for s_x = 1 : number_superpixels_x
                for s_y = 1 : number_superpixels_y                        
                    % The pixel locations within this superpixel
                    s = (s_y - 1) * number_superpixels_x + s_x;
                    pixels_ind = find(superpixel_matrix == s);

                    % The average location is used
                    [pixels_y, pixels_x] = ind2sub([rows_data, columns_data], pixels_ind);
                    new_superpixel_centres(s_x, s_y, 1) = mean(pixels_x);
                    new_superpixel_centres(s_x, s_y, 2) = mean(pixels_y);
                    
                    % The average feature data is used
                    new_superpixel_feature_data = zeros(1, number_features);
                    
                    for f = 1 : number_features
                        feature = feature_data(:, f);
                        feature_superpixel = feature(pixels_ind);
                        
                        new_superpixel_feature_data(f) = mean(feature_superpixel);
                    end

                    new_superpixel_feature_data_cell{s_x, s_y} = new_superpixel_feature_data;
                end
            end
            
            %--% Check for convergence %--%
            difference_superpixels = zeros(number_superpixels_x, number_superpixels_y);
            
            for s_x = 1 : number_superpixels_x
                for s_y = 1 : number_superpixels_y
                    % This superpixel's (new and old) feature data
                    new_superpixel_feature_data = new_superpixel_feature_data_cell{s_x, s_y};
                    superpixel_feature_data = superpixel_feature_data_cell{s_x, s_y};
                    
                    % This superpixel's (new and old) location
                    new_superpixel_x = new_superpixel_centres(s_x, s_y, 1);
                    new_superpixel_y = new_superpixel_centres(s_x, s_y, 2);
                    
                    superpixel_x = superpixel_centres(s_x, s_y, 1);
                    superpixel_y = superpixel_centres(s_x, s_y, 2);
                    
                    % The difference between the new and old superpixels
                    difference_location = sqrt(((new_superpixel_x - superpixel_x)/superpixel_size)^2 + ((new_superpixel_y - superpixel_y)/superpixel_size)^2);
                    difference_features = sqrt(sum(new_superpixel_feature_data - superpixel_feature_data).^2);
                    
                    difference = sqrt(difference_features^2 + difference_location^2 * compactness^2);
                    
                    difference_superpixels(s_x, s_y) = difference;
                end
            end
            
            % The maximum difference is used to check whether the convergence threshold is met
            if max(max(difference_superpixels)) < convergence_threshold
                convergence = true;
            end
            
            % If the convergence data is not met, the locations are updated
            superpixel_feature_data_cell = new_superpixel_feature_data_cell;
            superpixel_centres = new_superpixel_centres;            
        end
        
    %%% Disconnected regions are appended to the nearest superpixel %%%        
        if strcmp(Connectivity, 'Yes')            
            % Before the connectivity is determined, it is smoothed to remove tiny irregularities and greatly speed up the adjacency testing
            window_width = 3;       % double-sided window width

            superpixel_matrix_smoothed = medfilt2(superpixel_matrix, [window_width window_width]);
            superpixel_matrix_smoothed = round(superpixel_matrix_smoothed);

            % Zero entries are replaced to avoid false entries
            zero_ind = superpixel_matrix_smoothed == 0;
            superpixel_matrix_smoothed(zero_ind) = superpixel_matrix(zero_ind);
            superpixel_matrix = superpixel_matrix_smoothed;

            % Find the regions within the superpixel matrix
            regions_indices_cell = cell(1, number_superpixels);

            for s = 1 : number_superpixels
                % The pixel locations within this superpixel
                pixels_ind = superpixel_matrix == s;         

                % Separate, connected regions of this superpixel are found
                superpixel_regions = bwconncomp(pixels_ind, 8);     % 8-connectivity means that corners are considered as well

                % The information of the regions is saved
                regions_indices = superpixel_regions.PixelIdxList;

                % The largest region is however removed, and this one will be skipped
                regions_area = cellfun(@length, regions_indices);
                [~, largest_region] = max(regions_area);
                regions_indices(largest_region) = [];

                regions_indices_cell{s} = regions_indices;
            end

            regions_indices_cell = horzcat(regions_indices_cell{:});
            number_regions = length(regions_indices_cell);

            % The feature data of each region
            regions_feature_data_cell = cell(1, number_regions);

            for r = 1 : number_regions
                region_indices = regions_indices_cell{r};

                feature_data_region = feature_data(region_indices, :);

                regions_feature_data_cell{r} = feature_data_region;
            end

            % Cycle through the smaller regions, in order of size
            region_size_list = cellfun(@length, regions_indices_cell);
            [~, order] = sort(region_size_list, 'descend');

            regions_indices_cell = regions_indices_cell(order);

            regions_superpixel_list = zeros(1, number_regions);

            % Parallel loop
            DQ = parallel.pool.DataQueue;
            tick = 0;
            N = number_regions;
            afterEach(DQ, @ProgressUpdate_adjacency);

            parfor r = 1 : number_regions
                % This region's indices
                region_indices = regions_indices_cell{r};

                % A temporary matrix that contains this region's indices
                superpixel_matrix_region = superpixel_matrix;
                superpixel_matrix_region(region_indices) = number_superpixels + 1;

                % Determine adjacency                    
                indices_N = max(region_indices - 1, 1);
                indices_S = min(region_indices + 1, rows_data * columns_data);
                indices_W = max(region_indices - rows_data, 1);
                indices_E = min(region_indices + rows_data, rows_data * columns_data);

                indices_neighbours = [indices_N; indices_S; indices_W; indices_E];
                adjacent_pixels = superpixel_matrix_region(indices_neighbours);

                % The most commonly adjacent pixel that isn't its own region or zero
                adjacent_pixels(adjacent_pixels == number_superpixels + 1) = [];
                adjacent_pixels(adjacent_pixels == 0) = [];

                % The number of times each superpixel is adjacent  
                adjacent_superpixels = unique(adjacent_pixels);
                number_adjacent_pixels = length(adjacent_superpixels);

                adjacent_pixel_counts = histcounts(adjacent_pixels, number_adjacent_pixels);

                adjacent_pixel_counts = adjacent_pixel_counts / max(adjacent_pixel_counts);     % Normalised

                % Determine the difference in feature data to the adjacent superpixels
                difference_superpixels = zeros(1, number_adjacent_pixels);

                if number_adjacent_pixels > 1
                    % This region's feature data
                    region_feature_data = regions_feature_data_cell{r};
                    region_feature_data = mean(region_feature_data, 1);

                    for s = 1 : number_adjacent_pixels
                        % The adjacent superpixel's feature data
                        superpixel = adjacent_superpixels(s);                        
                        superpixel_feature_data = superpixel_feature_data_cell{superpixel};

                        % The distance in feature space
                        difference = sum(sqrt((superpixel_feature_data - region_feature_data).^2)) / number_features;
                        difference_superpixels(s) = difference;
                    end
                end

                % The optimal superpixel is at a minimum distance and has the most adjacent superpixels
                difference_total = difference_superpixels - adjacent_pixel_counts;

                [~, ind] = min(difference_total);                
                superpixel = adjacent_superpixels(ind);

                % The region is given this superpixel's value
                regions_superpixel_list(r) = superpixel(1);

                % Progress update
                send(DQ, r);
            end

            % Change the superpixel matrix
            for r = 1 : number_regions
                region_indices = regions_indices_cell{r};
                superpixel = regions_superpixel_list(r);

                superpixel_matrix(region_indices) = superpixel;
            end
        end
        
    %%% The superpixel feature data %%%   
        superpixel_indices_cell = label2idx(superpixel_matrix);
    
        % The cell which contains the mean feature data of each superpixel
        for s = 1 : number_superpixels
            superpixel_ind = superpixel_indices_cell{s};

            % The average feature data is used
            superpixel_feature_data = zeros(1, number_features);

            for f = 1 : number_features
                feature = feature_data(:, f);
                feature_superpixel = feature(superpixel_ind);
                feature_superpixel = mean(feature_superpixel);

                % The data is reverted to the original format
                lb = lb_list(f);
                ub = ub_list(f);

                feature_superpixel = feature_superpixel * (ub - lb) + lb;

                superpixel_feature_data(f) = feature_superpixel;
            end

            superpixel_feature_data_cell{s} = superpixel_feature_data;
        end
        
    %%% A plot of the superpixels %%%
        % The colourmap
        try
            SP_cmap = cbrewer('qual', 'Set1', number_superpixels);
        catch
            SP_cmap = colormap(jet(number_superpixels));
        end
        
        figure(555)
        
        contourf(superpixel_matrix, 1:number_superpixels, 'LineWidth', 3)
        set(gca, 'YDir', 'reverse')
        axis off
        
        % Set the size and white background color
        set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1])
        set(gcf, 'color', [1, 1, 1])
                
        colormap(SP_cmap);
        caxis([1, number_superpixels]);
        
        if number_superpixels < 20
            cb = colorbar;
            cb.Ticks = linspace(1.5, number_superpixels - 0.5, number_superpixels);
            cb.TickLabels = string(1 : number_superpixels);
            cb.Title.String = 'Superpixel';
            cb.FontSize = 15;
        end
        
        if strcmp(Connectivity, 'Yes')
            figure_name = sprintf('CLIC_connected_%s_SP_%s_compactness.png', num2str(number_superpixels), num2str(compactness));
        else
            figure_name = sprintf('CLIC_%s_SP_%s_compactness.png', num2str(number_superpixels), num2str(compactness));
        end
        
        try
            export_fig(555, figure_name);
        catch
            frame = getframe(555);
            im = frame2im(frame);
            [imind, cm] = rgb2ind(im, 256);
            imwrite(imind, cm, figure_name);
        end
        
        close(555);
        
        fprintf('The superpixels have been determined and are shown in %s \n', figure_name);
        
    % Progress function for determining the first superpixels
    function ProgressUpdate_SP(~)
        tick = tick + 1;
        
        % Ensures that at most every 25 percent is printed
        percentage_increment = 25;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = tick / N * 100;
            
            fprintf('   Progress in finding the closest superpixels for iteration %g: %g%% \n', iteration, round(progress));
        end
    end

    % Progress function for ensuring adjacency
    function ProgressUpdate_adjacency(~)
        tick = tick + 1;
        
        % Ensures that at most every 5 percent is printed
        percentage_increment = 5;
        fraction_list = (percentage_increment : percentage_increment : 100) / 100;
        
        tick_list = round(N * fraction_list);
        
        if ismember(tick, tick_list)
            progress = tick / N * 100;
            
            fprintf('   Progress in ensuring the superpixels are adjacent: %g%% \n', round(progress));
        end
    end
end