% The aim of this function is to deal with temporal discontinuities in the classification results and create new smoothed files, called SLC_YYYY.tif

% This file is suited for parallel computing
function Temporal_Classification_Smoothing_P(LC_data_cell, R_CLC, Data_source, rows_data, columns_data, year_list, number_years, chunk_string)
    %%% Inputs %%%
        % Substition parameters
        minimum_class_length = 5;   % Minimum successive amount of years that a class must cover
        window_width = 2;           % Single-sided window width used for substition       

    %%% Create smoothed classification results %%%
        Original_Classification_TS_cell = Data_Packer(LC_data_cell, number_years, columns_data, rows_data, 'Unpack');
        Smoothed_Classification_TS_cell = cell(rows_data, columns_data);

        % Parallel computing loop
        DQ = parallel.pool.DataQueue;
        tick = 0;
        N = rows_data * columns_data;
        afterEach(DQ, @ProgressUpdate);
        
        parfor i = 1 : rows_data * columns_data
            classification_TS = Original_Classification_TS_cell{i};

            % First iteration
            majority_class = mode(classification_TS);

            [~, ~, class_indices] = unique(classification_TS);
            class_indices = class_indices';

            diff_indices = diff(class_indices);
            change_indices = find(diff_indices ~= 0) + 1;
            change_indices = [1, change_indices, number_years + 1];

            % Determine the indices to be substituted
            ind_sub_list = [];

            for c = 1 : length(change_indices) - 1
                length_change = change_indices(c + 1) - change_indices(c);

                LC_class = classification_TS(change_indices(c));

                if length_change < minimum_class_length & LC_class ~= majority_class        % The minimum class length has to be respected, and the class must not be the majority within the time series
                    ind_sub = change_indices(c) : change_indices(c + 1) - 1;
                    ind_sub_list = [ind_sub_list, ind_sub];
                end
            end

            LC_sub_list = zeros(1, length(ind_sub_list));

            for s = 1:length(ind_sub_list)
                ind_sub = ind_sub_list(s);

                first_index = max(1, ind_sub - window_width);
                last_index = min(number_years, ind_sub + window_width);

                LC_window = classification_TS(first_index : last_index);

                LC_sub = mode(LC_window);
                LC_sub_list(s) = LC_sub;
            end

            classification_TS(ind_sub_list) = LC_sub_list;

            % As a second iteration, in case the minimum class length is not respected everywhere
            majority_class = mode(classification_TS);

            [~, ~, class_indices] = unique(classification_TS);
            class_indices = class_indices';

            diff_indices = diff(class_indices);
            change_indices = find(diff_indices ~= 0) + 1;
            change_indices = [1, change_indices, number_years + 1];

            % Substitute short arrays with the majority class
            for c = 1 : length(change_indices) - 1
                length_change = change_indices(c + 1) - change_indices(c);

                LC_class = classification_TS(change_indices(c));

                if length_change < minimum_class_length & LC_class ~= majority_class
                    ind_sub = change_indices(c) : change_indices(c + 1) - 1;
                    classification_TS(ind_sub) = majority_class;
                end
            end
			
            % Smoothed result
            Smoothed_Classification_TS_cell{i} = classification_TS;
            
            % Update the progress bar
            send(DQ, i);
        end

        % Pack the smoothed results into land cover maps again
        Smoothed_Classification_IMG_cell = Data_Packer(Smoothed_Classification_TS_cell, number_years, columns_data, rows_data, 'Pack');

    %%% Create new land cover files %%%      
        R_land_cover = R_CLC;
        R_land_cover.RasterSize = [rows_data, columns_data];   % The size is altered, in case the data was coarsened
    
        for y = 1:number_years
            Smoothed_Classification_Results = Smoothed_Classification_IMG_cell{y};

            % Create a new .tif file
            year = year_list(y);

            SLC_file_name = sprintf('SLC_%s_%s', num2str(year), chunk_string);
            
            if strcmp(Data_source, 'Copernicus')
                geotiffwrite(SLC_file_name, uint16(Smoothed_Classification_Results), R_land_cover);
            elseif strcmp(Data_source, 'MODIS')
                Key_file = load('MODIS_NDVI_Geokey.mat');
                Tag = Key_file.Tag;

                geotiffwrite(SLC_file_name, uint16(Smoothed_Classification_Results), R_land_cover, 'GeoKeyDirectoryTag', Tag);
            end

            fprintf('The file %s has been created successfully. \n', SLC_file_name)
        end
        
    % Progress function
    function ProgressUpdate(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Temporal classification smoothing progress: %g%% \n', round(progress));
        end
    end
end



    
    