% This is the main file for land cover classification

clear all
close all
clc

%%% Required files %%%
    % TYPE              NUMBER              NAME
    % ---------------------------------------------------------------------------------------------------------------
    
    %                     Input data, any combination of the following
    % NDVI data:        number_files        NDVI_YYYYMMDD.tiff
    % NDVI comp. data:  6                   NDVI_Trend*.mat, NDVI_Season*.mat, NDVI_T_UB*.mat, NDVI_T_LB*.mat, NDVI_S_UB*.mat, NDVI_S_LB*.mat
    
    % LAI data:         number_files        LAI_YYYYMMDD.tiff
    % FAPAR data:       number_files        FAPAR_YYYYMMDD.tiff
    % Fcover data:      number_files        Fcover_YYYYMMDD.tiff
    
    % Spectral data:    number_bands        LS-B*.mat
    
    %                     Corine reference land cover data and classes
    % CLC files:        1-4                 CLC_YYYY_REGION.tif
    % Grouped classes:  1                   Grouped_Classes*.xls

%%% Inputs %%%
    %--% Input data %--%
        % Resolution downscaling should always be 'Off' if the native resolution is desired.
        Resolution_downscaling  = 'Off';        % [On/Off]
        coarsening_factor 		= 02;           % Integer value
        
        % The desired input data types [Yes/No]

        % NDVI (Copernicus / MODIS)
        NDVI_Data               = 'Yes';        % The pre-processed NDVI data
        NDVI_Component_Data     = 'No';        % The decomposed NDVI data

        % Vegetation properties (Copernicus)
        LAI_Data                = 'No';        % The leaf area index data
        FAPAR_Data              = 'No';        % The fraction of absorbed photosynthetically active radiation data
        Fcover_Data             = 'No';        % The fraction of vegetation cover data

        % Spectral data (Landsat)
        Blue_band               = 'No';        % Blue
        Green_band              = 'No';        % Green
        Red_band                = 'No';        % Red
        NIR_band                = 'No';        % Near infrared
        SWIR_band               = 'No';        % Shortwave infrared
        SWIR2_band              = 'No';        % Shortwave infrared (better at night)
        LGTH_band               = 'No';        % Low gain thermal
        HGTH_band               = 'No';        % High gain thermal
        Panchr_band             = 'No';        % Panchromatic band
        
        Spectral_Data = {Blue_band, Green_band, Red_band, NIR_band, SWIR_band, SWIR2_band, LGTH_band, HGTH_band, Panchr_band};      % All the spectral data choices

        Data_choices = {NDVI_Data, NDVI_Component_Data, LAI_Data, FAPAR_Data, Fcover_Data, Spectral_Data};                          % All data choices
        
    %--% Post-processing %--%
        % Whether the produced land cover should be temporally smoothed or not after it is created
        Smoothing               = 'On';            % [On/Off]
        
    %--% Land cover maps %--%
        % The Corine colormap is prettier, but less distinguishable
        Color_Choice            = 'Custom';         % [Corine, Colorbrewer, Custom]   

    %--% Division of the data into chunks %--%
        % The size of the data set that is classified at any given time can be reduced by dividing it into chunks in x and y
        number_chunks_x         = 02;
        number_chunks_y         = 02;

        % The partitions of the chunks can also be optimised s.t. they are better suited for classification
        Cluster_Optimisation    = 'On';             % [On/Off]

        % Whether the produced land cover should be merged or not, in case chunks are created
        Merging                 = 'On';        % [On, Off]

        % The files for the specific chunks can be deleted if desired, and if the chunks are merged
        chunk_deletion          = 'On';         % [On/Off]

%%% Start the parallel pool %%%
    try
        number_cores = 8;
        parpool('local', number_cores, 'IdleTimeout', 500);
    catch
        disp('Parallel pool was already running, or could not be started for other reasons.')
    end
    
%%% Classification %%%
    % Determine the size of the data and divide it into chunks
    [rows_chunks, columns_chunks, rows_data, columns_data, R_CLC, zero_string, Data_source] = Smart_Clusterer(Cluster_Optimisation, Data_choices, number_chunks_x, number_chunks_y);

    % Each chunk is classified independently
    for c_x = 1 : number_chunks_x
        
    % Bounds of this chunk
    column_W = columns_chunks(c_x) + 1;
    column_E = columns_chunks(c_x + 1);
                
        for c_y = 1 : number_chunks_y
            
        row_N = rows_chunks(c_y) + 1;
        row_S = rows_chunks(c_y + 1);

        % String to identify this chunk, and the data produced with it        
        s_x = num2str(c_x, zero_string);
        s_y = num2str(c_y, zero_string);

        chunk_string = sprintf('x%sy%s', s_x, s_y);
        
        %--% Data reading and converting to time series format %--%
        tic;

        % NDVI time series
        if strcmp(NDVI_Data, 'Yes')
            % Copernicus
            if strcmp(Data_source, 'Copernicus')
                [NDVI_IMG_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = Copernicus_NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
            elseif strcmp(Data_source, 'MODIS')
                [NDVI_IMG_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = MODIS_NDVI_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
            end
            
            NDVI_TS_cell = Data_Packer(NDVI_IMG_cell, number_files, columns_chunk, rows_chunk, 'Unpack');
            clear NDVI_IMG_cell
        else
            NDVI_TS_cell = {};
        end

        % NDVI trend and seasonal component data
        if strcmp(NDVI_Component_Data, 'Yes')
            if strcmp(Data_source, 'Copernicus')
                Data_prefix = 'NDVI';
            elseif strcmp(Data_source, 'MODIS')
                Data_prefix = 'MOD13Q1';
            end
            
            [NDVI_Trend_cell, NDVI_Season_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = Decomposition_Data_Reader(Data_prefix, Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
        else
            NDVI_Trend_cell = {};
            NDVI_Season_cell = {};
        end
        
        % LAI data
        if strcmp(LAI_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
            [LAI_IMG_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = LAI_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
            LAI_TS_cell = Data_Packer(LAI_IMG_cell, number_files, columns_chunk, rows_chunk, 'Unpack');
            clear LAI_IMG_cell
        else
            LAI_TS_cell = {};
        end

        % FAPAR data
        if strcmp(FAPAR_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
            [FAPAR_IMG_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = FAPAR_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
            FAPAR_TS_cell = Data_Packer(FAPAR_IMG_cell, number_files, columns_chunk, rows_chunk, 'Unpack');
            clear FAPAR_IMG_cell
        else
            FAPAR_TS_cell = {};
        end

        % Fcover data
        if strcmp(Fcover_Data, 'Yes') & strcmp(Data_source, 'Copernicus')
            [Fcover_IMG_cell, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = Fcover_Data_Reader_P(Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
            Fcover_TS_cell = Data_Packer(Fcover_IMG_cell, number_files, columns_chunk, rows_chunk, 'Unpack');
            clear Fcover_IMG_cell
        else
            Fcover_TS_cell = {};
        end
        
        % Spectral data
        number_bands = length(Spectral_Data);
        Spectral_Fourier_Coefficients_cell = cell(1, number_bands);

        if strcmp(Data_source, 'Landsat')    
            Landsat_bands = {'B1', 'B2', 'B3', 'B4', 'B5', 'B7', 'B6-VCID-1', 'B6-VCID-2', 'B8'};
            
            % Retrieve each band's Fourier series
            for b = 1 : number_bands
                if strcmp(Spectral_Data{b}, 'Yes')
                    Landsat_band = Landsat_bands{b};
                    
                    [Landsat_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_chunk, columns_chunk, number_files, ~, ~, year_list] = Landsat_Fourier_Coefficients_Retriever(Landsat_band, Resolution_downscaling, coarsening_factor, row_N, row_S, column_W, column_E);
                
                    Spectral_Fourier_Coefficients_cell{b} = Landsat_Fourier_Coefficients_cell;
                end
            end
        else
            Fourier_series_length = [];
            Landsat_last_time = [];
        end

        t_data = toc;

        fprintf('   Reading the data of chunk %s took %g seconds \n', chunk_string, t_data);
    
        %--% Classification %--%
        tic;
        [LC_data_cell, classified_year_list, number_classified_years, number_classes, class_values, class_names, RGB_codes] = Annual_Classifier_P(Resolution_downscaling, coarsening_factor, Data_choices, Data_source, NDVI_TS_cell, NDVI_Trend_cell, NDVI_Season_cell, LAI_TS_cell, FAPAR_TS_cell, Fcover_TS_cell, Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_chunk, columns_chunk, year_list, chunk_string, Color_Choice, R_CLC, row_N, row_S, column_W, column_E);
        t_classification = toc;

        fprintf('   Classifying the land cover data of chunk %s took %g seconds \n', chunk_string, t_classification);

        %--% Land cover smoothing %--% 
        if strcmp(Smoothing, 'On')
            tic;                        
            Temporal_Classification_Smoothing_P(LC_data_cell, R_CLC, Data_source, rows_chunk, columns_chunk, classified_year_list, number_classified_years, chunk_string);
            t_smoothing = toc;

            fprintf('   Creating smoothed classification results of chunk %s took %g seconds \n', chunk_string, t_smoothing);
        end
        
        % Free memory
        clear LC_data_cell & NDVI_TS_cell & NDVI_Trend_cell & NDVI_Season_cell & LAI_TS_cell & FAPAR_TS_cell & Fcover_TS_cell
        
        end
    end
    
%%% The data is merged %%%    
    if strcmp(Merging, 'On')
        tic;

        % The non-smoothed land cover is merged
        classified_year_list = string(classified_year_list);

        file_format = 'geotiff';
        Plot = 'On';

        Data_Merger_P('LC', Data_source, R_CLC, Resolution_downscaling, coarsening_factor, number_chunks_x, number_chunks_y, rows_chunks, columns_chunks, zero_string, file_format, number_classified_years, classified_year_list, Plot, RGB_codes, class_values, chunk_deletion);

        % The smoothed land cover is merged
        if strcmp(Smoothing, 'On')
            Data_Merger_P('SLC', Data_source, R_CLC, Resolution_downscaling, coarsening_factor, number_chunks_x, number_chunks_y, rows_chunks, columns_chunks, zero_string, file_format, number_classified_years, classified_year_list, Plot, RGB_codes, class_values, chunk_deletion);
        end

        % The confusion matrices are merged
        Confusion_Matrix_Merger(number_chunks_x, number_chunks_y, class_names, number_classes, zero_string, chunk_deletion);

        t_merging = toc;

        fprintf('   Merging the data took %g seconds \n', t_merging);
    end
    
    