% The aim of this file is to handle the inputs and outputs for the classification file, Decision_Tree_Classifier.m

% This file makes use of parallel computing
function [LC_data_cell, classified_year_list, number_classified_years, number_classes, class_values, class_names, RGB_codes] = Annual_Classifier_P(Resolution_downscaling, coarsening_factor, Data_choices, Data_source, NDVI_TS_cell, NDVI_Trend_cell, NDVI_Season_cell, LAI_TS_cell, FAPAR_TS_cell, Fcover_TS_cell, Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, rows_data, columns_data, year_list, chunk_string, Color_Choice, R_CLC, row_N, row_S, col_W, col_E)

    %%% Inputs %%%        
        %--% Classification tree inputs %--%
        Ensemble_Technique      = 'Bagging';    % [Bagging, RF], whether a random forest or bootstrap aggregated ensemble of trees is created
        
        Number_Trees            = 200;          % The number of trees that are used for the ensemble technique (majority vote)  
                                                    % The suggested value is 200
        Number_Splits           = 200;          % The maximum number of splits contained in each tree
                                                    % The suggested value is 200
        number_data_partitions  = 004;          % Number of partitions that are used. Note that the last partition is never used for training the classifier and is used only to determine the classification accuracy
                                                    % The suggested value is 4     
                                                
        %--% Sampling inputs %--%
        Sampling_Technique  = 'Oversampling';   % [Oversampling, B-Oversampling, SMOTE, B-SMOTE], standard oversampling, borderline oversampling, SMOTE or Borderline-SMOTE
        
        Beta          = 1.0;        % Beta is the ratio to which the class will be balanced. 0 is not at all, 1 results in the classes being perfectly balanced  
                                        % The suggested value is 1.0
                                        
                                    % Note: These inputs aren't used for every method
        k             = 005;        % Number of nearest neighbours
        Alpha         = 5.0;        % Degree of oversampling the minority (rare) class. If it is 2.0, twice as many samples are created as originally exist and you end up with 300% the original amount

                                          
        %--% Data division inputs %--%
        % Pixel- or superpixel-based classification, the superpixels are based off of the average features of each section
        Classification_scale    = 'Pixel';     % [Superpixel, Pixel]
        
        % Segmentation can be performed in a raster, or by CLIC
        % For superpixel-based classification, CLIC is suggested
        Segmentation_method     = 'Raster';           % [Raster, CLIC, Off]
        
        % The suggested segment size is 60 x 60
        Section_size            = 060;              % Rows and columns of each section that will be classified (so area is Section_size^2)
        
        % Overlap is only used if segmentation is performed in a raster, an 80% overlap resulting in 10 iterations is suggested
        Section_overlap         = 080;              % Percentage of overlap between each section, also influences the number of values for the second layer of majority voting
                                                       
        %--% Inputs for dealing with missing data %--%
        land_cover_threshold    = 050;              % The percentage of pixels that are required to have valid land cover (i.e. not water or missing)
        max_cluster_size        = 100;              % Maximum allowed size of a cluster that is out of the Corine bounds
        
        %--% Whether or not textural feature data is created %--%
        Textural_features       = 'Yes';            % [Yes/No], based on the first feature data entry
        
        %--% Corine reference land cover data parameters %--%
        CLC_year_list       = [2000, 2006, 2012, 2018];                     % Years in which CLC data is available ([2000, 2006, 2012, 2018])
                                                                            % Only data within the range of each CLC year is classified
                                                                            % If years are not of interest, change that value to 0
        
        % The ranges of years that are classified with each CLC data set
        if strcmp(Data_source, 'Copernicus')
            CLC_year_range_cell = {1999:2002, 2003:2008, 2009:2013, 2014:2018}; % These ranges are chosen s.t. they are close in time, and the same sensor (VGT-1, VGT-2, PROBA-V) is used
                                                                                % It is normally {1999:2002, 2003:2008, 2009:2013, 2014:2018}
        elseif strcmp(Data_source, 'MODIS')
            CLC_year_range_cell = {2001:2003, 2004:2009, 2010:2015, 2016:2019}; % These are chosen s.t. they are roughly evenly spaced
            
        elseif strcmp(Data_source, 'Landsat')
            CLC_year_range_cell = {2001:2003, 2004:2009, 2010:2015, 2016:2019}; % These are chosen s.t. they are roughly evenly spaced
        end 
            
        %--% Messages regarding the chosen inputs %--%
        disp('--------------------------------------------------------'); 
        disp('The following Corine data sets are used:')
        disp(CLC_year_list(CLC_year_list ~= 0))
        
        disp('CLASSIFICATION TREE PARAMETERS')        
        fprintf('A %s ensemble is created \n', Ensemble_Technique);        
        fprintf('The number of trees is %g \n', Number_Trees);
        fprintf('The max. number of splits is %g \n', Number_Splits);
        fprintf('The data will be partitioned into %g parts \n', number_data_partitions);
        
        disp('--------------------------------------------------------');
        disp('DATA DIVISION PARAMETERS')
        fprintf('The data is divided using the %s method \n', Segmentation_method);
        fprintf('Each section is %g pixels high and %g pixels wide \n', Section_size, Section_size); 
        
        if strcmp(Segmentation_method, 'Raster')
            fprintf('The overlap between each section is %g%% \n', Section_overlap);
            
            Number_iterations = 100 / (100 - Section_overlap) * 2;
            Number_iterations = floor(Number_iterations);

            fprintf('The resulting number of iterations is %g \n', Number_iterations)
        else
            Number_iterations = 1;
        end
        
        disp('--------------------------------------------------------');
        disp('OVERSAMPLING PARAMETERS')
        fprintf('The sampling method is: %s \n', Sampling_Technique);
        fprintf('The data will be %g%% balanced \n', Beta * 100);
        fprintf('If applicable: %g nearest neighbours are used \n', k);
        fprintf('If applicable: a %g%% degree of oversampling is used \n', Alpha);
        disp('--------------------------------------------------------');
        disp('MISSING DATA PARAMETERS')
        fprintf('The minimum percentage of valid pixels is %g%% \n', land_cover_threshold);
        fprintf('The max. missing cluster size is %g \n', max_cluster_size);
        disp('--------------------------------------------------------');
        
        if strcmp(Textural_features, 'Yes')
            disp('Textural features are also computed');
            disp('--------------------------------------------------------');       
        end
        
    %%% The data that is used to train the classifier %%%
        NDVI_Data = Data_choices{1};
        NDVI_Component_Data = Data_choices{2};
        
        LAI_Data = Data_choices{3};
        FAPAR_Data = Data_choices{4};
        Fcover_Data = Data_choices{5};
        
        Spectral_Data = Data_choices{6};
        
    %%% The years that are classified %%%
        % The Corine years that are supposed to be used for classification
        CLC_year_ind = find(CLC_year_list ~= 0);
        CLC_year_list = CLC_year_list(CLC_year_ind);
        
        % The range of years to be classified
        CLC_year_range_cell = CLC_year_range_cell(CLC_year_ind);
        CLC_year_range_list = horzcat(CLC_year_range_cell{:});
        
        % The indices of the input data corresponding to these years
        year_indices_list = ismember(year_list, CLC_year_range_list);
        year_indices_list = find(year_indices_list == 1);
        classified_year_list = year_list(year_indices_list);
        number_classified_years = length(classified_year_list);
        
%%% Determine the water pixels, which are assumed to be constant %%%
        water_margin = 1e-4;      % Allowed difference for a pixel to be considered a water pixel
        
        if strcmp(NDVI_Data, 'Yes')
            % Compute the mean values
            mean_NDVI = cellfun(@mean, NDVI_TS_cell);

            if strcmp(Data_source, 'Copernicus')
                Water_NDVI = 254/250 - 0.08;
            elseif strcmp(Data_source, 'MODIS')
                Water_NDVI = -0.3;
            end

            % The water pixels are found
            diff_NDVI = abs(mean_NDVI - Water_NDVI);          
            Water_indices = find(diff_NDVI < water_margin);
            
            % Free memory
            clear NDVI_TS_year_cell & mean_NDVI
  
        elseif strcmp(NDVI_Component_Data, 'Yes')
            % Compute the mean values
            mean_NDVI = cellfun(@mean, NDVI_Trend_cell);
            
            if strcmp(Data_source, 'Copernicus')
                Water_NDVI = 254/250 - 0.08;
            elseif strcmp(Data_source, 'MODIS')
                Water_NDVI = 0;
            end

            % The water pixels are found
            diff_NDVI = abs(mean_NDVI - Water_NDVI);
            Water_indices = find(diff_NDVI < water_margin);
            
            % Free memory
            clear NDVI_Trend_year_cell & mean_NDVI
            
        elseif strcmp(LAI_Data, 'Yes')
            % Compute the mean values
            mean_LAI = cellfun(@mean, LAI_TS_cell);
            
            Water_LAI = 255/30;

            % The water indices are found
            diff_LAI = abs(mean_LAI - Water_LAI);
            Water_indices = find(diff_LAI < water_margin);
            
            % Free memory
            clear LAI_TS_year_cell & mean_LAI
            
        elseif strcmp(FAPAR_Data, 'Yes')
            % Compute the mean values
            mean_FAPAR = cellfun(@mean, FAPAR_TS_cell);
            
            Water_FAPAR = 255/250;

            % The water indices are found
            diff_FAPAR = abs(mean_FAPAR - Water_FAPAR);
            Water_indices = find(diff_FAPAR < water_margin);
            
            % Free memory
            clear FAPAR_TS_year_cell & mean_FAPAR
            
        elseif strcmp(Fcover_Data, 'Yes')
            % Compute the mean values
            mean_Fcover = cellfun(@mean, Fcover_TS_cell);
            
            Water_Fcover = 255/250;

            % The water indices are found
            diff_Fcover = abs(mean_Fcover - Water_Fcover);
            Water_indices = find(diff_Fcover < water_margin);
            
            % Free memory
            clear Fcover_TS_year_cell & mean_Fcover
            
        % Spectral Landsat data
        elseif strcmp(Data_source, 'Landsat')
            % The Landsat data does not have a flag for water pixels
            % Instead, the NIR band provides good contrast between water and land
            NIR_index = 4;
            NIR_band = Spectral_Data{NIR_index};
            
            if strcmp(NIR_band, 'Yes')
                % To this end, the constants of the Fourier series are used
                func_constants = @(FS) FS(1);
    
                NIR_constants = cellfun(func_constants, Spectral_Fourier_Coefficients_cell{NIR_index});
                
                % A threshold of 30 is used
                Water_NIR = 30;
                
                Water_indices = find(NIR_constants <= Water_NIR);
                
            elseif strcmp(NIR_band, 'No')
                % Otherwise, water indices cannot be detected
                Water_indices = [];
            end
        end
        
        Water_indices = reshape(Water_indices, [1, length(Water_indices)]);   % Ensures horizontal shape
        
    %%% The feature data is determined %%%
        %--% Tonal features %--%
        % The number of samples per year
        if strcmp(Data_source, 'Copernicus')
            number_annual_timestamps = 36;      % 10-daily
        elseif strcmp(Data_source, 'MODIS')
            number_annual_timestamps = 23;      % 16-daily
        elseif strcmp(Data_source, 'Landsat')
            number_annual_timestamps = 365;     % Daily
        end
            
        % NDVI
        if strcmp(NDVI_Data, 'Yes')           
            [NDVI_feature_data_years_cell, ~, ~] = NDVI_metrics_P(NDVI_TS_cell, year_indices_list, number_classified_years, number_annual_timestamps);
            clear NDVI_TS_cell
        else
            NDVI_feature_data_years_cell = cell(1, number_classified_years);
        end
        % NDVI (decomposed)
        if strcmp(NDVI_Component_Data, 'Yes')
            [NDVI_component_feature_data_years_cell, ~, ~] = NDVI_component_metrics_P(NDVI_Trend_cell, NDVI_Season_cell, year_indices_list, number_classified_years, number_annual_timestamps);     
            clear NDVI_Trend_cell & NDVI_Season_cell
        else
            NDVI_component_feature_data_years_cell = cell(1, number_classified_years);
        end
        
        % LAI
        if strcmp(LAI_Data, 'Yes')
            [LAI_feature_data_years_cell, ~, ~] = LAI_metrics_P(LAI_TS_cell, year_indices_list, number_classified_years, number_annual_timestamps);
            clear LAI_TS_cell
        else
            LAI_feature_data_years_cell = cell(1, number_classified_years);
        end
        % FAPAR
        if strcmp(FAPAR_Data, 'Yes')
            [FAPAR_feature_data_years_cell, ~, ~] = FAPAR_metrics_P(FAPAR_TS_cell, year_indices_list, number_classified_years, number_annual_timestamps);
            clear FAPAR_TS_cell
        else
            FAPAR_feature_data_years_cell = cell(1, number_classified_years);  
        end
        % Fcover
        if strcmp(Fcover_Data, 'Yes')
            [Fcover_feature_data_years_cell, ~, ~] = Fcover_metrics_P(Fcover_TS_cell, year_indices_list, number_classified_years, number_annual_timestamps);
            clear Fcover_TS_cell
        else
            Fcover_feature_data_years_cell = cell(1, number_classified_years);            
        end
        
        % Spectral data
        if strcmp(Data_source, 'Landsat')
            [Spectral_feature_data_years_cell, ~, ~] = Spectral_metrics_P(Spectral_Fourier_Coefficients_cell, Fourier_series_length, Landsat_last_time, year_indices_list, number_classified_years, number_annual_timestamps);
            clear Spectral_Fourier_Coefficients_cell
        else
            Spectral_feature_data_years_cell = cell(1, number_classified_years);            
        end
                
        %--% The tonal features are merged %--%
        tonal_feature_data_years_cell = cell(1, number_classified_years);
        
        for y = 1 : number_classified_years
            % This year's data is concatenated
            tonal_feature_data_years_cell{y} = horzcat(NDVI_feature_data_years_cell{y}, NDVI_component_feature_data_years_cell{y}, LAI_feature_data_years_cell{y}, FAPAR_feature_data_years_cell{y}, Fcover_feature_data_years_cell{y}, Spectral_feature_data_years_cell{y}); 
        end
        
        % Free memory
        clear NDVI_feature_data_years_cell & NDVI_component_feature_data_years_cell & LAI_feature_data_years_cell & FAPAR_feature_data_years_cell & Fcover_feature_data_years_cell & Spectral_feature_data_years_cell        

        %--% Textural features %--%
        if strcmp(Textural_features, 'Yes')
            % Textural features are computed based on the first tonal feature data
            feature_data_IMG_cell = cell(1, number_classified_years);
            
            for y = 1 : number_classified_years
                feature_data_IMG_cell{y} = tonal_feature_data_years_cell{y}{1};
            end

            [textural_feature_data_years_cell, ~, ~] = Textural_Features(feature_data_IMG_cell, number_classified_years, rows_data, columns_data);
        else
            textural_feature_data_years_cell = cell(1, number_classified_years);
        end

        %--% The tonal and textural feature data is merged %--%
        feature_data_years_cell = cell(1, number_classified_years);
        
        for y = 1 : number_classified_years
            feature_data_years_cell{y} = horzcat(tonal_feature_data_years_cell{y}, textural_feature_data_years_cell{y});
        end
        
        number_features = length(feature_data_years_cell{1});
        
        % Free memory
        clear tonal_feature_data_years_cell & textural_feature_data_years_cell
        
    %%% Dimensions of the sections %%%    
        %--% Raster segmentation %--%
        if strcmp(Segmentation_method, 'Raster') | strcmp(Segmentation_method, 'Off')
            % If data division is not desired, the size of the section equals the data size
            if strcmp(Segmentation_method, 'Off')
                Section_size    = max(rows_data, columns_data);
                Section_overlap = 0;
            end
        
            [Indices_sections, Number_sections] = Raster_Segmentation(Section_size, Section_overlap, rows_data, columns_data);
            
        %--% CLIC segmentation %--%
        elseif strcmp(Segmentation_method, 'CLIC')
            % CLIC segmentation is performed based on the first year's data
            [Indices_sections, Number_sections, ~] = CLIC(feature_data_years_cell{1}, number_features, rows_data, columns_data, Section_size);
        end
                        
    %%% Loop through the available ground truth data %%%
        % At each point in the loop, only the relevant feature and ground truth data is used   
        LC_data_cell = cell(1, number_classified_years);

        for c = 1:length(CLC_year_list)
            
        %%% Define the years that can be classified using this CLC ground truth data %%%
            CLC_year = CLC_year_list(c); 

            CLC_year_range_list = CLC_year_range_cell{c};

            year_indices = ismember(classified_year_list, CLC_year_range_list);
            year_indices = find(year_indices == 1);
            year_list_c = classified_year_list(year_indices);
            number_years = length(year_list_c);

            % If there are no dates that can be classified with this CLC data file, the script will continue
            if number_years == 0 || CLC_year == 0
                continue
            end   

        %%% Ground truth data and years to be classified using each ground truth data file %%%
            [CLC_data, number_classes, class_names, class_values, RGB_codes] = Corine_Class_Data(Resolution_downscaling, coarsening_factor, CLC_year, 'On', Color_Choice, row_N, row_S, col_W, col_E);

            % Define the class value used to define the water bodies class
            class_value_water = class_values(class_values > 500 & class_values < 600);
            class_value_water = class_value_water(1);

            % Substitute pixels that are missing and are water pixels
            Missing_indices = find(CLC_data == 999);
            Int_indices = intersect(Missing_indices, Water_indices);
            CLC_data(Int_indices) = class_value_water;                % They are given the water bodies class value
       
            % Determine the indices that are still missing after substitution
            Missing_indices = find(CLC_data == 999);
            Missing_indices = reshape(Missing_indices, [1, length(Missing_indices)]);   % Ensures horizontal shape
            
            % The year from which feature data will be taken to train the classifier, is at the minimum distance to the CLC year
            y_diff_list = abs(year_list_c - CLC_year);
            training_ind = y_diff_list == min(y_diff_list);
            training_year = year_list_c(training_ind);
                 
        %%% Deal with sections that are not suitable to be used for training the classifier %%%       
            % Note that as the missing and water pixels are mostly invariant with time, the sections obtained for the last ground truth data set are used as input
            [valid_sections, invalid_sections, sea_sections, Indices_sections, Number_invalid_clusters, Indices_invalid_clusters, Indices_invalid_clusters_training] = Invalid_Regions(land_cover_threshold, max_cluster_size, rows_data, columns_data, Indices_sections, Water_indices, Missing_indices, Number_sections, Section_size);

        %%% Specify the reference land cover data (ground truth data) of the segments and invalid clusters %%%
            ground_truth_data_sections_cell = cell(1, Number_sections);
            
            for s = 1 : Number_sections
                indices = Indices_sections{s};
                
                % For sea segments, the Corine ground truth data is not necessary
                if sea_sections(s) == 0       
                    ground_truth_data_sections_cell{s} = CLC_data(indices);
                else
                    ground_truth_data_sections_cell{s} = class_value_water;
                end
            end
        
            ground_truth_data_invalid_clusters_cell = cell(1, Number_invalid_clusters);
            
            for r = 1 : Number_invalid_clusters
                indices = Indices_invalid_clusters_training{r};
                
                ground_truth_data_invalid_clusters_cell{r} = CLC_data(indices);
            end
            
            % Free memory
            clear CLC_data
            
        %%% The feature data is divided %%%
            % Valid sections
            feature_data_sections_cell = cell(Number_sections, number_features, number_years);

            % Invalid clusters
            training_feature_data_invalid_clusters_cell = cell(1, Number_invalid_clusters);
            feature_data_clusters_cell = cell(1, Number_invalid_clusters);

            for r = 1 : Number_invalid_clusters
                feature_data_clusters_cell{r} = cell(number_features, number_years);
            end

            % The feature data is appended for each year
            for y = 1 : number_years
                % This year's data
                year = year_list_c(y);
                feature_data = feature_data_years_cell{y};
                
                % Valid sections
                for f = 1 : number_features
                    feature = feature_data{f};

                    for s = 1 : Number_sections
                        indices = Indices_sections{s};

                        if sea_sections(s) == 0     % Feature data for sea segments is unecessary
                            feature_data_sections_cell{s, f, y} = feature(indices);
                        end
                    end
                end

                % The data used to train the classifier
                if year == training_year
                    training_feature_data_sections_cell = feature_data_sections_cell(:, :, y);
                end

                % Invalid clusters
                for r = 1 : Number_invalid_clusters
                    indices = Indices_invalid_clusters{r};

                    for f = 1 : number_features
                        feature_data_clusters_cell{r}{f, y} = feature_data{f}(indices);
                    end

                    % The data used to train the classifier
                    if year == training_year
                        indices = Indices_invalid_clusters_training{r};

                        training_feature_data_invalid_clusters_cell{r} = cell(1, number_features);

                        for f = 1 : number_features
                            training_feature_data_invalid_clusters_cell{r}{f} = feature_data{f}(indices);
                        end
                    end                    
                end

                % Free memory
                feature_data_years_cell{y} = [];
                clear feature_data
            end

            % Free memory and remove the invalid entries for the parallel loop of the valid sections            
            feature_data_sections_cell(invalid_sections, :, :) = [];
            training_feature_data_sections_cell(invalid_sections, :) = [];
            ground_truth_data_sections_cell(invalid_sections) = [];
            Indices_sections(invalid_sections) = [];
            sea_sections(invalid_sections) = [];
               
            %--% Superpixel-based classification %--%
            if strcmp(Classification_scale, 'Superpixel')
                % Each superpixel will be assessed as a pixel within one section/cluster
                number_valid_superpixels = length(valid_sections);
                number_invalid_superpixel_clusters = Number_invalid_clusters;
                
                valid_sections = 1;
                sea_sections(1) = 0;                                                                % The first section is not a sea section anymore, if it was before
                
                Number_invalid_clusters = min(1, Number_invalid_clusters);                          % In case there are no invalid clusters for this domain
                
                % The feature data for valid sections
                averaged_feature_data_sections = cellfun(@mean, feature_data_sections_cell);        % Matrix containing the average feature data

                feature_data_sections_cell = cell(1, number_features, number_years);

                for f = 1 : number_features
                    for y = 1 : number_years
                        feature_data_sections_cell{1, f, y} = averaged_feature_data_sections(:, f, y);
                    end
                end
                
                % The training data for valid sections
                averaged_training_feature_data = cellfun(@mean, training_feature_data_sections_cell);
                
                training_feature_data_sections_cell = cell(1, number_features);
                
                for f = 1 : number_features
                    training_data = averaged_training_feature_data(:, f);                    
                    training_feature_data_sections_cell{f} = training_data;
                end
                
                % The feature data for invalid clusters
                averaged_feature_data_clusters = cell(1, Number_invalid_clusters);
                averaged_training_data_clusters = cell(1, Number_invalid_clusters);
                
                for r = 1 : Number_invalid_clusters
                    feature_data = feature_data_clusters_cell{r};
                    training_data = training_feature_data_invalid_clusters_cell{r};
                    
                    averaged_feature_data = cellfun(@mean, feature_data);                           % Matrix containing the average feature data
                    averaged_training_data = cellfun(@mean, training_data);
                    
                    averaged_feature_data_clusters{r} = averaged_feature_data;
                    averaged_training_data_clusters{r} = averaged_training_data;
                end
                
                averaged_feature_data_clusters_cell = cell(number_features, number_years);
                averaged_training_data_clusters_cell = cell(1, number_features);
                
                for f = 1 : number_features
                    for y = 1 : number_years
                        clusters_feature_data = zeros(1, Number_invalid_clusters);
                        clusters_training_data = zeros(1, Number_invalid_clusters);
                        
                        for r = 1 : Number_invalid_clusters
                            clusters_feature_data(r) = averaged_feature_data_clusters{r}(f, y);
                            clusters_training_data(r) = averaged_training_data_clusters{r}(f, y);
                        end
                        
                        averaged_feature_data_clusters_cell{f, y} = clusters_feature_data;
                        averaged_training_data_clusters_cell{f} = clusters_training_data;
                    end
                end
                
                feature_data_clusters_cell = {averaged_feature_data_clusters_cell};
                training_feature_data_invalid_clusters_cell = {averaged_training_data_clusters_cell};
                
                % The most common land cover type is used as ground truth data
                ground_truth_data_sections = cellfun(@mode, ground_truth_data_sections_cell);
                ground_truth_data_sections_cell = {ground_truth_data_sections};
                
                ground_truth_data_clusters = cellfun(@mode, ground_truth_data_invalid_clusters_cell);
                ground_truth_data_invalid_clusters_cell = {ground_truth_data_clusters};
                
            else
                number_valid_superpixels = 0;
                number_invalid_superpixel_clusters = 0;
            end
            
        %%% Classifying each section %%%
            fprintf('The data is being classified using CLC data set %g \n', CLC_year);
                    
            % The sections that are valid and invalid will be classified independently            
            
            %--% Valid sections %--%
            Number_valid_sections = length(valid_sections);
            
            Classified_Land_Cover_valid_sections_cell = cell(1, Number_valid_sections);
            
            % The test data is saved for the calculation of the classification accuracy
            Classified_Test_Data_cell = cell(1, Number_valid_sections);
            Ground_Truth_Data_cell = cell(1, Number_valid_sections);
            
            % Parallel computing loop  
            DQ = parallel.pool.DataQueue;   % Progress bar
            
            tick = 0;
            N = Number_valid_sections;
            clock_time_previous = clock;
            
            afterEach(DQ, @ProgressUpdate_Valid);
                   
            parfor s = 1 : Number_valid_sections
                % Number of pixels in the section
                indices = Indices_sections{s};
                
                if strcmp(Classification_scale, 'Pixel')
                    number_pixels_section = length(indices);
                elseif strcmp(Classification_scale, 'Superpixel')
                    number_pixels_section = number_valid_superpixels;
                end
                
                % Classification can be skipped if it is a sea section
                if sea_sections(s) == 1
                    Classified_Land_Cover_cell = cell(1, number_years);
                    
                    for y = 1:number_years
                        Classified_Land_Cover_cell{y} = class_value_water * ones(1, number_pixels_section);
                    end
                    
                else
                    % feature data of the pixels to be classified
                    class_feature_data_cell = feature_data_sections_cell(s, :, :);
                    class_feature_data_cell = reshape(class_feature_data_cell, [number_features, number_years]);

                    % Training data and ground truth data
                    train_feature_data_cell = training_feature_data_sections_cell(s, :); 
                    ground_truth_data = ground_truth_data_sections_cell{s};
                    
                    % Land cover classification
                    SaveTestData = 'Yes';   % The test data is desired as an output
                    [Classified_Test_Data, Ground_truth_data_test, Classified_Land_Cover_cell] = Decision_Tree_Classifier(Sampling_Technique, Ensemble_Technique, SaveTestData, number_years, class_feature_data_cell, train_feature_data_cell, ground_truth_data, number_pixels_section, number_features, Number_Trees, Number_Splits, k, Alpha, Beta, number_data_partitions);
                           
                    % The test data results and ground truth data are appended s.t. that classification accuracy can be determined, only if the section is valid to avoid duplicates
                    Classified_Test_Data_cell{s} = Classified_Test_Data;
                    Ground_Truth_Data_cell{s} = Ground_truth_data_test;

                    % Free memory
                    feature_data_sections_cell(s, :, :) = cell(1, number_features, number_years);
                    training_feature_data_sections_cell(s, :) = cell(1, number_features);
                    ground_truth_data_sections_cell{s} = [];
                end
                
                % The classification results, per section
                Classified_Land_Cover_valid_sections_cell{s} = Classified_Land_Cover_cell;
                
                % Update the progress bar
                send(DQ, s);
            end
            
            %--% Invalid clusters %--%
            Classified_Land_Cover_invalid_clusters_cell = cell(1, Number_invalid_clusters);
            
            % Parallel computing loop  
            DQ = parallel.pool.DataQueue;   % Progress bar
            
            tick = 0;
            N = Number_invalid_clusters;
            clock_time_previous = clock;
            
            afterEach(DQ, @ProgressUpdate_Invalid);
                        
            parfor r = 1:Number_invalid_clusters
                % Number of pixels within the cluster
                indices = Indices_invalid_clusters{r};
                
                if strcmp(Classification_scale, 'Pixel')
                    number_pixels_cluster = length(indices);
                elseif strcmp(Classification_scale, 'Superpixel')
                    number_pixels_cluster = number_invalid_superpixel_clusters;
                end
                
                % Training data and ground truth data
                train_feature_data_cell = training_feature_data_invalid_clusters_cell{r};
                ground_truth_data = ground_truth_data_invalid_clusters_cell{r};
                      
                % Feature data
                feature_data_cell = feature_data_clusters_cell{r};
                feature_data_cell = reshape(feature_data_cell, [number_features, number_years]);

                % Land cover classification
                SaveTestData = 'No';    % Test data is not desired for invalid sections
                [~, ~, Classified_Land_Cover_cell] = Decision_Tree_Classifier(Sampling_Technique, Ensemble_Technique, SaveTestData, number_years, feature_data_cell, train_feature_data_cell, ground_truth_data, number_pixels_cluster, number_features, Number_Trees, Number_Splits, k, Alpha, Beta, number_data_partitions);
                
                Classified_Land_Cover_invalid_clusters_cell{r} = Classified_Land_Cover_cell;
                
                % Free memory
                training_feature_data_invalid_clusters_cell{r} = [];
                ground_truth_data_invalid_clusters_cell{r} = [];
                feature_data_clusters_cell{r} = [];
                    
                % Update the progress bar
                send(DQ, r);
            end
                        
        %%% Perform the second layer of majority voting and arrive at the final results %%%
            % If the classification was super-pixel based, the results are presented on a pixel level
            if strcmp(Classification_scale, 'Superpixel')
                % Valid sections            
                Number_valid_sections = number_valid_superpixels;
                
                Land_cover_type_cell_sections = cell(1,  number_valid_superpixels);
                
                for s = 1 : number_valid_superpixels
                    Land_cover_type_cell_sections{s} = cell(1, number_years);
                    
                    for y = 1 : number_years
                        Land_cover_type = Classified_Land_Cover_valid_sections_cell{1}{y}(s);
                        Land_cover_type_cell_sections{s}{y} = Land_cover_type;
                    end
                end
                
                Classified_Land_Cover_valid_sections_cell = Land_cover_type_cell_sections;
                
                % Invalid clusters
                Number_invalid_clusters = number_invalid_superpixel_clusters;
                
                Land_cover_type_cell_clusters = cell(1, number_invalid_superpixel_clusters);
                
                for r = 1 : number_invalid_superpixel_clusters
                    Land_cover_type_cell_clusters{r} = cell(1, number_years);
                    
                    for y = 1 : number_years
                        Land_cover_type = Classified_Land_Cover_invalid_clusters_cell{1}{y}(r);
                        Land_cover_type_cell_clusters{r}{y} = Land_cover_type;
                    end
                end
                
                Classified_Land_Cover_invalid_clusters_cell = Land_cover_type_cell_clusters;
            end
        
            Classified_Land_Cover_cell = cell(1, number_years);
            
            for y = 1:number_years                
                Land_Cover_Iteration_matrix = NaN(rows_data, columns_data, Number_iterations);     % NaN such that areas where fewer than the number of iterations of information is available, the right mode is still taken
                
                % Valid sections                
                for s = 1 : Number_valid_sections
                    Classified_Land_Cover_section = Classified_Land_Cover_valid_sections_cell{s}{y};
                    
                    % Append only if the entire section is empty
                    loop_return = false;    % To prevent a single section from appending multiple times
                    
                    for n = 1 : Number_iterations
                        indices = Indices_sections{s} + (n - 1) * rows_data * columns_data;
                        
                        if isempty(find(~isnan(Land_Cover_Iteration_matrix(indices)), 1)) & loop_return == false
                            Land_Cover_Iteration_matrix(indices) = Classified_Land_Cover_section;
                            
                            loop_return = true;
                        end
                    end
                end

                % Invalid clusters
                for r = 1 : Number_invalid_clusters
                    Classified_Land_Cover_cluster = Classified_Land_Cover_invalid_clusters_cell{r}{y};
                    
                    % Append only if the entire section is empty
                    loop_return = false;    % To prevent a single cluster from appending multiple times
                    
                    for n = 1 : Number_iterations
                        indices = Indices_invalid_clusters{r} + (n - 1) * rows_data * columns_data;
                        
                        if isempty(find(~isnan(Land_Cover_Iteration_matrix(indices)), 1)) & loop_return == false
                            Land_Cover_Iteration_matrix(indices) = Classified_Land_Cover_cluster;
                            
                            loop_return = true;
                        end
                    end
                end

                % Second layer of majority voting
                Classified_Land_Cover_matrix = mode(Land_Cover_Iteration_matrix, 3);
                
                % The MODIS/Copernicus data sometimes disagrees with the Corine data as to what a water bodies pixel is
                % This results in false classifications, therefore these pixels are given the water bodies class value a posteriori
                if strcmp(Data_source, 'Copernicus') | strcmp(Data_source, 'MODIS')
                    Classified_Land_Cover_matrix(Water_indices) = class_value_water;
                end

                Classified_Land_Cover_cell{y} = Classified_Land_Cover_matrix;
            end
                        
            % Append the results
            LC_data_cell(year_indices) = Classified_Land_Cover_cell;
            
            % Free memory
            clear Land_Cover_Iteration_matrix & Classified_Land_Cover_valid_sections_cell & Classified_Land_Cover_invalid_clusters_cell & Classification_Land_Cover_cell & Classified_Land_Cover_matrix

        %%% Determine the classification accuracy and create the confusion matrix %%%
            % Confusion matrix
            if ishandle(1)      % It is impossible to plot a confusion matrix over an existing figure handle
                close(1)
            end
            
            % Combine the test data
            Ground_Truth_Data_total = vertcat(Ground_Truth_Data_cell{:})';
            Classified_Test_Data_total = horzcat(Classified_Test_Data_cell{:});
                        
            % Not all classes may exist in the domain
            class_values_present = unique([unique(Ground_Truth_Data_total), unique(Classified_Test_Data_total)]);
            number_classes_present = length(class_values_present);
            
            class_names_present = cell(1, number_classes_present);
            
            for v = 1:number_classes_present
                class_value = class_values_present(v);
                class_ind = class_values == class_value;
                
                class_name = class_names{class_ind};
                class_names_present{v} = class_name;
            end
            
            % The confusion matrix data            
            if ~isempty(Ground_Truth_Data_total)  
                figure(1)
                confusion_matrix = confusionchart(Ground_Truth_Data_total, Classified_Test_Data_total);
                confusion_matrix_data = confusion_matrix.NormalizedValues;
                close(1)
            else                            % In case the cluster contains only sea pixels
                confusion_matrix_data = zeros(number_classes);
                class_names_present = class_names;
            end
            
            % Figure, where the class values are replaced with the names of the classes
            figure(1)
        
            confusion_matrix = confusionchart(confusion_matrix_data, class_names_present);
            confusion_matrix.RowSummary = 'row-normalized';
            confusion_matrix.ColumnSummary = 'column-normalized'; 
            confusion_matrix.Title = ''; 
            
            % Set the size and white background color
            set(gcf, 'Units', 'Normalized', 'Position', [0 0 1 1]);
            set(gcf, 'color', [1, 1, 1]);
            
            set(gca, 'FontSize', 25);
                        
            % Save the figure, s.t. the accuracy of the land cover classification can be investigated in more detail
            CM_name = sprintf('CM_%s_%s', num2str(CLC_year_list(c)), chunk_string);

            try
                export_fig([CM_name, '.png'])
            catch
                frame = getframe(1);
                im = frame2im(frame);
                [imind, cm] = rgb2ind(im, 256);
                imwrite(imind, cm, [CM_name, '.png']);
            end
                        
            % Save the confusion matrix
            save([CM_name, '.mat'], 'confusion_matrix');
            
            close(1)
            
            % Free memory
            clear Ground_Truth_Data & Total_Test_Results
            
        %%% Classification result files %%%
            % Modify the spatial reference data
            R_land_cover = R_CLC;
            R_land_cover.RasterSize = [rows_data, columns_data];        % Size is adjusted in case the resolution is coarsened
        
            for y = 1:number_years
                Classified_Land_Cover = Classified_Land_Cover_cell{y};
                
                % Create new .tif file
                year = year_list_c(y);
                
                LC_file_name = sprintf('LC_%s_%s', num2str(year), chunk_string);  
                
                if strcmp(Data_source, 'Copernicus')
                    geotiffwrite(LC_file_name, uint16(Classified_Land_Cover), R_land_cover);
                elseif strcmp(Data_source, 'MODIS')
                    Key_file = load('MODIS_NDVI_Geokey.mat');
                    Tag = Key_file.Tag;
                    
                    geotiffwrite(LC_file_name, uint16(Classified_Land_Cover), R_land_cover, 'GeoKeyDirectoryTag', Tag);
                elseif strcmp(Data_source, 'Landsat')
                    Key_file = load('Landsat_Geokey.mat');
                    Tag = Key_file.Tag;
                    
                    geotiffwrite(LC_file_name, uint16(Classified_Land_Cover), R_land_cover, 'GeoKeyDirectoryTag', Tag);
                end
                
                fprintf('The file %s has been created successfully. \n', LC_file_name)
            end     
        end
  
	% Progress function
    function ProgressUpdate_Valid(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Classification progress for the valid sections of this CLC data set: %g%% \n', round(progress));

            % Print the current time
        	time = datestr(now, 'HH:MM:SS');
            fprintf('       t = %s \n', time);
        end
                
        if tick > 1
            % Print the elapsed time
            if floor(progress) - floor(progress_last) >= 1
                % Save the elapsed time
                clock_time = clock;
                
                elapsed_time = etime(clock_time, clock_time_previous) / 60;  % Elapsed time in minutes
                
                fprintf('       delta_t = %g minutes\n', elapsed_time);
                
                clock_time_previous = clock;
            end
        end  
    end

	% Progress function
    function ProgressUpdate_Invalid(~)
        tick = tick + 1;
        
        % Ensures that at most every percent is printed
        progress_last = (tick - 1) / N * 100;
        progress = tick / N * 100;
        
        if floor(progress) - floor(progress_last) >= 1
            fprintf('   Classification progress for the invalid sections of this CLC data set: %g%% \n', round(progress));

            % Print the current time
        	time = datestr(now, 'HH:MM:SS');
            fprintf('       t = %s \n', time);
        end
                
        if tick > 1
            % Print the elapsed time
            if floor(progress) - floor(progress_last) >= 1
                % Save the elapsed time
                clock_time = clock;
                
                elapsed_time = etime(clock_time, clock_time_previous) / 60;  % Elapsed time in minutes
                
                fprintf('       delta_t = %g minutes\n', elapsed_time);
                
                clock_time_previous = clock;
            end
        end     
    end
end



